/*
Copyright (c) 2000-2003 Lee Thomason (www.grinninglizard.com)

Grinning Lizard Utilities. Note that software that uses the 
utility package (including Lilith3D and Kyra) have more restrictive
licences which applies to code outside of the utility package.


This software is provided 'as-is', without any express or implied 
warranty. In no event will the authors be held liable for any 
damages arising from the use of this software.

Permission is granted to anyone to use this software for any 
purpose, including commercial applications, and to alter it and 
redistribute it freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must 
not claim that you wrote the original software. If you use this 
software in a product, an acknowledgment in the product documentation 
would be appreciated but is not required.

2. Altered source versions must be plainly marked as such, and 
must not be misrepresented as being the original software.

3. This notice may not be removed or altered from any source 
distribution.
*/


#include "glmemorypool.h"
#include "gldebug.h"
#include <stdlib.h>
#include <memory.h>


GlMemoryPool::GlMemoryPool( const char* _name, unsigned _objectSize, unsigned _blockSize )
{
	name = _name;

	chunkSize = _objectSize;
	if ( chunkSize < sizeof( Chunk ) )
		chunkSize = sizeof( Chunk );
	chunkSize = ( chunkSize + 3 ) / 4;	// convert to word
	blockSize = ( _blockSize + 3 ) / 4; // convert to word

	chunksPerBlock = ( blockSize - 1 ) / chunkSize;
	GLASSERT( chunksPerBlock > 0 );

	#ifdef DEBUG
	numBlocks = 0;
	numChunks = 0;
	#endif

	rootBlock = 0;
	head = 0;
}


GlMemoryPool::~GlMemoryPool()
{
	GLASSERT( numChunks == 0 );
	FreePool();
}


void* GlMemoryPool::Alloc()
{
	void* ret = 0;

	if ( !head )
	{
		NewBlock();
		GLASSERT( head );
	}

	ret = head;
	head = head->nextChunk;

	#ifdef DEBUG
		++numChunks;
		memset( ret, 0xaa, chunkSize );
	#endif

	return ret;
}


void GlMemoryPool::Free( void* mem )
{
	Chunk* chunk = (Chunk*) mem;

	#ifdef DEBUG
		--numChunks;
		memset( mem, 0xbb, chunkSize );
	#endif

	chunk->nextChunk = 0;
	head = chunk;
}


void GlMemoryPool::NewBlock()
{
	GLASSERT( head == 0 );

	Block* block = (Block*) malloc( blockSize*4 );
	GLASSERT( block );

	block->nextBlock = rootBlock;
	rootBlock = block;

	#ifdef DEBUG
	++numBlocks;
	#endif

	void* vPtr = &block->chunk;
	head = ( Chunk*) vPtr;
	for( unsigned i=0; i<chunksPerBlock-1; ++i )
	{
		void* nextPtr = (unsigned*) vPtr + chunkSize;
		Chunk* cptr = ( Chunk* ) vPtr;
		cptr->nextChunk = (Chunk*) nextPtr;
		vPtr = nextPtr;
	}
	Chunk* cptr = ( Chunk* ) vPtr;
	cptr->nextChunk = 0;
}

void GlMemoryPool::FreePool()
{
	Block* block = rootBlock;
	while ( block )
	{
		Block* temp = block->nextBlock;
		free( block );
		block = temp;
	}
	numBlocks = numChunks = 0;
	rootBlock = 0;
	head = 0;
}


GlLinearMemoryPool::GlLinearMemoryPool( unsigned totalMemory )
{
	base = (char* ) malloc( totalMemory );
	GLASSERT( base );

	current = base;
	end = base + totalMemory;

	#ifdef DEBUG
		memset( base, 0xaa, totalMemory );
	#endif
}


GlLinearMemoryPool::~GlLinearMemoryPool()
{
	free( base );
}


void* GlLinearMemoryPool::Alloc( unsigned allocate )
{
	if ( current < end )
	{
		char* ret = current;
		current += allocate;

		// Out of memory check.
		GLASSERT( current <= end );
		if ( current <= end )
			return ret;
		else
			return 0;
	}
	else
	{
		// Out of memory!
		GLASSERT( 0 );
		return 0;
	}
}
