/*
Copyright (c) 2000-2003 Lee Thomason (www.grinninglizard.com)

Grinning Lizard Utilities. Note that software that uses the 
utility package (including Lilith3D and Kyra) have more restrictive
licences which applies to code outside of the utility package.


This software is provided 'as-is', without any express or implied 
warranty. In no event will the authors be held liable for any 
damages arising from the use of this software.

Permission is granted to anyone to use this software for any 
purpose, including commercial applications, and to alter it and 
redistribute it freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must 
not claim that you wrote the original software. If you use this 
software in a product, an acknowledgment in the product documentation 
would be appreciated but is not required.

2. Altered source versions must be plainly marked as such, and 
must not be misrepresented as being the original software.

3. This notice may not be removed or altered from any source 
distribution.
*/



#ifndef GL_MEMORY_POOL
#define GL_MEMORY_POOL

#include "gldebug.h"

/*	A memory pool that will dynamically grow as memory is needed.
*/
class GlMemoryPool
{
  public:
	GlMemoryPool( const char* _name, unsigned objectSize, unsigned blockSize = 4096 );
	~GlMemoryPool();

	void* Alloc();
	void Free( void* mem );

	void FreePool();

  private:
	struct Chunk
	{
		Chunk* nextChunk;
	};

	struct Block
	{
		Block* nextBlock;
		Chunk* chunk;		// treated as an array of chunks.
	};

	void NewBlock();

	unsigned chunkSize;			// size of chunk in 32bit words
	unsigned blockSize;			// size of block in words
	unsigned chunksPerBlock;
	
	unsigned numBlocks;
	unsigned numChunks;
	
	Block* rootBlock;
	Chunk* head;

	const char* name;
};

/*	A memory pool that has a fixed allocation.
	Essentially a cross between a linked list and an array.
*/
template < class T, int COUNT >
class GlFixedMemoryPool
{
  private:
	struct Chunk
	{
		Chunk* next;
	};

  public:
	GlFixedMemoryPool()
	{
		GLASSERT( sizeof( T ) >= sizeof( Chunk* ) );
		for( int i=0; i<COUNT-1; ++i )
		{
			( (Chunk*)(&memory[i]) )->next = (Chunk*)(&memory[i+1]);
		}
		( (Chunk*)(&memory[COUNT-1]) )->next = 0;
		root = ( (Chunk*)(&memory[0]) );
		inUse = 0;
	}

	~GlFixedMemoryPool()	{}

	T* Alloc()
	{
		T* ret = 0;
		if ( root )
		{
			ret = (T*) root;
			root = root->next;
			++inUse;
		}
		return ret;
	}

	void Free( T* _mem )
	{
		if ( _mem )
		{
			Chunk* mem = (Chunk*) _mem;
			#ifdef DEBUG
				memset( mem, 0xfe, sizeof( T ) );
			#endif
			mem->next = root;
			root = mem;
			--inUse;
		}
	}

	unsigned InUse()	{	return inUse; }
	unsigned Remain()	{	return COUNT - inUse; }
	bool     Contains( T* mem )	{ return mem >= memory && mem < &memory[COUNT]; }

  private:
	T memory[ COUNT ];
	unsigned inUse;
	Chunk* root;
};


/* 	This is memory allocation for when you know exactly how much 
	memory is going to be needed. So it's a way to pre-allocate 
	while still using the new and delete operators.
*/

class GlLinearMemoryPool
{
  public:
	/// Construct, and pass in the amount of memory needed, in bytes.
	GlLinearMemoryPool( unsigned totalMemory );

	~GlLinearMemoryPool();

	/// Allocate a chunk of memory.
	void* Alloc( unsigned allocate );

	/** Note this does nothing.
	*/
	void Free( void* mem )		{}

	/// Return true of this is out of memory.
	bool OutOfMemory()			{ return current == end; }

  public:
	char*	base;
	char*	current;
	char*	end;
};

#endif
