/**
 *  @file    value.c
 *  @author  Guillaume Bour. 2002
 
 *  @version 0.1
 *    @date  28/02/2002

 *  values management
 */
/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdarg.h>
#include <gtk/gtk.h>
#include "value.h"
#include "mem.h"

/**
   Value initialisation.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1. value is not a standalone variable. it's always included into
	       a class (or one of its subtypes).
	       So we don't have to allocate it.

   @param   myval                       the value (a pointer)
   @param   type                        the value type(int, bool, ...)
   @param   value                       the value value(3, TRUE, ...)

   @return  <none>
 */
void value_set(struct s_value *myval, valuetype_t type, ...)
{
  va_list ap;

  myval->valuetype = type;

  va_start(ap, type);
  switch(type)
    {
    case T_INT   : myval->content.vint = va_arg(ap, guint);     break;
    case T_FLOAT : myval->content.vfloat = va_arg(ap, gdouble); break;
    case T_BOOL  : myval->content.vbool = va_arg(ap, gboolean); break;
    case T_STRING: 
      {
	gchar *value = va_arg(ap, gchar *);
	myval->content.vstring = (gchar *)
	  _XMEMDUP(value, sizeof(gchar) * strlen((gchar *)value)+1);
	break;
      }
    }
  va_end(ap);
}


/**
   Value desinitialisation.<br>

   <b>visibility :: <i>public</i></b>

   @remarks:
            1. the value is not a standalone variable, so we don't have to free
	       it. We must just free it's content if it's a string pointer

   @param   val                         the value to clean
   @return  <none>
 */
void value_unset(struct s_value myval)
{
  if(myval.valuetype == T_STRING)
    { _XFREE(myval.content.vstring); }
  
  myval.valuetype = T_NONE;
}

/**
   Value translation.<br>

   <b>visibility :: <i>public</i></b>

   @param   to                          the destination value
   @param   from                        the source value
   @return  <none>
 */
void value_translate(struct s_value to, struct s_value from)
{
  switch(from.valuetype)
    {
    case T_INT   : to.content.vint = from.content.vint; break;
    case T_FLOAT : to.content.vfloat = from.content.vfloat; break;
    case T_BOOL  : to.content.vbool = from.content.vbool; break;
    case T_STRING: to.content.vstring = from.content.vstring;
    }

  to.valuetype = from.valuetype;
  from.valuetype = T_NONE;
}


/**
   Value copy.<br>

   <b>visibility :: <i>public</i></b>

   @param   to                          the destination value
   @param   from                        the source value
   @return  <none>
 */
void value_copy(struct s_value *to, struct s_value *from)
{
  switch(from->valuetype)
    {
    case T_INT   : to->content.vint = from->content.vint; break;
    case T_FLOAT : to->content.vfloat = from->content.vfloat; break;
    case T_BOOL  : to->content.vbool = from->content.vbool; break;
    case T_STRING: to->content.vstring = strdup(from->content.vstring);
    }

  to->valuetype = from->valuetype;
}


#ifdef DEBUG
gchar *value_subdebug(struct s_value val)
{
  gchar *valstr = _XNEW(gchar, 10);

  switch(val.valuetype)
    {
    case T_INT   : 
      g_snprintf(valstr, 10, "%d", val.content.vint);
      break;
    case T_FLOAT : 
      g_snprintf(valstr, 10, "%f", val.content.vfloat);
      break;
    case T_BOOL  : 
      g_snprintf(valstr, 10, "%s", 
		 val.content.vbool == TRUE?"<true>":"false");
      break;
    case T_STRING: 
      g_snprintf(valstr, 10, "\"%s\"", val.content.vstring);
      break;
    default      : 
      _XFREE(valstr);
      valstr = NULL;
    }

  return(valstr);
}
#endif

