/**
 *  @file    variable.c
 *  @author  Guillaume Bour. 2002
 
 *  @version 0.1
 *    @date  05/02/2002
 *  @version 0.2
 *    @date  28/02/2002 - object structuring

 *  variables management
 */
/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */
#include <stdio.h>
#include <gtk/gtk.h>
#include "variable.h"
#include "debug.h"

#ifdef DEBUG
  #include "turtle.h"
  extern struct s_turtle my_turtle;
#endif

/**
   Variable allocation.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1. the default variable type is T_NONE

   @param   name                        the variable name
   @return  the allocated variable
 */
struct s_variable *var_alloc(gchar *name)
{
  struct s_variable *myvar = _XNEW0(struct s_variable, 1);

  CLASS(myvar)->classtype = CLASS_VAR;
  CLASS(myvar)->refcnt = 0;
  myvar->name = _XSTRDUP(name);
  CLASS_VALUE(myvar).valuetype = T_NONE;

  return(myvar);
}


/**
   Variable freeing.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1. the second argument is not used (just for GTK compatibility)

   @param   myvar                       the variable to free
   @param   nil                         not used
   @return  <none>
 */
void var_free_from_list(gpointer myvar, gpointer nil)
{
  if(--(CLASS(myvar)->refcnt) == 0)
    {
      value_unset(CLASS_VALUE(myvar));
      _XFREE(VAR(myvar)->name);
      _XFREE(myvar);
    }
}


/**
   Variable comparison.<br>

   <b>visibility :: <i>public</i></b>

   @param   myvar                       the variable
   @param   name                        the searched variable name

   @return  0 is the variable <i>myvar</i> is named <i>name</i>,
            not 0 else.
 */
gint var_comparison(gconstpointer myvar, gconstpointer name)
{
  return(strcmp(VAR(myvar)->name, (gchar *)name));
}


#ifdef DEBUG
void var_debug(gpointer myvar, gpointer nil)
{
  debug_append2("    (*%d: %s)\n", myvar, VAR(myvar)->name); 
}

gchar *var_subdebug(struct s_variable *myvar)
{
  gchar *svar = _XNEW(gchar, 25);
  g_snprintf(svar, 25, "<var::*%d::%s>", myvar, myvar->name);

  return(svar);
}
#endif
