/**
 *  @file    callbacks.c
 *  @author  Guillaume Bour. 2000/2001                                  
 
 *  @version 0.1
 *    @date  2002/07/21 
 *    @date  2002/08/05 - debugging

 *  interface callback functions (called by buttons, menus, ...)

 *
 */

/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */

#include <stdio.h>
#include <pthread.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

//unused for the moment => useful for resizing, rotating, ... drawing area
/*#ifdef DEBUG
  #include <libart_lgpl/art_alphagamma.h>
  #include <libart_lgpl/art_filterlevel.h>
  #include <libart_lgpl/art_pixbuf.h>
  #include <libart_lgpl/art_rgb_pixbuf_affine.h>
  #include <libart_lgpl/art_affine.h>
  #include <libart_lgpl/art_rgb.h>
  #include <libart_lgpl/art_rgb_affine.h>
  #include <libart_lgpl/art_rgb_rgba_affine.h>
#endif
*/
#include <unistd.h>
#include "callbacks.h"
#include "turtle.h"
#include "mem.h"
#include "io.h"

extern struct s_turtle my_turtle;

/**
   Show specified dialog window ('Open program', 'Save as program', ...).<br>

   @remarks
      + It's a generic function

   @param   window_ref                  the window to open (index)

   @return  the status (showing ok/nok)
 */
void show_window_clb(gpointer window_ref)
{
  gtk_widget_show(my_turtle.boxes[(guint) window_ref]);
}


/**
   Quit the application. Must perform all desinitialisation things 
   (for the moment, nothing :)

   @remarks
     We test if the Logo program has been modified before exit.<br>
     If it is the case, we ask confirmation to the user.

   @param   widget                      <b>unused</b>
   @param   data                        optional user data (<b>unused</b>)

   @return  nil
 */
void quit_application_clb(GtkWidget *widget, gpointer data)
{
  GtkWidget *confirm;
  gint response = GTK_RESPONSE_YES;

  /**
     If the current logo program as been modified without saving, 
     we ask a confirmation to the user
   */
  if(STATUS_IS_MODIFIED(my_turtle))
    {
      confirm = gtk_message_dialog_new(
	GTK_WINDOW(my_turtle.main_app_window),
	GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
	GTK_MESSAGE_QUESTION,
	GTK_BUTTONS_YES_NO,
	_("Are you sure you want to quit without saving your program ?"));

      response = gtk_dialog_run(GTK_DIALOG(confirm));
      gtk_widget_destroy(confirm);
    }

  if(response != GTK_RESPONSE_YES)
    { return; }

  gtk_main_quit();
}


/**
   Open logo program callback.<br>
   Load the logo program into 

   @remarks
            the widget contain the filename to open

   @param   widget                      the file-selection widget
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint open_program_clb(GtkWidget *widget, gpointer data)
{
  GtkWidget *confirm;
  G_CONST_RETURN gchar *filename;
  FILE *instream;
  gchar inbuf[512];
  guint pos, size;
  gint response = GTK_RESPONSE_YES;

  /**
     If the current logo program as been modified without saving, 
     we ask a confirmation to the user
   */
  if(STATUS_IS_MODIFIED(my_turtle))
    {
      confirm = gtk_message_dialog_new(
	GTK_WINDOW(my_turtle.main_app_window),
	GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
	GTK_MESSAGE_QUESTION,
	GTK_BUTTONS_YES_NO,
	_("Are you sure you want to delete current logo program ?"));

      response = gtk_dialog_run(GTK_DIALOG(confirm));
      gtk_widget_destroy(confirm);
    }

  gtk_widget_hide(my_turtle.boxes[OPEN_PROGRAM_BOX]);

  if(response != GTK_RESPONSE_YES)
    { return(FALSE); }

  filename = gtk_file_selection_get_filename(
	       GTK_FILE_SELECTION(my_turtle.boxes[OPEN_PROGRAM_BOX]));

  //filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(widget));

  return(read_program(filename)); 
}


/**
   New logo program callback.<br>
   Clear the current program (with confirmation => might be configurable)

   @param   widget                      the file-selection widget
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint new_program_clb(GtkWidget *widget, gpointer data)
{
  GtkWidget *confirm;
  gint response = GTK_RESPONSE_YES;

  /**
     If the current logo program as been modified without saving, 
     we ask a confirmation to the user
   */
  if(STATUS_IS_MODIFIED(my_turtle))
    {
      confirm = gtk_message_dialog_new(
	GTK_WINDOW(my_turtle.main_app_window),
	GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
	GTK_MESSAGE_QUESTION,
	GTK_BUTTONS_YES_NO,
	_("Are you sure you want to delete current logo program ?"));

      response = gtk_dialog_run(GTK_DIALOG(confirm));
      gtk_widget_destroy(confirm);
    }

  if(response == GTK_RESPONSE_YES)
    { gtk_editable_delete_text(GTK_EDITABLE(my_turtle.source_txt), 0, -1); }

  return(TRUE);
}


/**
   Save the current logo program callback.<br>
     + if it as been previously saved, override previous version
     + if not, ask for a program path/name

   @remarks
     + the widget contain the filename to open
     + if the program has not been modified since last save, do nothing

   @param   widget                      the file-selection widget
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint save_program_clb(GtkWidget *widget, gpointer data)
{
  /* no modifications have been done since last save */
  if(!STATUS_IS_MODIFIED(my_turtle))
    { return(FALSE); }
      

  /* do a saveas if the current program has no name */
  if(!STATUS_IS_NAMED(my_turtle))
    { 
      show_window_clb(SAVE_PROGRAM_BOX); 
      return(FALSE); 
    }

  return(write_program(my_turtle.filename));
}


/**
   Save as the current logo program callback.<br>
     + ask the program path/name to save to

   @remarks
     + if the program has not been modified since last save, do nothing

   @param   widget                      the file-selection widget
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint saveas_program_clb(GtkWidget *widget, gpointer data)
{
  GtkWidget *confirm;
  G_CONST_RETURN gchar *filename; 
  gint response = GTK_RESPONSE_YES;

  if(!STATUS_IS_MODIFIED(my_turtle))
    { return(FALSE); }

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(my_turtle.boxes[SAVE_PROGRAM_BOX]));

  /**
     If the file yet exist, we ask for confirmation
   */
  if(g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      confirm = gtk_message_dialog_new(
	GTK_WINDOW(my_turtle.main_app_window),
	GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
	GTK_MESSAGE_QUESTION,
	GTK_BUTTONS_YES_NO,
	_("The file \"%s\" always exist. Are you sure you want to overwrite it ?"),
	filename);

      response = gtk_dialog_run(GTK_DIALOG(confirm));
      gtk_widget_destroy(confirm);
    }

  if(response != GTK_RESPONSE_YES)
    { return(FALSE); }
      
  g_free(my_turtle.filename);
  my_turtle.filename = g_strdup(filename);
  SET_STATUS_NAMED(my_turtle);

  printf("saving\n");
  gtk_widget_hide(my_turtle.boxes[SAVE_PROGRAM_BOX]);
  return(write_program(my_turtle.filename));
}


/**
   Save the current drawing.<br>

   @remarks
      + the widget contain the filename to open

   @param   widget                      the file-selection widget
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint save_drawing_clb(GtkWidget *widget, gpointer data)
{
  GError *error;
  GdkPixbuf *pixbuf;
  G_CONST_RETURN gchar *filename;
  GdkColormap *colormap;

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(widget));

  colormap = gtk_widget_get_colormap(my_turtle.drawing_field);
  pixbuf = gdk_pixbuf_get_from_drawable(
    NULL, 
    my_turtle.pixmap,
    colormap,
    0, 0, 0, 0, 
    -1,
    -1);

  gdk_pixbuf_save(pixbuf, filename, "png", &error, NULL);

  return(TRUE);
}


/**
   Switch logo locale.<br>

   @remarks
            + the locale is contained into the date
	    + <b>we must switch the current locale pixmap</b>
	    
   @param   locale                      the logo locale to use 
                                        (of locale_t type)

   @return  the status (ok/nok)
 */
gint switch_logo_locale_clb(gpointer locale)
{
  printf("choose %d logo locale\n", (locale_t) locale);
  my_turtle.locale = (locale_t) locale;
  return(TRUE);
}


/**
   Create & show about box.<br>

   @remarks
            + <b>instanciate it at creation time ??</b>
	    
   @param   widget                      <b>unused</b>
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint open_aboutbox_clb(GtkWidget *widget, gpointer data)
{
  GtkWidget 
    *dialog, 
    *label, 
    *button;

  dialog = gtk_dialog_new();
  //  gtk_window_set_title(GTK_WINDOW(dialog), _(" propos ..."));
  gtk_window_set_title(GTK_WINDOW(dialog), _("About ..."));

  /** line #1 */
  label = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  /** line #2 */
  label = gtk_label_new("+ main developper: guillaume bour <gbour@nomade.fr>"); 
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  /** line #3 */
  label = gtk_label_new("\n\n  dedicaced to my little syster, anne-hlne  \n");
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  /** buttons */
  button = gtk_button_new_with_label("Close");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		      button, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(button), "released",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));
  gtk_widget_show(button);

  gtk_widget_show(dialog);
  return TRUE;
}


/**
   Compile & execute the logo program currently set.<br>

   @remarks
	    
   @param   widget                      <b>unused</b>
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint execute_program_clb(GtkButton *button, gpointer source)
{
  GError *error;
  //pthread_t my_thread;
  GThread *my_thread;

  /* 1. thread initialization */


  /* 2. thread execution        */
  g_thread_create(execute_program_clb_threaded, source, FALSE, &error);
  //my_thread = g_thread_create(execute_program_clb_threaded, source, TRUE, &error);
  //g_thread_join(my_thread);
  //pthread_create(&my_thread, NULL, execute_program_clb_threaded, &source);
  printf("continue\n");
}


/**
   Compile & execute the logo program currently set.<br>

   @remarks
	    
   @param   widget                      <b>unused</b>
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
//gint execute_program_threaded_clb(GtkButton *button, gpointer source)
gpointer execute_program_clb_threaded(gpointer source)
{ 
  /*  sleep(1);
  printf("hello\n"); 
  sleep(1);

  g_thread_exit(0);
}

void *none_execute_program_clb_threaded(void *args)
{*/
//  gpointer source;
  gchar *code;

  printf("execute_program_clb_threaded\n");
  gdk_threads_enter();
//  source = (gpointer) args;

  /* read code */
  code = gtk_editable_get_chars(GTK_EDITABLE(source), 0, -1);
  
  /* compilation initialisation */
  mc_init();

  /* starting compilation        */
  gtk_statusbar_push(GTK_STATUSBAR(my_turtle.wstatus), 1, 
		     _("*** compiling code source"));
  debug_clear();
  launch_parser(code);

  /* compilation failed ?       */
  if(my_turtle.compilation_failed)
    {
      gtk_statusbar_push(GTK_STATUSBAR(my_turtle.wstatus), 1, 
			 _("*** compilation has failed"));
      gtk_notebook_set_page(GTK_NOTEBOOK(my_turtle.notebook), 1);

      //      gdk_threads_leave();
      g_thread_exit(FALSE);
      //return(FALSE);
    } else {
      /* printing 3 adresses code         */
#ifdef DEBUG1
      mc_debug();
#endif

      /* starting execution               */
#ifdef DEBUG
      if(my_turtle.draw)
	{
#endif
	  gtk_statusbar_push(GTK_STATUSBAR(my_turtle.wstatus), 1, 
			     _("*** executing compiled code"));

	  //gdk_threads_enter();
	  exec_execute_all();
	  //gdk_threads_leave();
	  //g_thread_exit(exec_execute_all());
	  //return(exec_execute_all());
	}
#ifdef DEBUG
    }
#endif

  gdk_threads_leave();
  printf("end thread\n");
  g_thread_exit(TRUE);
    //return(TRUE);
}


#ifdef DEBUG

/**
   Switch between debug and drawing window.<br>

   @remark
     + <b>only used if DEBUG is set at compile time</b>
	    
   @param   widget                      <b>unused</b>
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
void debug_switch_frame_clb(GtkWidget *widget, gpointer data)
{
  if(gtk_notebook_get_current_page(GTK_NOTEBOOK(my_turtle.notebook)) == 0)
    { gtk_notebook_set_page(GTK_NOTEBOOK(my_turtle.notebook), 1); }
  else
    { gtk_notebook_set_page(GTK_NOTEBOOK(my_turtle.notebook), 0); }
}


/**
   Activate or not the drawing.<br>

   @remark
     + <b>if desactivate, only the debug trace is done during execution</b>
	    
   @param   widget                      <b>unused</b>
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
void debug_draw_picture_clb(GtkWidget *widget, gpointer data)
{
  my_turtle.draw = !my_turtle.draw;
}

#endif



/** FOLLOWINGS ARE AUTOMATED EVENTS (like refreshing the drawing area, ...) **/

/**
   Post-window open configure event.<br>

   @param   widget                      <b>unused</b>
   @param   user data                   <b>unused</b>

   @return  the status (ok/nok)
 */
gint window_configure_event(GtkWidget *widget, GdkEventConfigure *event)
{
  gtk_adjustment_set_value(my_turtle.hadj, 200.0);
  gtk_adjustment_set_value(my_turtle.vadj, 200.0);
}


/**
   Post-drawarea creation configure event.<br>

   @param   widget                      <b>unused</b>
   @param   event                       <b>unused</b>

   @return  the status (ok/nok)
 */
gint drawarea_configure_event(GtkWidget *widget, GdkEventConfigure *event)
{
  GdkPixmap *pixmap = NULL;

  my_turtle.x = ((my_turtle.drawing_field)->allocation).width / 2;
  my_turtle.y = ((my_turtle.drawing_field)->allocation).height / 2;


  /* (re)creating & blanking background pixmap */
  if (pixmap)
    gdk_pixmap_unref(pixmap);
 
  pixmap = gdk_pixmap_new(widget->window,
			  widget->allocation.width,
			  widget->allocation.height,
			  -1);
  gdk_draw_rectangle(pixmap,
		     widget->style->white_gc,
		     TRUE,
		     0, 0,
		     widget->allocation.width,
		     widget->allocation.height);

  my_turtle.pixmap = pixmap;
  return TRUE;
}


/**
   expose event - redraw the drawing area.<br>

   @param   widget                      the drawarea
   @param   event                       the box to redraw

   @return  the status (ok/nok)
 */
gint drawarea_expose_event(GtkWidget *widget, GdkEventExpose *event)
{
  // gdk_threads_enter();
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  my_turtle.pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

  gdk_flush();
  //  gdk_threads_leave();

  return TRUE;
}




























gint turtled_configure_event(GtkWidget *widget,
			     GdkEventExpose *event )
{
  GdkPixbuf *image;
  int x, y;
  GError *error;

  //  printf("turtled_configure\n");
  image = gdk_pixbuf_new_from_file("./pixmaps/turtle2.png", &error);

  x = gdk_pixbuf_get_width(image);
  y = gdk_pixbuf_get_height(image);

  gdk_pixbuf_render_to_drawable_alpha(image,
				      widget->window,
				      //widget->style->white_gc,
				      0, 0,
				      0, 0, 
				      x, y,
				      GDK_PIXBUF_ALPHA_BILEVEL, 10,
				      GDK_RGB_DITHER_NONE,
				      0, 0);
  


  return FALSE;
}

/*
gint motion_notify_event( GtkWidget *widget,
                                                  GdkEventMotion *event )
{
printf("motion notify event\n");                  
  
  return TRUE;
}*/

void show_clbk(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  //gtk_widget_show(GTK_WIDGET(my_turtle.filesel));
}

void show_clbk2(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  //gtk_widget_show(GTK_WIDGET(my_turtle.saveassel));
}

void open_callback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gchar *filename;

  gtk_widget_hide(data);
  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));
  printf("filename: '%s'\n", filename);
}


#ifdef MEMPROOF
gint memitem_response(GtkWidget *widget, GdkEvent *event)
{
  GtkWidget *dial, *label, *btn;
  gchar *llabel;

  //  printf("memitem_response callback executed\n");

  dial = gtk_dialog_new();

  llabel = g_new(gchar, 50);
  sprintf(llabel, "\n  mem proof: <%d>alloc, <%d>free, <%d>null-free  \n", 
	  _alloc_cnt, _free_cnt, _null_free_cnt);

  label = gtk_label_new (llabel);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dial)->vbox),
		      label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  btn = gtk_button_new_with_label("free mem");
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dial)->action_area),
		      btn, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(btn), "released",
			    GTK_SIGNAL_FUNC(free_mem),
			    label);
  gtk_widget_show(btn);

  btn = gtk_button_new_with_label("reset counters");
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dial)->action_area),
		      btn, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(btn), "released",
			    GTK_SIGNAL_FUNC(reset_counters),
			    label);
  gtk_widget_show(btn);

  btn = gtk_button_new_with_label("close");
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dial)->action_area),
		      btn, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(btn), "released",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dial));
  gtk_widget_show(btn);


  gtk_widget_show(dial);
  // gtk_dialog_run(GTK_DIALOG(dial));

  g_free(llabel);
  return TRUE;
}

gboolean free_mem(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gchar *llabel;
  printf("freeing memory\n");

  mc_free();

  llabel = g_new(gchar, 70);
  snprintf(llabel, 69, 
	   "\n  mem proof: <%d>alloc, <%d>free, <%d>null-free  \n", 
	  _alloc_cnt, _free_cnt, _null_free_cnt);
  gtk_label_set_text(GTK_LABEL(widget), llabel);
  g_free(llabel);
		    
  return(FALSE);
}

gboolean reset_counters(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gchar *llabel;
  printf("reseting counters\n");

  mc_free();
  _alloc_cnt = 0;
  _free_cnt = 0;
  _null_free_cnt = 0;

  llabel = g_new(gchar, 50);
  sprintf(llabel, "\n  mem proof: <%d>alloc, <%d>free, <%d>null-free  \n", 
	  _alloc_cnt, _free_cnt, _null_free_cnt);
  gtk_label_set_text(GTK_LABEL(widget), llabel);
  g_free(llabel);
		    
  return(FALSE);
}

#endif


/* another callback */
/*gboolean quit_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
printf("quit event\n");
  return(FALSE);
}*/

void destroy(GtkWidget *widget, gpointer data)
{
  //  printf("destroy event\n");
  gtk_main_quit();
}



void text_activate(GtkWidget *widget, gint a1, gint a2, gpointer data)
{
  printf("text activate\n");
  gtk_signal_handler_block(GTK_OBJECT(widget), "insert-text");

}


void text_activate2(GtkEditable *editable,
		    gchar *new_text,
		    gint new_text_length,
		    gint *position,
		    gpointer user_data)
{
  printf("text activate 2\n");
  SET_STATUS_MODIFIED(my_turtle);
  //g_signal_stop_emission_by_name (GTK_OBJECT (editable), "insert_text"); 
 //  gtk_signal_disconnect(GTK_OBJECT(editable), "insert-text");
}

/*
g_signal_handlers_unblock_by_func (GTK_OBJECT (editable), 
                                     insert_text_handler, data);

  g_signal_stop_emission_by_name (GTK_OBJECT (editable), "insert_text"); 

*/
