/**
 *  @file    exec.c
 *  @author  Guillaume Bour. 2002
 
 *  @version 0.1
 *    @date  2002/03/20
 *    @date  2002/08/05 - warnings corrections

 *  compiled logo programs execution.
 */
/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */
#include <math.h>
#include <time.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include "exec.h"
#include "turtle.h"
#include "mc.h"
#include "class.h"
#include "type.h"
#include "list.h"
#include "proc.h"
#include "instr.h"

extern struct s_turtle my_turtle;

/*>>> operations callbacks         */
typedef struct s_instr *(*exec_clb_t)(struct s_instr *);
exec_clb_t exec_clb[44] = {
  exec_noop,
  exec_proccall,
  exec_return,
  exec_moveforward,
  exec_moveback,
  exec_turnright,
  exec_turnleft,
  exec_setpos,
  exec_setheading,
  exec_clearscreen,
  exec_hideturtle,
  exec_showturtle,
  exec_hidepen,
  exec_showpen,
  exec_setpredefcolor,
  exec_setuserdefcolor,
  exec_notrace,
  exec_trace,
  exec_write,
  exec_stop,
  exec_affect,
  exec_locaffect,
  exec_danse,
  exec_sleep,
  exec_getheading,
  exec_hazard,
  exec_repeat,
  exec_loopcnt,
  exec_if,
  exec_else,
  exec_while,
  exec_jump,
  exec_and,
  exec_or,
  exec_eq,
  exec_lt,
  exec_gt,
  exec_leq,
  exec_geq,
  exec_add,
  exec_sub,
  exec_mul,
  exec_div,
  exec_uneg
};

/*>> predefined colors table  */
gushort predefined_colors[16][3] = {
  {     0,     0,     0}, // 0 black
  {     0,     0, 65535}, // 1 blue
  {     0, 65535,     0}, // 2 green
  {     0, 65535, 65535}, // 3 cyan
  { 65535,     0,     0}, // 4 red
  { 65535,     0, 65535}, // 5 magenta
  { 65535, 65535,     0}, // 6 yellow
  { 65535, 65535, 65535}, // 7 white
  { 39835, 24672, 15163}, // 8 brown
  { 50629, 34952,  4626}, // 9 light-brown
  { 25700, 41634, 16448}, //10 mid-green
  { 30840, 48059, 48059}, //11 blue-green
  { 65535, 38293, 30583}, //12 salmon
  { 37008, 29041, 53456}, //13 blue-ish
  { 65535, 41891,     0}, //14 orange
  { 47031, 47031, 47031}  //15 or
};


/**
   main execution function.<br>

   <b>visibility :: <i>public</i></b>

   @return  <none>
 */
void exec_execute_all()
{
  gtk_notebook_set_page(GTK_NOTEBOOK(my_turtle.notebook), 0);

  debug_append("*** Starting execution\n");
  exec_execute_proc(my_turtle.main_proc);
  debug_append("*** Ending execution\n");

}


/**
   execute a procedure.<br>

   <b>visibility :: <i>public</i></b>

   @return  <none>
 */
void exec_execute_proc(struct s_proc *proc)
{
  GSList *curelem = proc->instructions;
  register struct s_instr *instr, *jmp_instr;

  for(; curelem != NULL;)
    {
      /* *** */
      //gdk_threads_enter();
      //printf("pulse\n");
      gtk_progress_bar_pulse(my_turtle.progressbar);
      //gdk_threads_leave();
      /* *** */
      
      instr = (struct s_instr *) curelem->data;

      jmp_instr = (* exec_clb[instr->op])(instr);

      if(jmp_instr == NULL)
	{ curelem = g_slist_next(curelem); }
      else
	{ curelem = g_slist_find(proc->instructions, jmp_instr); }
    }
}


/**
   'NOOP':: do nothing.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_noop(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_noop()\n");
#endif
  
  return(NULL);
}


/**
   'PROCCAL':: procedure call.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_proccall(struct s_instr *instr)
{
  struct s_proc *caller, *called;
  register GSList *listof_arguments, *listof_values;

#ifdef DEBUG
  debug_append("  # executing exec_proccall()\n");
#endif
	       
  called = PROC(instr->arg1);

  caller = my_turtle.cur_proc;
  my_turtle.cur_proc = called; 

  /*!!! pb dans l'ordre !!!*/
  /* context save */
  context_save(caller);

  /* on initialise les arguments */
  if(instr->arg2 != NULL)
    {
      listof_arguments = called->parameters;
      listof_values = LIST(instr->arg2)->GTKList;

      while(listof_arguments != NULL)
	{
	  value_copy(&(CLASS(listof_arguments->data)->value),
		     &(CLASS(listof_values->data)->value));

	  /*printf("setting %d/%s value to: %d\n", listof_arguments->data,
		 VAR(listof_arguments->data)->name,
		 CLASS(listof_values->data)->value.content.vint);*/

	  listof_arguments = g_slist_next(listof_arguments);
	  listof_values = g_slist_next(listof_values);
	}
    }


  /* we must set the arguments !! */
  exec_execute_proc(called);

  my_turtle.cur_proc = caller;
  /* restoring save */
  context_restore(caller);
  
  return(NULL);
}


/**
   'RETURN':: ???.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_return(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_return()\n");
#endif

  return(NULL);
}


/**
   'MOVE FORWARD':: move the turtle forward.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_moveforward(struct s_instr *instr)
{
  register int steps;
#ifdef DEBUG
  debug_append("  # executing exec_moveforward()\n");
#endif

  VALUE_INFERENCE(T_INT, &steps, instr->arg1);
  /*printf("move forward steps: %d\n", steps);*/

  backdrawing_moveforward(steps);
  return(NULL);
}

/**
   'MOVE BACK':: move the turtle back.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_moveback(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_moveback()\n");
#endif

  return(NULL);
}


/**
   'TURN RIGHT':: turn the turtle right.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_turnright(struct s_instr *instr)
{
  gdouble degree_angle;

#ifdef DEBUG
  debug_append("  # executing exec_turnright()\n");
#endif

  if(VALUE_INFERENCE(T_FLOAT, &degree_angle, instr->arg1))
    {
      my_turtle.virtual_heading = 
	fmod(my_turtle.virtual_heading + degree_angle, 360.0);

      /*printf("left turn angle: %f, new angle: %f\n", degree_angle, 
	my_turtle.virtual_heading);*/

      /* degrees2radians conversion */
      my_turtle.angle = 
	fmod((my_turtle.angle + degree_angle * (G_PI/180)), (2 * G_PI));
    }


  return(NULL);
}


/**
   'TURN LEFT':: turn the turtle left.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_turnleft(struct s_instr *instr)
{
  gdouble degree_angle;

#ifdef DEBUG
  debug_append("  # executing exec_turnleft()\n");
#endif

  if(VALUE_INFERENCE(T_FLOAT, &degree_angle, instr->arg1))
    {
      my_turtle.virtual_heading = 
	fmod(my_turtle.virtual_heading + (360.0 - degree_angle), 360.0);

      /*printf("left turn angle: %f, new angle: %f\n", degree_angle, 
	my_turtle.virtual_heading);*/
      /* degrees2radians conversion */
      my_turtle.angle = 
	fmod((my_turtle.angle + 2 * G_PI - degree_angle * (G_PI/180)), 
	     (2 * G_PI));
    }

  return(NULL);
}

/**
   'SET POS':: set the position of the turtle.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_setpos(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_setpos()\n");
#endif

  VALUE_INFERENCE(T_FLOAT, &(my_turtle.x), instr->arg1);
  VALUE_INFERENCE(T_FLOAT, &(my_turtle.y), instr->arg2);

  return(NULL);
}

/**
   'SET HEADING':: set the turtle heading.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_setheading(struct s_instr *instr)
{
  gdouble degree_angle;

#ifdef DEBUG
  debug_append("  # executing exec_setheading()\n");
#endif

  if(VALUE_INFERENCE(T_FLOAT, &degree_angle, instr->arg1))
    {
      /*printf("setpos angle: %f\n", degree_angle);*/
      /* degrees2radians conversion */
      my_turtle.angle = fmod(BASE_CAP + degree_angle * (G_PI/180), (2 * G_PI));
    }

  return(NULL);
}


/**
   'CLEARSCREEN':: clear the screen.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_clearscreen(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_clearscreen()\n");
#endif

  backdrawing_clearall();
  return(NULL);
}


/**
   'HIDE TURTLE':: hide the turtle.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_hideturtle(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_hideturtle()\n");
#endif

  return(NULL);
}


/**
   'SHOW TURTLE':: show the turtle.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_showturtle(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_showturtle()\n");
#endif

  return(NULL);
}


/**
   'HIDE PEN':: hide the pen (difference with NOTRACE ???).<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_hidepen(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_hidepen()\n");
#endif

  my_turtle.trace = FALSE;
  return(NULL);
}


/**
   'SHOW PEN':: show the pen (see up).<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_showpen(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_showpen()\n");
#endif

  my_turtle.trace = TRUE;
  return(NULL);
}


/**
   'SET PREDEF COLOR':: set predefined color.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_setpredefcolor(struct s_instr *instr)
{
  gint predefcolor;
  
#ifdef DEBUG
  debug_append("  # executing exec_setpredefcolor()\n");
#endif

  VALUE_INFERENCE(T_INT, &predefcolor, instr->arg1);
  /*printf("setting predefined color: %d\n", predefcolor);*/
  my_turtle.pencolor.red   = predefined_colors[predefcolor][0];
  my_turtle.pencolor.green = predefined_colors[predefcolor][1];
  my_turtle.pencolor.blue  = predefined_colors[predefcolor][2];

  return(NULL);
}


/**
   'SET USERDEF COLOR':: set userdef color.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_setuserdefcolor(struct s_instr *instr)
{
  GSList *colors;
  gint redc, greenc, bluec;

#ifdef DEBUG
  debug_append("  # executing exec_setuserdefcolor()\n");
#endif

  list_inference(&colors, instr->arg1);
  VALUE_INFERENCE(T_INT, &redc, CLASS(colors->data));
  VALUE_INFERENCE(T_INT, &greenc, CLASS(colors->next->data));
  VALUE_INFERENCE(T_INT, &bluec, CLASS(colors->next->next->data));

  /*printf("setting userdefined color: %d-%d-%d\n", redc, greenc, bluec);*/
  my_turtle.pencolor.red   = redc;
  my_turtle.pencolor.green = greenc;
  my_turtle.pencolor.blue  = bluec;

  return(NULL);
}


/**
   'NO TRACE':: don't trace (???).<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_notrace(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_notrace()\n");
#endif

  my_turtle.trace = FALSE;
  return(NULL);
}


/**
   'TRACE':: trace (???).<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_trace(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_trace()\n");
#endif

  my_turtle.trace = TRUE;
  return(NULL);
}


/**
   'WRITE':: write a string.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_write(struct s_instr *instr)
{
  gchar *text;

#ifdef DEBUG
  debug_append("  # executing exec_write()\n");
#endif

  VALUE_INFERENCE(T_STRING, &text, CLASS(instr->arg1));
  /*printf("text to write: %s\n", text);*/
  backdrawing_writetext(text);

  return(NULL);
}


/**
   'STOP':: stop the execution.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_stop(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_stop()\n");
#endif

  return(NULL);
}


/**
   'AFFECT':: affect a value to a variable.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_affect(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_affect()\n");
#endif

  value_copy(&CLASS_VALUE(instr->arg1), &CLASS_VALUE(instr->arg2));

  return(NULL);
}


/**
   'LOCAFFECT':: affect a value to a local variable.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_locaffect(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_locaffect()\n");
#endif

  value_copy(&CLASS_VALUE(instr->arg1), &CLASS_VALUE(instr->arg2));

  return(NULL);
}


/**
   'DANSE':: make the turtle dancing :)).<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_danse(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_danse()\n");
#endif

  return(NULL);
}


/**
   'DANSE':: make the turtle sleeping :)).<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_sleep(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_sleep()\n");
#endif

  return(NULL);
}


/**
   'GET HEADING':: get the current heading.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_getheading(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_getheading()\n");
#endif

  CLASS_VALUE(instr).valuetype = T_FLOAT;
  CLASS_VALUE(instr).content.vfloat = my_turtle.virtual_heading;

  return(NULL);
}


/**
   'HAZARD':: choose a hazardous number.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_hazard(struct s_instr *instr)
{
  register guint maxvalue, hazvalue;

#ifdef DEBUG
  debug_append("  # executing exec_hazard()\n");
#endif

  VALUE_INFERENCE(T_INT, &maxvalue, instr->arg1);

  hazvalue = (int) (((double)maxvalue)*rand()/(RAND_MAX+1.0));
  CLASS(instr)->value.content.vint = hazvalue;
  CLASS(instr)->value.valuetype = T_INT;
  /*printf("hazvalue: %d\n", CLASS(instr)->value.content.vint);*/

  return(NULL);
}


/**
   'REPEAT':: repeat structure.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_repeat(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_repeat()\n");
#endif

  /* first time we execute the repeat instruction (in the current context) */
  if(VALUE_TYPE(instr) == T_NONE)
    {
      CLASS(instr)->value.valuetype = T_INT;
      /* use VALUE_VALUE ?? */
      VALUE_INFERENCE(T_INT, &(CLASS(instr)->value.content.vint), instr->arg2);
    }

  if(CLASS(instr)->value.content.vint-- == 0)
    { 
      CLASS(instr)->value.valuetype = T_NONE;
      return(INSTR(instr->arg1)); 
    }
  else
    { return(NULL); }
}


/**
   'LOOPCNT':: get the current loop counter.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_loopcnt(struct s_instr *instr)
{
  register gint loopcnt, startval;

#ifdef DEBUG
  debug_append("  # executing exec_loopcnt()\n");
#endif

  VALUE_INFERENCE(T_INT, &loopcnt, CLASS(instr->arg1));
  VALUE_INFERENCE(T_INT, &startval, CLASS(INSTR(instr->arg1)->arg2));

  CLASS_VALUE(instr).valuetype = T_INT;
  CLASS_VALUE(instr).content.vint = startval - loopcnt;

  return(NULL);
}


/**
   'IF':: if instruction.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_if(struct s_instr *instr)
{
  register gboolean condition;
  
#ifdef DEBUG
  debug_append("  # executing exec_if()\n");
#endif

  VALUE_INFERENCE(T_BOOL, &condition, CLASS(instr->arg2));

  /*printf("condition: %d\n", condition);*/
  if(!condition)
    { /*printf("false\n");*/ return(INSTR(instr->arg1)); }
  else
    { /*printf("true\n");*/ return(NULL); }
}


/**
   'ELSE':: else instruction.<br>
   IS IT USEFUL ???

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_else(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_else()\n");
#endif

  return(NULL);
}


/**
   'WHILE':: while instruction.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_while(struct s_instr *instr)
{
  register gboolean condition;

#ifdef DEBUG
  debug_append("  # executing exec_while()\n");
#endif

  VALUE_INFERENCE(T_BOOL, &condition, CLASS(instr->arg2));

  /*printf("while condition: %d\n", condition);*/
  if(!condition)
    { /*printf("false\n");*/ return(INSTR(instr->arg1)); }
  else
    { /*printf("true\n");*/ return(NULL); }

  return(NULL);
}


/**
   'JUMP':: unconditional jump.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_jump(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_jump()\n");
#endif

  return(INSTR(instr->arg1)); 
}


/**
   'AND':: and boolean operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_and(struct s_instr *instr)
{
  gboolean lop, rop;

#ifdef DEBUG
  debug_append("  # executing exec_and()\n");
#endif

  VALUE_INFERENCE(T_BOOL, &lop, CLASS(instr->arg1));
  VALUE_INFERENCE(T_BOOL, &rop, CLASS(instr->arg2));

  CLASS_VALUE(instr).valuetype = T_BOOL;
  CLASS_VALUE(instr).content.vbool = (lop & rop);

  return(NULL);
}


/**
   'OR':: or boolean operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_or(struct s_instr *instr)
{
  gboolean lop, rop;

#ifdef DEBUG
  debug_append("  # executing exec_or()\n");
#endif

  VALUE_INFERENCE(T_BOOL, &lop, CLASS(instr->arg1));
  VALUE_INFERENCE(T_BOOL, &rop, CLASS(instr->arg2));

  CLASS_VALUE(instr).valuetype = T_BOOL;
  CLASS_VALUE(instr).content.vbool = (lop | rop);

  return(NULL);
}


valuetype_t comparison_allowed_types[6][6] = {
  /*                T_ILLEGAL, T_NONE   , T_INT    , T_FLOAT  , T_BOOL    , 
		    T_STRING                                                 */
  /* T_ILLEGAL */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_NONE    */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_INT     */ { T_ILLEGAL, T_ILLEGAL, T_INT    , T_FLOAT  , T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_FLOAT   */ { T_ILLEGAL, T_ILLEGAL, T_FLOAT  , T_FLOAT  , T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_BOOL    */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_BOOL    ,
                    T_ILLEGAL},
  /* T_STRING  */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_STRING }
};

/**
   'EQ':: eq comparison operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_eq(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_eq()\n");
#endif

  switch(comparison_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop == rop)?TRUE:FALSE;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop == rop)?TRUE:FALSE;
      }
      break;
    case T_BOOL:
      {
	gboolean lop, rop;
	VALUE_INFERENCE(T_BOOL, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_BOOL, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop == rop)?TRUE:FALSE;
      }
      break;
    case T_STRING:
      {
	gchar *lop, *rop;
	VALUE_INFERENCE(T_STRING, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_STRING, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (strcmp(lop, rop) == 0)?TRUE:FALSE;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }


  return(NULL);
}


/**
   'LT':: lower-than comparison operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_lt(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_lt()\n");
#endif

  switch(comparison_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop < rop)?TRUE:FALSE;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop < rop)?TRUE:FALSE;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'GT':: greater-than comparison operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_gt(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_gt()\n");
#endif

  switch(comparison_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop > rop)?TRUE:FALSE;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop > rop)?TRUE:FALSE;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'LEQ':: lower-or-equal comparison operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_leq(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_leq()\n");
#endif

  switch(comparison_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop <= rop)?TRUE:FALSE;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop <= rop)?TRUE:FALSE;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'GEQ':: greater-or-equal comparison operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_geq(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_geq()\n");
#endif

  switch(comparison_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop >= rop)?TRUE:FALSE;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_BOOL;
	CLASS(instr)->value.content.vbool = (lop >= rop)?TRUE:FALSE;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}

valuetype_t arithmetic_allowed_types[6][6] = {
  /*                T_ILLEGAL, T_NONE   , T_INT    , T_FLOAT  , T_BOOL    , 
		    T_STRING                                                 */
  /* T_ILLEGAL */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_NONE    */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_INT     */ { T_ILLEGAL, T_ILLEGAL, T_INT    , T_FLOAT  , T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_FLOAT   */ { T_ILLEGAL, T_ILLEGAL, T_FLOAT  , T_FLOAT  , T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_BOOL    */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_ILLEGAL},
  /* T_STRING  */ { T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL, T_ILLEGAL ,
                    T_ILLEGAL}
};

/**
   'ADD':: add arithmetic operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_add(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_add()\n");
#endif

  switch(arithmetic_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_INT;
	CLASS(instr)->value.content.vint = lop + rop;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_FLOAT;
	CLASS(instr)->value.content.vfloat = lop + rop;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'SUB':: sub arithmetic operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_sub(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_sub()\n");
#endif


  switch(arithmetic_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_INT;
	CLASS(instr)->value.content.vint = lop - rop;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_FLOAT;
	CLASS(instr)->value.content.vfloat = lop - rop;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'MUL':: mul arithmetic operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_mul(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_mul()\n");
#endif


  switch(arithmetic_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_INT;
	CLASS(instr)->value.content.vint = lop * rop;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_FLOAT;
	CLASS(instr)->value.content.vfloat = lop * rop;

	/*printf("mul res: %f*%f = %f\n", lop, rop,
	  CLASS(instr)->value.content.vfloat);*/
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'DIV':: div arithmetic operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_div(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_div()\n");
#endif

  switch(arithmetic_allowed_types[CLASS(instr->arg1)->value.valuetype][CLASS(instr->arg2)->value.valuetype])
    {
    case T_INT:
      {
	gint lop, rop;
	VALUE_INFERENCE(T_INT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_INT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_INT;
	CLASS(instr)->value.content.vint = lop / rop;
      }
      break;
    case T_FLOAT:
      {
	gdouble lop, rop;
	VALUE_INFERENCE(T_FLOAT, &lop, CLASS(instr->arg1));
	VALUE_INFERENCE(T_FLOAT, &rop, CLASS(instr->arg2));

	CLASS(instr)->value.valuetype = T_FLOAT;
	CLASS(instr)->value.content.vfloat = lop / rop;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}


/**
   'UNEG':: uneg arithmetic operator.<br>

   <b>visibility :: <i>public</i></b>

   @param   instr                       the instruction     
   @return  <null>
 */
struct s_instr *exec_uneg(struct s_instr *instr)
{
#ifdef DEBUG
  debug_append("  # executing exec_uneg()\n");
#endif


  switch(CLASS_VALUE(instr->arg1).valuetype)
    {
    case T_INT:
      {
	gint op;
	VALUE_INFERENCE(T_INT, &op, CLASS(instr->arg1));

	CLASS(instr)->value.valuetype = T_INT;
	CLASS(instr)->value.content.vint = - op;
      }
      break;
    case T_FLOAT:
      {
	gdouble op;
	VALUE_INFERENCE(T_FLOAT, &op, CLASS(instr->arg1));

	CLASS(instr)->value.valuetype = T_FLOAT;
	CLASS(instr)->value.content.vfloat = - op;
      }
      break;
    default:
      CLASS(instr)->value.valuetype = T_ILLEGAL;
    }

  return(NULL);
}
