/**
 *  @file    instr.c
 *  @author  Guillaume Bour. 2002
 
 *  @version 0.1
 *    @date  01/02/2002

 *  instructions generation
 */
/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */
#include <stdio.h>
#include <gtk/gtk.h>
#include "instr.h"
#include "variable.h"
#include "list.h"
#include "op.h"
#include "debug.h"
#include "mem.h"

#ifdef DEBUG
  #include "turtle.h"
  #include "variable.h"
  extern struct s_turtle my_turtle;

gchar *op_str[44] = {
  "OP_NOOP",
  "OP_PROCCALL",
  "OP_RETURN",
  "OP_MOVEFORWARD",
  "OP_MOVEBACK",
  "OP_TURNRIGHT",
  "OP_TURNLEFT",
  "OP_SETPOS",
  "OP_SETHEADING",
  "OP_CLEARSCREEN",
  "OP_HIDETURTLE",
  "OP_SHOWTURTLE",
  "OP_HIDEPEN",
  "OP_SHOWPEN",
  "OP_SETPREDEFCOLOR",
  "OP_SETUSERDEFCOLOR",
  "OP_NOTRACE",
  "OP_TRACE",
  "OP_WRITE",
  "OP_STOP",
  "OP_AFFECT",
  "OP_LOCAFFECT",
  "OP_DANCE",
  "OP_SLEEP",
  "OP_GETHEADING",
  "OP_HAZARD",
  "OP_REPEAT",
  "OP_LOOPCNT",
  "OP_IF",
  "OP_ELSE",
  "OP_WHILE",
  "OP_JUMP",
  "OP_AND",
  "OP_OR",
  "OP_EQ",
  "OP_LT",
  "OP_GT",
  "OP_LEQ",
  "OP_GEQ",
  "OP_ADD",
  "OP_SUB",
  "OP_MUL",
  "OP_DIV",
  "OP_UNEG"
};
#endif

/**
   Instruction allocation.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1.the instruction current value is set to <null>

   @param   op                          the instruction operator(ADD, SUB, ...)
   @param   type                        the instruction type(int, bool, ...)
   @param   arg1                        the first argument
   @param   arg2                        the second argument

   @return  the allocated instruction
 */
struct s_instr *instr_alloc(op_t op, classtype_t type, 
			    struct s_class *arg1, struct s_class *arg2)
{
  struct s_instr *myinstr = _XNEW0(struct s_instr, 1);

  CLASS(myinstr)->classtype = CLASS_INSTR;
  CLASS(myinstr)->refcnt = 0;
  CLASS_VALUE(myinstr).valuetype = type;
  myinstr->op = op;

  /* pb de refcnt */
  myinstr->arg1 = arg1;
  myinstr->arg2 = arg2;

  return(myinstr);
}


/**
   Instruction freeing.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1. the second argument is not used (just for GTK compatibility)

   @param   instr                       the instruction to free
   @param   nil                         not used
   @return  <none>
 */
void instr_free_from_list(gpointer instr, gpointer nil)
{
  if(--(CLASS(instr)->refcnt) == 0)
    {
      class_free_from_list(INSTR(instr)->arg1, NULL);
      class_free_from_list(INSTR(instr)->arg2, NULL);

      value_unset(CLASS_VALUE(instr));
      _XFREE(instr);
    }
}


#ifdef DEBUG
void instr_debug(gpointer instr, gpointer nil)
{
  gchar *sarg1, *sarg2;

  sarg1 = class_subdebug(INSTR(instr)->arg1);
  sarg2 = class_subdebug(INSTR(instr)->arg2);
  debug_append2("    (*%d: %s [%s, %s])\n", instr, op_str[INSTR(instr)->op], 
		sarg1, sarg2);

  _XFREE(sarg1);
  _XFREE(sarg2);
}

gchar *instr_subdebug(struct s_instr *instr)
{
  gchar *sinstr = _XNEW(gchar, 25);
  g_snprintf(sinstr, 25, "<instr::*%d>", instr);

  return(sinstr);
}
#endif
