/**
 *  @file    mc.c
 *  @author  Guillaume Bour. 2002
 
 *  @version 0.1
 *    @date  01/02/2002

 *  middle-code for the logo compiler.
 */
/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */
#include <string.h>
#include <gtk/gtk.h>
#include "mc.h"
#include "class.h"
#include "variable.h"

#include "turtle.h"
#include "debug.h"

/*>>> here is the declaration of the turtle            */
struct s_turtle my_turtle;


/**
   Freeing and reinitializing all middle-code informations.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1. you must call this function before any code compilation
   @return  <none>
 */
void mc_init(void)
{
  mc_free();

  /* allocation the main (& default) procedure */
  my_turtle.main_proc = proc_alloc("*:main:*");
  my_turtle.cur_proc = my_turtle.main_proc;
  my_turtle.procedures = g_slist_append(my_turtle.procedures, 
					my_turtle.main_proc);

  class_ref(my_turtle.main_proc);
  class_ref(my_turtle.main_proc);
  class_ref(my_turtle.main_proc);
}


void mc_free(void)
{
  g_slist_foreach(my_turtle.procedures, proc_free_from_list, NULL);
  g_slist_free(my_turtle.procedures);
  my_turtle.procedures = NULL;

  if(my_turtle.cur_proc != NULL)
    { 
      proc_free_from_list(my_turtle.cur_proc, NULL); 
      my_turtle.cur_proc = NULL;
    }
  if(my_turtle.main_proc != NULL)
    { 
      proc_free_from_list(my_turtle.main_proc, NULL); 
      my_turtle.main_proc = NULL;
    }
}

/**
   Entering a new procedure.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
            1. procedures aren't recursives
            1. procedure name must be unique

   @param   name                        the procedure name

   @return  <true> if ok, <false> if a same-name procedure ever exist.
 */
gboolean mc_beginning_procedure(gchar *name)
{
  if(g_slist_find_custom(my_turtle.procedures, name, proc_comparison) != NULL)
    { 
      // error message printing
      // setting error-switch to TRUE
      return(FALSE); 
    }

  class_unref(my_turtle.cur_proc);
  my_turtle.cur_proc = proc_alloc(name);
  class_ref(my_turtle.cur_proc);
  my_turtle.procedures = g_slist_append(my_turtle.procedures, 
					my_turtle.cur_proc);  

  return(TRUE);
}


/**
   Leaving the current procedure.<br>

   <b>visibility :: <i>public</i></b>

   @remarks:
            1. the default procedure is the main one
   @return  <none>
 */
void mc_ending_procedure(void)
{
  class_unref(my_turtle.cur_proc);
  my_turtle.cur_proc = my_turtle.main_proc;
  class_ref(my_turtle.cur_proc);
}


/**
   Adding a parameter to the current procedure.<br>

   <b>visibility :: <i>public</i></b>

   @remarks:
            1. the parameters when the procedure will be call shall be in the
               same order
            2. parameter names must be unique

   @param   name                        the parameter name

   @return  <none>
 */
gboolean mc_add_proc_parameter(gchar *name)
{
  struct s_variable *param;

  if(g_slist_find_custom((my_turtle.cur_proc)->parameters, name, 
			 var_comparison) != NULL)
    { 
      // error message printing
      // setting error-switch to TRUE
      return(FALSE); 
    }

  param = var_alloc(name);
  class_ref(CLASS(param));
  (my_turtle.cur_proc)->parameters = 
    g_slist_append((my_turtle.cur_proc)->parameters, param);
  return(TRUE);
}


/**
   !!! FAUX (l-expr)

   Adding a variable reference (i.e. used in a right-expression).<br>

   <b>visibility :: <i>public</i></b>

   @remarks:
            1. we must check that the variable exist
	       (local variable, global variable or parameter)
   @param   name                        its name

   @return  the mc-instruction
 */
struct s_class *mc_add_variable(var_locality_t locality, gchar *varname)
{
  struct s_variable *var = NULL;
  GSList *varInList = NULL;

  /* 1. we search if the variable already exist */
  /* 0. on recherche les variables comme variable locale */
  varInList = g_slist_find_custom((my_turtle.cur_proc)->variables, varname, 
				  var_comparison);


  /* 1. on recherche les variables comme argument de la fonction */
  if(varInList == NULL)
    {
      varInList = g_slist_find_custom((my_turtle.cur_proc)->parameters, 
				      varname, 
				      var_comparison);
    }

  /* 2. on recherche les variables comme variable globale        */
  if(varInList == NULL)
    {
      varInList = g_slist_find_custom((my_turtle.main_proc)->variables, 
				      varname, var_comparison);
    }

  /*
    in a second step, we will make the distinction between
    the different scopes (global, local, proc arg)
  */
  if(varInList != NULL)
    { 
      class_ref(CLASS(varInList->data)); 
      return(CLASS(varInList->data)); 
    }
    

  var = var_alloc(varname);
  class_ref(CLASS(var));

  if(locality == VAR_GLOBAL)
    {
      (my_turtle.main_proc)->variables = 
	g_slist_append((my_turtle.main_proc)->variables, var);
    } else {
      (my_turtle.cur_proc)->variables = 
	g_slist_append((my_turtle.cur_proc)->variables, var);

    }
  /* !!! we must check that the variable exist, and its type */
  /*var = find_variable(scope, varname, &uvn);
  if(var == NULL)
    { _EMIT_ERROR("variable %s is not declared", varname); } 
  else if(var->type == T_ILLEGAL)
    { 
      _EMIT_ERROR("variable type for %s is unknown", varname);
      var = NULL;
    } 
  */

  return(CLASS(var));
}

/**
 */
struct s_class *mc_add_variable_reference(gchar *varname)
{
  GSList *varInList;

  /* 0. on recherche les variables comme variable locale */
  varInList = g_slist_find_custom((my_turtle.cur_proc)->variables, varname, 
				  var_comparison);


  /* 1. on recherche les variables comme argument de la fonction */
  if(varInList == NULL)
    {
      varInList = g_slist_find_custom((my_turtle.cur_proc)->parameters, 
				      varname, 
				      var_comparison);
    }

  /* 2. on recherche les variables comme variable globale        */
  if(varInList == NULL)
    {
      varInList = g_slist_find_custom((my_turtle.main_proc)->variables, 
				      varname, var_comparison);
    }

  if(varInList == NULL)
    { 
      debug_append2("### unknown variable '%s'\n", varname);
      my_turtle.compilation_failed = TRUE; 
      return(NULL); 
    }
  else
    { return(CLASS(varInList->data)); }
}


/**
   Add a expression with no argument (i.e "CS").<br>

   <b>visibility :: <i>public</i></b>

   @param   op                          the operation      
   @return  the resulting expression
 */
struct s_class *mc_add_noarg_op(op_t op)
{
  struct s_instr *instr;
  valuetype_t type;
  
  //itype = type_is_coherent(op);
  type = T_NONE;
  instr = instr_alloc(op, type, NULL, NULL);
  class_ref(CLASS(instr));
  (my_turtle.cur_proc)->instructions = 
    g_slist_append((my_turtle.cur_proc)->instructions, instr);

  return(CLASS(instr));
}


/**
   Add a unary expression (i.e "TD 10").<br>

   <b>visibility :: <i>public</i></b>

   @param   op                          the operation      
   @param   arg                         the argument
   @return  the resulting expression
 */
struct s_class *mc_add_unary_op(op_t op, struct s_class *arg)
{
  struct s_instr *instr;
  valuetype_t type;
  
  //type = type_is_coherent(op, MC_GET_ARG_TYPE(arg));
  type = T_NONE;
  if(type == T_ILLEGAL)
    { 
      //emitting error
      /*_EMIT_ERROR("invalid opration %s on type %s", 
		  op_to_string(op), type_to_string(MC_GET_ARG_TYPE(arg)));
      */
    }

  instr = instr_alloc(op, type, arg, NULL);
  class_ref(CLASS(instr));
  (my_turtle.cur_proc)->instructions = 
    g_slist_append((my_turtle.cur_proc)->instructions, instr);

  if(arg != NULL)
    { class_ref(arg); }
  return(CLASS(instr));
}


/**
   Add a binary expression (i.e "a + b").<br>

   <b>visibility :: <i>public</i></b>

   @param   op                          the operation      
   @param   arg1                        the first argument
   @param   arg2                        the second argument
   @return  the resulting expression
 */
struct s_class *mc_add_binary_op(op_t op, 
				 struct s_class *arg1, struct s_class *arg2)
{
  struct s_instr *instr;
  valuetype_t type;
  
  //type = type_is_coherent(op, MC_GET_ARG_TYPE(arg1), MC_GET_ARG_TYPE(arg2));
  type = T_NONE;
  if(type == T_ILLEGAL)
    { 
      //emitting error
      /*_EMIT_ERROR("invalid opration %s on types (%s, %s)", 
		  op_to_string(op), 
		  type_to_string(MC_GET_ARG_TYPE(arg1)), 
		  type_to_string(MC_GET_ARG_TYPE(arg2)));
      */
    }

  instr = instr_alloc(op, type, arg1, arg2);
  class_ref(CLASS(instr));
  (my_turtle.cur_proc)->instructions = 
    g_slist_append((my_turtle.cur_proc)->instructions, instr);

  if(arg1 != NULL)
    { class_ref(arg1); }
  if(arg2 != NULL)
    { class_ref(arg2); }

  return(CLASS(instr));
}


/**
   Update the destination of a branch instruction.<br>

   <b>visibility :: <i>public</i></b>

   @remarks:
            1. the destination of a branch is the first argument of the
               branch instruction
	       (the second is the optional condition)

   @param   instr                       the instruction to update
   @param   dest                        the destination

   @return  <none>
 */
void mc_update_branch(struct s_instr *instr, struct s_instr *dest)
{
  //gint index;
  //struct s_literal *lit;

  //index = g_slist_index((my_turtle.cur_proc)->instructions, dest);
  //lit = lit_alloc(T_INT, (gpointer) index);
  //instr->arg1 = mc_alloc(MC_LIT, lit);

  instr->arg1 = CLASS(dest);
  class_ref(dest);
}


struct s_proc *mc_find_procedure(gchar *name)
{
  GSList *elt;

  elt = g_slist_find_custom(my_turtle.procedures, name, proc_comparison);

  if(elt == NULL)
    {  
      debug_append2("### unknown prodecure '%s'\n", name);
      my_turtle.compilation_failed = TRUE; 
    }
  return(elt == NULL?NULL:PROC(elt->data));
}

#ifdef DEBUG
void mc_debug(void)
{
  gint pos;

  if(!my_turtle.draw)
    { gtk_notebook_set_page(GTK_NOTEBOOK(my_turtle.notebook), 1); }

  //debug_clear();
  debug_append("*** Starting debug\n");
  g_slist_foreach(my_turtle.procedures, proc_debug, NULL);
  
  debug_append("*** Ending debug\n");
}
#endif
