/**
 *  @file    proc.c      
 *  @author  Guillaume Bour. 2002
 
 *  @version 0.1
 *    @date  08/02/2002
 *  @version 0.2
 *    @date  26/02/2002 - adding context

 *  procedures management
 */
/*      Copyright (C) 2002 Guillaume Bour
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */
#include <stdio.h>
#include <gtk/gtk.h>
#include "proc.h"
#include "turtle.h"
#include "debug.h"

extern struct s_turtle my_turtle;

/**
   Procedure allocation.<br>

   <b>visibility :: <i>public</i></b>

   @remarks:
            1. the instructions, parameters & variables lists are set to NULL
	    2. the return value is set to NULL

   @param   name                        the procedure name

   @return  the allocated procedure
 */
struct s_proc *proc_alloc(gchar *name)
{
  struct s_proc *myproc = _XNEW0(struct s_proc, 1);

  CLASS(myproc)->classtype = CLASS_PROC;
  CLASS(myproc)->refcnt = 0;
  CLASS_VALUE(myproc).valuetype = T_NONE;

  myproc->name = _XSTRDUP(name);

  return(myproc);
}


/**
   Procedure freeing.<br>

   <b>visibility :: <i>public</i></b>

   @remarks
	    1. we firstly free the internal lists 
               (instructions, parameters & variables)

   @param   proc                        the procedure to free
   @return  <none>
 */
void proc_free_from_list(gpointer myproc, gpointer nil)
{
  if(--(CLASS(myproc)->refcnt) == 0)
    {
      g_slist_foreach(PROC(myproc)->instructions, instr_free_from_list, NULL);
      g_slist_free(PROC(myproc)->instructions);

      g_slist_foreach(PROC(myproc)->parameters, var_free_from_list, NULL);
      g_slist_free(PROC(myproc)->parameters);

      g_slist_foreach(PROC(myproc)->variables, var_free_from_list, NULL);
      g_slist_free(PROC(myproc)->variables);

      /* verify that contexts are NULL */
      if(PROC(myproc)->contexts != NULL)
	{ 
	  /*g_critical*/
	  printf("procedure \"%s\" contexts list not empty !\n", 
		 PROC(myproc)->name);
	  exit();
	  //g_slist_foreach(PROC(myproc)->contexts, context_free_from_list, NULL);
	  //g_slist_free(PROC(myproc)->contexts);
	}
      
      _XFREE(PROC(myproc)->name);
      value_unset(CLASS_VALUE(myproc));
      
      _XFREE(myproc);
    }
}


/**
   Procedure comparison.<br>

   <b>visibility :: <i>public</i></b>

   @param   myproc                      the procedure
   @param   name                        the searched procedure name

   @return  0 is the procedure <i>myproc</i> is named <i>name</i>,
            not 0 else.
 */
gint proc_comparison(gconstpointer myproc, gconstpointer name)
{
  return(strcmp(PROC(myproc)->name, (gchar *)name));
}

#ifdef DEBUG
void proc_debug(gpointer elt, gpointer dec)
{
  debug_append2("==> Procedure(*%d: %s)\n  :parameters\n", 
		PROC(elt), PROC(elt)->name);
  g_slist_foreach(PROC(elt)->parameters, var_debug, NULL);

  debug_append("\n  :variables\n");
  g_slist_foreach(PROC(elt)->variables, var_debug, NULL);

  debug_append("\n  :instructions\n");
  g_slist_foreach(PROC(elt)->instructions, instr_debug, NULL);

  debug_append("\n\n");
}

gchar *proc_subdebug(struct s_proc *proc)
{
  gchar *sproc = _XNEW(gchar, 50);
  g_snprintf(sproc, 50, "<proc::*%d::%s>", proc, proc->name);

  return(sproc);
}
#endif
