/**
 *  @file    rc.c
 *  @author  Guillaume Bour. 2000/2001                                  
 
 *  @version 0.1
 *    @date  2002/09/17
        + first version

 *  read/write rc files (default and user defined)
 *    + initialize environment from rc files

 *
 */

/*      Lafontaine - Graphical logo interpreter
 *      Copyright (C) 2002 Guillaume Bour <gbour@nomade.fr>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *  
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *  
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 *      MA 02111-1307, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include "turtle.h"
#include "rc.h"

struct s_turtle my_turtle;


static const GScannerConfig gtk_rc_scanner_config =
{
  (
   " \t\r\n"
   )			/* cset_skip_characters */,
  (
   G_CSET_a_2_z
   "_"
   G_CSET_A_2_Z
   )			/* cset_identifier_first */,
  (
   G_CSET_a_2_z
   "_-0123456789"
   G_CSET_A_2_Z
   )			/* cset_identifier_nth */,
  ( "#\n" )		/* cpair_comment_single */,
  
  TRUE			/* case_sensitive */,
  
  TRUE			/* skip_comment_multi */,
  TRUE			/* skip_comment_single */,
  TRUE			/* scan_comment_multi */,
  TRUE			/* scan_identifier */,
  FALSE			/* scan_identifier_1char */,
  FALSE			/* scan_identifier_NULL */,
  TRUE			/* scan_symbols */,
  TRUE			/* scan_binary */,
  TRUE			/* scan_octal */,
  TRUE			/* scan_float */,
  TRUE			/* scan_hex */,
  TRUE			/* scan_hex_dollar */,
  TRUE			/* scan_string_sq */,
  TRUE			/* scan_string_dq */,
  TRUE			/* numbers_2_int */,
  FALSE			/* int_2_float */,
  FALSE			/* identifier_2_string */,
  TRUE			/* char_2_token */,
  TRUE			/* symbol_2_token */,
  FALSE			/* scope_0_fallback */,
};

void rc_doit()
{
  printf("reading rc file\n");
  rc_parse("./lafontainerc");
}

/**
   Parse an rc file

   @param   filename              the long rc filename

   @return  <none>

   TESTER LES RETOURS DE FONCTIONS
 */
int rc_parse(const gchar *filename)
{
  int fd, i;
  GScanner *scanner;

  scanner = g_scanner_new(&gtk_rc_scanner_config);//NULL);

  fd = open(filename, O_RDONLY);
  g_scanner_input_file(scanner, fd);

  scanner->input_name = filename;

  
  for(i = 0; i < G_N_ELEMENTS(symbols); i++)
    { 
      g_scanner_scope_add_symbol(scanner, 0, symbols[i].name, 
				 GINT_TO_POINTER(symbols[i].token));

    }

  while(g_scanner_peek_next_token(scanner) != G_TOKEN_EOF)
    { rc_parse_group(scanner); }

  g_scanner_destroy(scanner);
}


/**
   @remarks
     1. group structure: <i>key { ... };</i>
        

 */
void rc_parse_group(GScanner *scanner)
{
  guint token;

  token = g_scanner_get_next_token(scanner); /* key */
  
  if(g_scanner_peek_next_token(scanner) != G_TOKEN_LEFT_CURLY)
    { /* error */ return; }

  g_scanner_get_next_token(scanner); /* eliminating the left curly */
  switch(token)
    {
    case RC_LOCALES:
      rc_parse_locales(scanner);
      break;
    case RC_HISTORY:
      rc_parse_history(scanner);
      break;
    case RC_SAVE:
      rc_parse_coresave(scanner);
      break;
    default:
    }

}

void rc_parse_coresave(GScanner *scanner)
{
  guint token;
  struct s_keyval keyval;

  while(g_scanner_peek_next_token(scanner) != G_TOKEN_RIGHT_CURLY)
    {
      token = g_scanner_peek_next_token(scanner);

      switch(token)
	{
	case RC_COREDUMP:
	  rc_parse_keyval(scanner, &keyval);
	  printf("core dump: %s\n", keyval.value);
	  break;
	case RC_FILE:
	  rc_parse_keyval(scanner, &keyval);
	  printf("coredump file: %s\n", keyval.value);
	  break;
	default:
	}
    }

  g_scanner_get_next_token(scanner); /* eliminating the right curly */
  g_scanner_get_next_token(scanner); /* must test the ; */
}

void rc_parse_history(GScanner *scanner)
{
  guint token;
  struct s_keyval keyval;

  while(g_scanner_peek_next_token(scanner) != G_TOKEN_RIGHT_CURLY)
    {
      token = g_scanner_peek_next_token(scanner);

      switch(token)
	{
	case RC_MAX:
	  rc_parse_keyval(scanner, &keyval);
	  printf("max history: %s\n", keyval.value);
	  break;
	case RC_FILE:
	  rc_parse_keyval(scanner, &keyval);
	  printf("history file: %s\n", keyval.value);
	  break;
	default:
	}
    }

  g_scanner_get_next_token(scanner); /* eliminating the right curly */
  g_scanner_get_next_token(scanner); /* must test the ; */
}

void rc_parse_locales(GScanner *scanner)
{
  guint token;
  struct s_keyval keyval;

  while(g_scanner_peek_next_token(scanner) != G_TOKEN_RIGHT_CURLY)
    {
      token = g_scanner_peek_next_token(scanner);

      switch(token)
	{
	case RC_DEFAULT:
	  rc_parse_keyval(scanner, &keyval);
	  printf("default locale: %s\n", keyval.value);
	  break;
	case RC_DEFAULT_PATH:
	  rc_parse_keyval(scanner, &keyval);
	  printf("default path: %s\n", keyval.value);
	  break;
	default: /* locales */ 
	  rc_parse_alocale(scanner);
	}
    }

  g_scanner_get_next_token(scanner); /* eliminating the right curly */
  g_scanner_get_next_token(scanner); /* must test the ; */
}

void rc_parse_alocale(GScanner *scanner)
{
  guint token;
  struct s_keyval keyval;
  gchar *key, *name, *lib, *desc;

  g_scanner_get_next_token(scanner);
  key = g_strdup(scanner->value.v_string);
  g_scanner_get_next_token(scanner); /* { */

  while(g_scanner_peek_next_token(scanner) != G_TOKEN_RIGHT_CURLY)
    {
      rc_parse_keyval(scanner, &keyval);
      switch(keyval.key)
	{
	case RC_NAME       : name = keyval.value; break;
	case RC_LIB        : lib  = keyval.value; break;
	case RC_DESCRIPTION: desc = keyval.value;
	}
    }

  printf("*** locale ***\n");
  printf("key : %s\n", key);
  printf("name: %s\n", name);
  printf("lib : %s\n", lib);
  printf("desc: %s\n", desc);

  g_scanner_get_next_token(scanner); /* } */
  g_scanner_get_next_token(scanner); /* ; */
}

void rc_parse_keyval(GScanner *scanner, struct s_keyval *keyval)
{
  keyval->key = g_scanner_get_next_token(scanner);
  g_scanner_get_next_token(scanner); /* = */
  g_scanner_get_next_token(scanner);
  keyval->value = g_strdup(scanner->value.v_string);

  g_scanner_get_next_token(scanner); /* ; */

}


