/*

    Rebuffering class
    Copyright (C) 2001 Jussi Laako

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <typeinfo>
#include <deque>


#ifndef REBUFFER2_HH
    #define REBUFFER2_HH


    /**
        Class for splitting data into buffers of different sizes.
        It's implemented using STL deque template.
    */
    class clReBuffer2
    {
            std::deque<float> dfBuffer;
            std::deque<double> ddBuffer;
        protected:
            void *Index (const std::type_info &TypeInfo, long lIndex)
                {
                    if (TypeInfo == typeid(float))
                    {
                        if (lIndex >= dfBuffer.size()) return NULL;
                        return (&dfBuffer[lIndex]);
                    }
                    else if (TypeInfo == typeid(double))
                    {
                        if (lIndex >= ddBuffer.size()) return NULL;
                        return (&ddBuffer[lIndex]);
                    }
                    else return NULL;
                }
        public:
            clReBuffer2 () {}
            clReBuffer2 (const clReBuffer2 &CopySrc)
                {
                    *this = CopySrc;
                }
            ~clReBuffer2 () {}
            /**
                Put data into FIFO.

                \param fpData Source buffer
                \param lCount Number of samples in source.
            */
            void Put (const float *fpData, long lCount)
                {
                    for (long lDataCntr = 0; lDataCntr < lCount; lDataCntr++)
                        dfBuffer.push_back(fpData[lDataCntr]);
                }
            /// \overload
            void Put (const double *dpData, long lCount)
                {
                    for (long lDataCntr = 0; lDataCntr < lCount; lDataCntr++)
                        ddBuffer.push_back(dpData[lDataCntr]);
                }
            /**
                Get data from FIFO.
                Returns false if there's not enough samples in FIFO to
                fill requested buffer.

                \param fpData Destination buffer
                \param lCount Number of samples to fetch
                \return Data available
            */
            bool Get (float *fpData, long lCount)
                {
                    if (dfBuffer.size() < (size_t) lCount) return false;
                    for (long lDataCntr = 0; lDataCntr < lCount; lDataCntr++)
                    {
                        fpData[lDataCntr] = dfBuffer.front();
                        dfBuffer.pop_front();
                    }
                    return true;
                }
            /// \overload
            bool Get (double *dpData, long lCount)
                {
                    if (ddBuffer.size() < (size_t) lCount) return false;
                    for (long lDataCntr = 0; lDataCntr < lCount; lDataCntr++)
                    {
                        dpData[lDataCntr] = ddBuffer.front();
                        ddBuffer.pop_front();
                    }
                    return true;
                }
            /**
                Get number of samples in FIFO.

                \return Number of samples
            */
            long GetCount () const
                { 
                    return ((dfBuffer.size() >= ddBuffer.size()) ? 
                        dfBuffer.size() : ddBuffer.size());
                }
            /**
                Clear FIFO contents and reset buffer size.
            */
            void Clear ()
                {
                    dfBuffer.clear();
                    ddBuffer.clear();
                }
            clReBuffer2 & operator= (const clReBuffer2 &Src)
                {
                    long lSrcCntr;
                    for (lSrcCntr = 0; 
                        lSrcCntr < Src.dfBuffer.size(); 
                        lSrcCntr++)
                    {
                        dfBuffer.push_back(Src.dfBuffer[lSrcCntr]);
                    }
                    for (lSrcCntr = 0;
                        lSrcCntr < Src.ddBuffer.size();
                        lSrcCntr++)
                    {
                        ddBuffer.push_back(Src.ddBuffer[lSrcCntr]);
                    }
                    return (*this);
                }
    };


#endif
