/*

    Class/template system for threading using pthreads
    Copyright (C) 2002 Jussi Laako

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <cstdio>
#include <climits>

#include "DynThreads.hh"


extern "C"
{

void *WrapDynThreadBase (void *vpParam)
{
    clDynThreadsBase::stpParams spParams =
        (clDynThreadsBase::stpParams) vpParam;
    clDynThreadsBase *Klass = spParams->Klass;
    void *vpICParam = spParams->vpParam;

    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
    
    delete spParams;
    return Klass->InternalCaller(vpICParam);
}

}


clDynThreadsBase::clDynThreadsBase ()
{
    iThreadCount = 0;
}


clDynThreadsBase::~clDynThreadsBase ()
{
    try
    {
        MtxBase.Wait();

        ThreadMap_t::iterator iterThreads;
    
        iterThreads = mapThreads.begin();
        while (iterThreads != mapThreads.end())
        {
            pthread_cancel((*iterThreads).second);
            pthread_join((*iterThreads).second, NULL);
            iterThreads++;
        }

        MtxBase.Release();
    }
    catch (...)
    {
    }
}


int clDynThreadsBase::Create (void *vpParam, bool bDetached)
{
    stpParams spParams;
    pthread_t ptidThread;
    int iThreadHandle = -1;
    
    spParams = new stParams;
    spParams->Klass = this;
    spParams->vpParam = vpParam;

    pthread_create(&ptidThread, NULL, WrapDynThreadBase, (void *) spParams);
    if (!bDetached)
    {
        MtxBase.Wait();

        while (mapThreads.find(iThreadCount) != mapThreads.end())
        {
            iThreadCount++;
            if (iThreadCount >= INT_MAX)
                iThreadCount = 0;
        }
        iThreadHandle = iThreadCount;
        mapThreads[iThreadCount++] = ptidThread;
        if (iThreadCount >= INT_MAX)
            iThreadCount = 0;

        MtxBase.Release();
    }
    else
    {
        pthread_detach(ptidThread);
    }
    return iThreadHandle;
}


void *clDynThreadsBase::Wait (int iThreadHandle)
{
    MtxBase.Wait();

    ThreadMap_t::iterator iterThread;
    void *vpReturn = NULL;

    iterThread = mapThreads.find(iThreadHandle);
    if (iterThread != mapThreads.end())
    {
        MtxBase.Release();

        pthread_join((*iterThread).second, &vpReturn);

        MtxBase.Wait();

        mapThreads.erase(iterThread);
    }

    MtxBase.Release();

    return vpReturn;
}


bool clDynThreadsBase::SetSched (pthread_t ptidThread, int iPolicy, int iDelta)
{
    int iEC;
    uid_t uidCurrent;
    struct sched_param sSchedParam;

    // Serialize uid operations
    MtxBase.Wait();
    uidCurrent = getuid();
    setuid(0);
#   ifndef BSDSYS
    sSchedParam.sched_priority = sched_get_priority_min(iPolicy) + iDelta;
#   else
    sSchedParam.sched_priority = iDelta;
#   endif
    iEC = pthread_setschedparam(pthread_self(), iPolicy, &sSchedParam);
    setuid(uidCurrent);
    MtxBase.Release();
    if (iEC != 0) return false;
    return true;
}
