/*

    FFT filter
    Copyright (C) 1999-2001 Jussi Laako

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>

#include "dsp/Filter.hh"


void clFilter::InitCoeffsS ()
{
    long lCoeffCntr;
    stpSCplx spSCCoeffs = CCoeffs;

    for (lCoeffCntr = 0; lCoeffCntr < lSpectPoints; lCoeffCntr++)
    {
        spSCCoeffs[lCoeffCntr].R = 1.0f;
        spSCCoeffs[lCoeffCntr].I = 0.0f;
    }
    ReadyFilterS();
}


void clFilter::InitCoeffsD ()
{
    long lCoeffCntr;
    stpDCplx spDCCoeffs = CCoeffs;

    for (lCoeffCntr = 0; lCoeffCntr < lSpectPoints; lCoeffCntr++)
    {
        spDCCoeffs[lCoeffCntr].R = 1.0;
        spDCCoeffs[lCoeffCntr].I = 0.0;
    }
    ReadyFilterD();
}


void clFilter::ReadyFilterS ()
{
    float *fpCoeffWin = CoeffWin;
    float *fpProc = Proc;
    stpSCplx spSCCoeffs = CCoeffs;
    
    fpCoeffWin = CoeffWin;
    IFFTo(fpProc, spSCCoeffs);
    Mul(fpProc, fpCoeffWin, lFFTSize);
    FFTi(spSCCoeffs, fpProc);
}


void clFilter::ReadyFilterD ()
{
    double *dpCoeffWin = CoeffWin;
    double *dpProc = Proc;
    stpDCplx spDCCoeffs = CCoeffs;

    dpCoeffWin = CoeffWin;
    IFFTo(dpProc, spDCCoeffs);
    Mul(dpProc, dpCoeffWin, lFFTSize);
    FFTi(spDCCoeffs, dpProc);
}


clFilter::clFilter ()
{
    bInitialized = false;
    bUseWindow = false;
}


clFilter::~clFilter ()
{
    if (bInitialized) Uninitialize();
}


bool clFilter::Initialize (long lWindowSize, const float *fpFiltCoeffs,
    bool bUseWindowFunc)
{
    float *fpTemp;
    float *fpCoeffWin;
    clDSPAlloc Temp;

    if (bInitialized) Uninitialize();
    bInitialized = true;
    bUseWindow = bUseWindowFunc;
    lWinSize = lWindowSize;
    lFFTSize = lWinSize * 2;
    lSpectPoints = lFFTSize / 2 + 1;
    Prev.Size(lWinSize * sizeof(float));
    Proc.Size(lFFTSize * sizeof(float));
    CCoeffs.Size(lSpectPoints * sizeof(stSCplx));
    CProc.Size(lSpectPoints * sizeof(stSCplx));
    CoeffWin.Size(lFFTSize * sizeof(float));

    fpTemp = (float *) Temp.Size(lFFTSize * sizeof(float));
    fpCoeffWin = CoeffWin;
    WinKaiserBessel(fpTemp, 4.0f, lFFTSize);
    //WinBlackmanHarris(fpTemp, lFFTSize);
    //WinHanning(fpTemp, lFFTSize);
    Copy(fpCoeffWin, &fpTemp[lFFTSize / 2], lFFTSize / 2);
    Copy(&fpCoeffWin[lFFTSize / 2], fpTemp, lFFTSize / 2);

    FFTInitialize(lFFTSize, true);
    if (fpFiltCoeffs != NULL)
    {
        SetCoeffs(fpFiltCoeffs);
    }
    else
    {
        InitCoeffsS();
    }
    Zero((float *) Prev, lWinSize);
    if (bUseWindow)
    {
        Win.Size(lFFTSize * sizeof(float));
        InvWin.Size(lFFTSize * sizeof(float));
        WinExactBlackman((float *) Win, lFFTSize);
        //WinHanning((float *) Win, lFFTSize);
        Div1x((float *) InvWin, (float *) Win, lFFTSize);
    }
    return true;
}


bool clFilter::Initialize (long lWindowSize, const double *dpFiltCoeffs,
    bool bUseWindowFunc)
{
    double *dpTemp;
    double *dpCoeffWin;
    clDSPAlloc Temp;

    if (bInitialized) Uninitialize();
    bInitialized = true;
    bUseWindow = bUseWindowFunc;
    lWinSize = lWindowSize;
    lFFTSize = lWinSize * 2;
    lSpectPoints = lFFTSize / 2 + 1;
    Prev.Size(lWinSize * sizeof(double));
    Proc.Size(lFFTSize * sizeof(double));
    CCoeffs.Size(lSpectPoints * sizeof(stDCplx));
    CProc.Size(lSpectPoints * sizeof(stDCplx));

    dpTemp = (double *) Temp.Size(lFFTSize * sizeof(double));
    dpCoeffWin = CoeffWin;
    WinKaiserBessel(dpTemp, 4.0, lFFTSize);
    //WinBlackmanHarris(dpTemp, lFFTSize);
    //WinHanning(dpTemp, lFFTSize);
    Copy(dpCoeffWin, &dpTemp[lFFTSize / 2], lFFTSize / 2);
    Copy(&dpCoeffWin[lFFTSize / 2], dpTemp, lFFTSize / 2);

    FFTInitialize(lFFTSize, true);
    if (dpFiltCoeffs != NULL)
    {
        SetCoeffs(dpFiltCoeffs);
    }
    else
    {
        InitCoeffsD();
    }
    Zero((double *) Prev, lWinSize);
    if (bUseWindow)
    {
        Win.Size(lFFTSize * sizeof(double));
        InvWin.Size(lFFTSize * sizeof(double));
        WinExactBlackman((double *) Win, lFFTSize);
        //WinHanning((double *) Win, lFFTSize);
        Div1x((double *) InvWin, (double *) Win, lFFTSize);
    }
    return true;
}


bool clFilter::InitFloat (long lWindowSize, bool bUseWindowFunc)
{
    float *fpNoCoeffs = NULL;

    return Initialize(lWindowSize, fpNoCoeffs, bUseWindowFunc);
}


bool clFilter::InitDouble (long lWindowSize, bool bUseWindowFunc)
{
    double *dpNoCoeffs = NULL;

    return Initialize(lWindowSize, dpNoCoeffs, bUseWindowFunc);
}


void clFilter::Uninitialize ()
{
    FFTUninitialize();
    Win.Free();
    InvWin.Free();
    CoeffWin.Free();
    Prev.Free();
    Proc.Free();
    CCoeffs.Free();
    CProc.Free();
}


void clFilter::SetCoeffs (const float *fpFiltCoeffs)
{
    long lCoeffCntr;
    stpSCplx spSCCoeffs = CCoeffs;

    for (lCoeffCntr = 0; lCoeffCntr < lSpectPoints; lCoeffCntr++)
    {
        spSCCoeffs[lCoeffCntr].R = fpFiltCoeffs[lCoeffCntr];
        spSCCoeffs[lCoeffCntr].I = 0.0f;
    }
    ReadyFilterS();
}


void clFilter::SetCoeffs (const double *dpFiltCoeffs)
{
    long lCoeffCntr;
    stpDCplx spDCCoeffs = CCoeffs;

    for (lCoeffCntr = 0; lCoeffCntr < lSpectPoints; lCoeffCntr++)
    {
        spDCCoeffs[lCoeffCntr].R = dpFiltCoeffs[lCoeffCntr];
        spDCCoeffs[lCoeffCntr].I = 0.0;
    }
    ReadyFilterD();
}


void clFilter::GetCoeffs (float *fpFiltCoeffs)
{
    long lCoeffCntr;
    stpSCplx spSCCoeffs = CCoeffs;

    for (lCoeffCntr = 0; lCoeffCntr < lSpectPoints; lCoeffCntr++)
    {
        fpFiltCoeffs[lCoeffCntr] = spSCCoeffs[lCoeffCntr].R;
    }
}


void clFilter::GetCoeffs (double *dpFiltCoeffs)
{
    long lCoeffCntr;
    stpDCplx spDCCoeffs = CCoeffs;

    for (lCoeffCntr = 0; lCoeffCntr < lSpectPoints; lCoeffCntr++)
    {
        dpFiltCoeffs[lCoeffCntr] = spDCCoeffs[lCoeffCntr].R;
    }
}


void clFilter::Process (float *fpData)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCCoeffs = CCoeffs;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpData, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Mul(spSCProc, spSCCoeffs, lSpectPoints);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpData, lWinSize);
    Copy(fpData, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (double *dpData)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCCoeffs = CCoeffs;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpData, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Mul(spDCProc, spDCCoeffs, lSpectPoints);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpData, lWinSize);
    Copy(dpData, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (float *fpData, const stpSCplx spSCCoeffs)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpData, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Mul(spSCProc, spSCCoeffs, lSpectPoints);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpData, lWinSize);
    Copy(fpData, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (double *dpData, const stpDCplx spDCCoeffs)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCProc = CProc;
    
    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpData, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Mul(spDCProc, spDCCoeffs, lSpectPoints);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpData, lWinSize);
    Copy(dpData, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (float *fpDest, const float *fpSrc)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCCoeffs = CCoeffs;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpSrc, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Mul(spSCProc, spSCCoeffs, lSpectPoints);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpSrc, lWinSize);
    Copy(fpDest, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (double *dpDest, const double *dpSrc)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCCoeffs = CCoeffs;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpSrc, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Mul(spDCProc, spDCCoeffs, lSpectPoints);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpSrc, lWinSize);
    Copy(dpDest, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (float *fpDest, const float *fpSrc, 
    const stpSCplx spSCCoeffs)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpSrc, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Mul(spSCProc, spSCCoeffs, lSpectPoints);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpSrc, lWinSize);
    Copy(fpDest, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::Process (double *dpDest, const double *dpSrc,
    const stpDCplx spDCCoeffs)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpSrc, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Mul(spDCProc, spDCCoeffs, lSpectPoints);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpSrc, lWinSize);
    Copy(dpDest, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessLP (float *fpData)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpData, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Zero(&spSCProc[lLPIdx], lSpectPoints - lLPIdx);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpData, lWinSize);
    Copy(fpData, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessLP (double *dpData)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpData, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Zero(&spDCProc[lLPIdx], lSpectPoints - lLPIdx);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpData, lWinSize);
    Copy(dpData, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessLP (float *fpDest, const float *fpSrc)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpSrc, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Zero(&spSCProc[lLPIdx], lSpectPoints - lLPIdx);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpSrc, lWinSize);
    Copy(fpDest, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessLP (double *dpDest, const double *dpSrc)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpSrc, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Zero(&spDCProc[lLPIdx], lSpectPoints - lLPIdx);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpSrc, lWinSize);
    Copy(dpDest, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessBP (float *fpData)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpData, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Zero(spSCProc, lBPLP);
    Zero(&spSCProc[lBPHP], lSpectPoints - lBPHP);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpData, lWinSize);
    Copy(fpData, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessBP (double *dpData)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpData, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Zero(spDCProc, lBPLP);
    Zero(&spDCProc[lBPHP], lSpectPoints - lBPHP);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpData, lWinSize);
    Copy(dpData, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessBP (float *fpDest, const float *fpSrc)
{
    float *fpWin = Win;
    float *fpInvWin = InvWin;
    float *fpPrev = Prev;
    float *fpProc = Proc;
    stpSCplx spSCProc = CProc;

    Copy(fpProc, fpPrev, lWinSize);
    Copy(&fpProc[lWinSize], fpSrc, lWinSize);
    if (bUseWindow) Mul(fpProc, fpWin, lFFTSize);
    FFTi(spSCProc, fpProc);
    Zero(spSCProc, lBPLP);
    Zero(&spSCProc[lBPHP], lSpectPoints - lBPHP);
    IFFTo(fpProc, spSCProc);
    if (bUseWindow) Mul(fpProc, fpInvWin, lFFTSize);
    Copy(fpPrev, fpSrc, lWinSize);
    Copy(fpDest, &fpProc[lWinSize / 2], lWinSize);
}


void clFilter::ProcessBP (double *dpDest, const double *dpSrc)
{
    double *dpWin = Win;
    double *dpInvWin = InvWin;
    double *dpPrev = Prev;
    double *dpProc = Proc;
    stpDCplx spDCProc = CProc;

    Copy(dpProc, dpPrev, lWinSize);
    Copy(&dpProc[lWinSize], dpSrc, lWinSize);
    if (bUseWindow) Mul(dpProc, dpWin, lFFTSize);
    FFTi(spDCProc, dpProc);
    Zero(spDCProc, lBPLP);
    Zero(&spDCProc[lBPHP], lSpectPoints - lBPHP);
    IFFTo(dpProc, spDCProc);
    if (bUseWindow) Mul(dpProc, dpInvWin, lFFTSize);
    Copy(dpPrev, dpSrc, lWinSize);
    Copy(dpDest, &dpProc[lWinSize / 2], lWinSize);
}


void clFilter::DesignLP (float *fpCorner, int iSampleRate)
{
    long lCoeffIdx;
    float fNyquist;
    float fResolution;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    fNyquist = (float) iSampleRate / 2.0F;
    fResolution = fNyquist / (float) lSpectPoints;
    lCoeffIdx = (long) ceil(*fpCorner / fResolution);
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(&fpCoeffs[lCoeffIdx], lSpectPoints - lCoeffIdx);
    *fpCorner = (float) lCoeffIdx * fResolution;
    lLPIdx = lCoeffIdx;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignLP (double *dpCorner, int iSampleRate)
{
    long lCoeffIdx;
    double dNyquist;
    double dResolution;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    dNyquist = (double) iSampleRate / 2.0;
    dResolution = dNyquist / (double) lSpectPoints;
    lCoeffIdx = (long) ceil(*dpCorner / dResolution);
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(&dpCoeffs[lCoeffIdx], lSpectPoints - lCoeffIdx);
    *dpCorner = (double) lCoeffIdx * dResolution;
    lLPIdx = lCoeffIdx;
    SetCoeffs(dpCoeffs);
}


void clFilter::DesignNarrowLP (float *fpCorner, int iSampleRate)
{
    long lCoeffIdx;
    float fNyquist;
    float fResolution;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    fNyquist = (float) iSampleRate / 2.0F;
    fResolution = fNyquist / (float) lSpectPoints;
    lCoeffIdx = (long) floor(*fpCorner / fResolution);
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(&fpCoeffs[lCoeffIdx], lSpectPoints - lCoeffIdx);
    *fpCorner = (float) lCoeffIdx * fResolution;
    lLPIdx = lCoeffIdx;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignNarrowLP (double *dpCorner, int iSampleRate)
{
    long lCoeffIdx;
    double dNyquist;
    double dResolution;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    dNyquist = (double) iSampleRate / 2.0;
    dResolution = dNyquist / (double) lSpectPoints;
    lCoeffIdx = (long) floor(*dpCorner / dResolution);
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(&dpCoeffs[lCoeffIdx], lSpectPoints - lCoeffIdx);
    *dpCorner = (double) lCoeffIdx * dResolution;
    lLPIdx = lCoeffIdx;
    SetCoeffs(dpCoeffs);
}


void clFilter::DesignHP (float *fpCorner, int iSampleRate)
{
    long lCoeffMax;
    float fNyquist;
    float fResolution;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    fNyquist = (float) iSampleRate / 2.0F;
    fResolution = fNyquist / (float) lSpectPoints;
    lCoeffMax = (long) floor(*fpCorner / fResolution);
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(fpCoeffs, lCoeffMax);
    *fpCorner = (float) lCoeffMax * fResolution;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignHP (double *dpCorner, int iSampleRate)
{
    long lCoeffMax;
    double dNyquist;
    double dResolution;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    dNyquist = (double) iSampleRate / 2.0;
    dResolution = dNyquist / (double) lSpectPoints;
    lCoeffMax = (long) floor(*dpCorner / dResolution);
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(dpCoeffs, lCoeffMax);
    *dpCorner = (double) lCoeffMax * dResolution;
    SetCoeffs(dpCoeffs);
}


void clFilter::DesignBP (float *fpLowCorner, float *fpHighCorner,
    int iSampleRate)
{
    long lCoeffIdx;
    long lCoeffCount1;
    long lCoeffCount2;
    float fNyquist;
    float fResolution;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    fNyquist = (float) iSampleRate / 2.0F;
    fResolution = fNyquist / (float) lSpectPoints;
    lCoeffCount1 = (long) floor(*fpLowCorner / fResolution);
    lCoeffIdx = (long) ceil(*fpHighCorner / fResolution);
    lCoeffCount2 = lSpectPoints - lCoeffIdx;
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(fpCoeffs, lCoeffCount1);
    Zero(&fpCoeffs[lCoeffIdx], lCoeffCount2);
    *fpLowCorner = (float) lCoeffCount1 * fResolution;
    *fpHighCorner = (float) lCoeffIdx * fResolution;
    lBPLP = lCoeffCount1;
    lBPHP = lCoeffIdx;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignBP (double *dpLowCorner, double *dpHighCorner,
    int iSampleRate)
{
    long lCoeffIdx;
    long lCoeffCount1;
    long lCoeffCount2;
    double dNyquist;
    double dResolution;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    dNyquist = (double) iSampleRate / 2.0;
    dResolution = dNyquist / (double) lSpectPoints;
    lCoeffCount1 = (long) floor(*dpLowCorner / dResolution);
    lCoeffIdx = (long) ceil(*dpHighCorner / dResolution);
    lCoeffCount2 = lSpectPoints - lCoeffIdx;
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(dpCoeffs, lCoeffCount1);
    Zero(&dpCoeffs[lCoeffIdx], lCoeffCount2);
    *dpLowCorner = (double) lCoeffCount1 * dResolution;
    *dpHighCorner = (double) lCoeffIdx * dResolution;
    lBPLP = lCoeffCount1;
    lBPHP = lCoeffIdx;
    SetCoeffs(dpCoeffs);
}


void clFilter::DesignNarrowBP (float *fpLowCorner, float *fpHighCorner,
    int iSampleRate)
{
    long lCoeffIdx;
    long lCoeffCount1;
    long lCoeffCount2;
    float fNyquist;
    float fResolution;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    fNyquist = (float) iSampleRate / 2.0F;
    fResolution = fNyquist / (float) lSpectPoints;
    lCoeffCount1 = (*fpLowCorner > 0.0F) ? 
        (long) ceil(*fpLowCorner / fResolution) : 0L;
    lCoeffIdx = (long) floor(*fpHighCorner / fResolution);
    lCoeffCount2 = lSpectPoints - lCoeffIdx;
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(fpCoeffs, lCoeffCount1);
    Zero(&fpCoeffs[lCoeffIdx], lCoeffCount2);
    *fpLowCorner = (float) lCoeffCount1 * fResolution;
    *fpHighCorner = (float) lCoeffIdx * fResolution;
    lBPLP = lCoeffCount1 + 1;
    lBPHP = lCoeffIdx - 1;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignNarrowBP (double *dpLowCorner, double *dpHighCorner,
    int iSampleRate)
{
    long lCoeffIdx;
    long lCoeffCount1;
    long lCoeffCount2;
    double dNyquist;
    double dResolution;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    dNyquist = (double) iSampleRate / 2.0;
    dResolution = dNyquist / (double) lSpectPoints;
    lCoeffCount1 = (*dpLowCorner > 0.0) ?
        (long) ceil(*dpLowCorner / dResolution) : 0L;
    lCoeffIdx = (long) floor(*dpHighCorner / dResolution);
    lCoeffCount2 = lSpectPoints - lCoeffIdx;
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(dpCoeffs, lCoeffCount1);
    Zero(&dpCoeffs[lCoeffIdx], lCoeffCount2);
    *dpLowCorner = (double) lCoeffCount1 * dResolution;
    *dpHighCorner = (double) lCoeffIdx * dResolution;
    lBPLP = lCoeffCount1 + 1;
    lBPHP = lCoeffIdx - 1;
    SetCoeffs(dpCoeffs);
}


void clFilter::DesignBR (float *fpLowCorner, float *fpHighCorner,
    int iSampleRate)
{
    long lCoeffIdx1;
    long lCoeffIdx2;
    long lCoeffCount;
    float fNyquist;
    float fResolution;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    fNyquist = (float) iSampleRate / 2.0F;
    fResolution = fNyquist / (float) lSpectPoints;
    lCoeffIdx1 = (long) floor(*fpLowCorner / fResolution);
    lCoeffIdx2 = (long) ceil(*fpHighCorner / fResolution);
    lCoeffCount = lSpectPoints - lCoeffIdx2;
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(&fpCoeffs[lCoeffIdx1], lCoeffCount);
    *fpLowCorner = (float) lCoeffIdx1 * fResolution;
    *fpHighCorner = (float) lCoeffIdx2 * fResolution;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignBR (double *dpLowCorner, double *dpHighCorner,
    int iSampleRate)
{
    long lCoeffIdx1;
    long lCoeffIdx2;
    long lCoeffCount;
    double dNyquist;
    double dResolution;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    dNyquist = (double) iSampleRate / 2.0;
    dResolution = dNyquist / (double) lSpectPoints;
    lCoeffIdx1 = (long) floor(*dpLowCorner / dResolution);
    lCoeffIdx2 = (long) ceil(*dpHighCorner / dResolution);
    lCoeffCount = lSpectPoints - lCoeffIdx2;
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(&dpCoeffs[lCoeffIdx1], lCoeffCount);
    *dpLowCorner = (double) lCoeffIdx1 * dResolution;
    *dpHighCorner = (double) lCoeffIdx2 * dResolution;
    SetCoeffs(dpCoeffs);
}


void clFilter::DesignLP (float fDivisor)
{
    long lCoeffIdx;
    float *fpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(float));

    fpCoeffs = Coeffs;
    lCoeffIdx = (long) floor(lSpectPoints / fDivisor);
    Set(fpCoeffs, 0.5F, lSpectPoints);
    Zero(&fpCoeffs[lCoeffIdx], lSpectPoints - lCoeffIdx);
    lLPIdx = lCoeffIdx;
    SetCoeffs(fpCoeffs);
}


void clFilter::DesignLP (double dDivisor)
{
    long lCoeffIdx;
    double *dpCoeffs;
    clDSPAlloc Coeffs(lSpectPoints * sizeof(double));

    dpCoeffs = Coeffs;
    lCoeffIdx = (long) floor(lSpectPoints / dDivisor);
    Set(dpCoeffs, 0.5, lSpectPoints);
    Zero(&dpCoeffs[lCoeffIdx], lSpectPoints - lCoeffIdx);
    lLPIdx = lCoeffIdx;
    SetCoeffs(dpCoeffs);
}

