/*

  Copyright 2005 Laurent Wacrenier

  This file is part of libhome

  libhome is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  libhome is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with libhome; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA

*/


#include "config.h"

static char const rcsid[] UNUSED =
"$Id: expand.c,v 1.5 2005/02/18 13:36:21 lwa Exp $";

#include <fnmatch.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#define BUFLEN 128

struct buffer {
  char *data;
  size_t cursor;
  size_t len;
  unsigned failed;
};

struct error {
  char message[512];
};

#undef DEBUG_EXPAND

#ifdef DEBUG_EXPAND
#define DEBUG(a, x...) fprintf(stderr, "***DEBUG " a "\n" , ##x)
#endif

static void error(struct error *err, const char *fmt, ...) {
  if (err->message[0] == 0) {
    va_list ap;
    va_start(ap, fmt);
    vsnprintf(err->message, sizeof(err->message), fmt, ap);
    va_end(ap);
    err->message[sizeof(err->message)-1] = 0;
#ifdef DEBUG_EXPAND
    DEBUG("err=%s", err->message);
#endif
  }
}

static int buffer_cat(char *string, size_t string_len, 
		      struct buffer *b, struct error *err) {
  if (b->failed)
    return -1;
  if (b->cursor + string_len >= b->len) {
    char *data = realloc(b->data, b->len + BUFLEN + string_len);
    if (data == NULL) {
      b->failed = 1;
      free(b->data);
      b->data = NULL;
      error(err, "Cannot allocate memory");
      return -1;
    } else {
      b->data = data;
    }
  }
  memcpy(b->data + b->cursor, string, string_len);
  b->cursor += string_len;
  b->data[b->cursor] = 0;
  return 0;
}

static char *hexpand_shell_full(char **pattern_p,
				char *end,
				char *(lookup(char *name, void *data)),
				void *data,
				struct error *err);

struct substitute_lookup_data {
  void *data;
  char *(*lookup)(char *name, void *data);
  char *empty;
};

static char *substitute_lookup(char *name, void *data) {
  struct substitute_lookup_data *xdata = data;
  if (!*name) {
    return strdup(xdata->empty);
  } else {
    return xdata->lookup(name, xdata->data);
  }
}

#define ANCHOR_START 0x1
#define ANCHOR_END   0x2

static int substitute1(char *pattern,
			char *match, size_t match_len,
			struct buffer *dest, struct error *err) {
  unsigned escaped = 0;
  while(*pattern) {
    if (escaped) {
      escaped = 0;
      if (buffer_cat(pattern, 1, dest, err) == -1)
	return -1;
    } else if (*pattern == '\\') {
      escaped = 1;
    } else if (*pattern == '&') {
      if (buffer_cat(match, match_len, dest, err) == -1)
	return -1;
    } else {
      if (buffer_cat(pattern, 1, dest, err) == -1)
	return -1;
    }
    pattern ++;
  }
  return 0;
}

static char *substitute_replacement(char *start, char *end,
				    char *value,
				    char *(lookup(char *name, void *data)),
				    void *data,
				    struct error *err) {
  struct substitute_lookup_data xdata;
  char *pattern = malloc(end - start + 1);
  char *ret;
  char *tmp;

  if (pattern == NULL)
    return NULL;

  memcpy(pattern, start, end - start);
  pattern[end-start] = 0;

  xdata.data = data;
  xdata.lookup = lookup;
  xdata.empty = value;
  
  tmp = pattern;
  ret = hexpand_shell_full(&tmp, "", substitute_lookup, &xdata, err);
  free(pattern);
  return ret;
}
				    

static char *substitute(char **pattern_p, char *value,
			char *(lookup(char *name, void *data)),
			void *data,
			struct error *err) {
  struct buffer from = {0,};
  struct buffer to = {0,};
  char *pattern = *pattern_p;
  unsigned anchor = 0;
  unsigned global = 0;
  int escaped = 0;
  char *replace_start = NULL;
  char *replace_end = NULL;
  size_t value_len = strlen(value);
  char limit;
  char *from_exp;
  size_t from_exp_len;
  char *tmp;

  if (!*pattern) {
    error(err, "Unterminated substitute");
    return NULL;
  }
  limit = *pattern ++; 

  if (*pattern == '^') {
    anchor |= ANCHOR_START;
    pattern ++;
  }

  while(*pattern) {
    if (escaped) {
      if (buffer_cat(pattern, 1, &from, err) == -1)
	return NULL;
      escaped = 0;
    } else if (*pattern == limit) {
      pattern ++;
      break;
    } else if (*pattern == '\\') {
      escaped = 1;
    } else if (*pattern == '$') {
      if (pattern[1] == limit) {
	anchor |= ANCHOR_END;
      } else {
	if (buffer_cat(pattern, 1, &from, err) == -1)
	  return NULL;
      }
    } else {
      if (buffer_cat(pattern, 1, &from, err) == -1)
	return NULL;
    }
    pattern ++;
  }

  if (!*pattern) {
    free(from.data);
    error(err, "Unterminated substitute at %s", *pattern_p-1);
    return NULL;
  }
  replace_start = pattern;
  
  tmp = from.data;
  from_exp = hexpand_shell_full(&tmp, "", lookup, data, err);
  free(from.data);
  
  if (from_exp == NULL) {
    return NULL;
  }
#ifdef DEBUG_EXPAND
  DEBUG("expanded from: %s", from_exp);
#endif
  from_exp_len = strlen(from_exp);

  while(*pattern) {
    if (escaped) {
      escaped = 0;
    } else if (*pattern == limit) {
      replace_end = pattern;
      pattern ++;
      break;
    } else if (*pattern == '\\') {
      escaped = 1;
    }
    pattern ++;
  }
  if (*pattern == 'g') {
    global = 1;
    pattern ++;
  }
  if (!*pattern) {
    free(from.data);
    error(err, "Unterminated modifier at '%s'", *pattern_p-1);
    return NULL;
  }
  if (*pattern != '}' && *pattern != ':') {
    free(from.data);
    error(err, "Unknown modifier '%c' at '%s'", *pattern, *pattern_p-1);
    return NULL;
  }
  *pattern_p = pattern;

  if (from_exp_len <= value_len) {
    switch(anchor) {
    case ANCHOR_START | ANCHOR_END:
      if (strcmp(value, from_exp) == 0) {
	char *replace = substitute_replacement(replace_start, replace_end,
					       from_exp, lookup, data, err);
	if ( replace == NULL ||
	     substitute1(replace,
			 from_exp, from_exp_len, &to, err) == -1) {
	  free(replace);
	  free(from_exp);
	  return NULL;
	}
      }
      break;
    case ANCHOR_START: 
      if (strncmp(value, from_exp, from_exp_len) == 0) {
	char *replace = substitute_replacement(replace_start, replace_end,
					       from_exp, lookup, data, err);
	if (substitute1(replace,
			 from_exp, from_exp_len, &to, err) == -1 ||
	    buffer_cat(value + from_exp_len, value_len - from_exp_len,
		       &to, err) == -1) {
	  free(replace);
	  free(from_exp);
	  return NULL;
	}
      }
      break;
    case ANCHOR_END:
      if (strncmp(value + value_len - from_exp_len, from_exp, from_exp_len)
	  == 0) {
	char *replace = substitute_replacement(replace_start, replace_end,
					       from_exp, lookup, data, err);
	if ( replace == NULL ||
	     buffer_cat(value, value_len - from_exp_len, &to, err) == -1  ||
	     substitute1(replace,
			 from_exp, from_exp_len, &to, err) == -1 ) {
	  free(replace);
	  free(from_exp);
	  return NULL;
	}
      }
      break;
    case 0: 
      if (from_exp) { /* non empty search string */
	char *here = NULL;
	char *here_prev = value;
	char *replace = NULL;
	while((here = strstr(here_prev, from_exp)) != NULL) {
	  if (replace == NULL) {
	    replace = substitute_replacement(replace_start, replace_end,
					     from_exp, lookup, data, err);
	  }
	  if (replace == NULL || 
	      buffer_cat(here_prev, here - here_prev, &to, err) == -1 ||
	      substitute1(replace,
			  from_exp, from_exp_len, &to, err) == -1) {
	    free(replace);
	    free(from_exp);
	    return NULL;
	  }
	  here_prev = here + from_exp_len;
	  if (!global)
	    break;
	}
	if (replace) { /* something found */
	  free(replace);
	  if (buffer_cat(here_prev, value + value_len - here_prev,
			 &to, err) == -1) {
	    free(from_exp);
	    return NULL;
	  }
	}
      }
      break;
    }
  }
  free(from_exp);
  return to.data ? to.data : value;
}

static int match(char **pattern_p, char *value, struct error *err) {
  char *pattern = *pattern_p;
  char *start = pattern;
  int escaped = 0;
  int ret;
  char *buf;
  while(*pattern) {
    if (escaped) {
      escaped = 0;
    } else if (*pattern == ':' || *pattern == '}') {
      break;
    }
    pattern++;
  }
  *pattern_p = pattern;
  buf = malloc(pattern - start + 1);
  if (buf == NULL) {
    error(err, "Cannot allocate memory");
    return -1;
  }
  strncpy(buf, start, pattern - start);
  buf[pattern - start] = 0;
  ret = fnmatch(buf, value, 0);
#ifdef DEBUG_EXPAND
  DEBUG("fnmatch(%s,%s)=%d", buf, value, ret);
#endif
  free(buf);
  return ret == 0;
}

static char *hexpand_shell_full(char **pattern_p,
				char *end,
				char *(lookup(char *name, void *data)),
				void *data,
				struct error *err
			) {
  char *pattern = *pattern_p;
  struct buffer b = {0,};
  int escaped = 0;

#ifdef DEBUG_EXPAND
  DEBUG("hexpand_shell_full %s", pattern);
#endif

  while(1) {
#ifdef DEBUG_EXPAND
    DEBUG("pattern=%s", pattern);
#endif
    if (*pattern == '\\') {
      escaped = 1;
    } else if (escaped) {
      escaped = 0;
      if (buffer_cat(pattern, 1, &b, err) == -1)
	return NULL;
    } else if (strchr(end, *pattern)) {
      *pattern_p = pattern;
      return b.data;
    } else if (*pattern == '$' && pattern[1] == '{') {
      char *var;
      char *varm;
      char *value;
      (pattern)+=2;
      var = pattern;
      while(*pattern != 0 &&
	    *pattern != ':' && *pattern != '}') {
	pattern++;
      }
      varm = malloc(pattern - var + 1);
      if (varm == NULL) {
	free(b.data);
	error(err, "Cannot allocate memory");
	return NULL;
      }
      memcpy(varm, var, pattern - var);
      varm[pattern - var] = 0;
      value = lookup(varm, data);
      if (value == NULL) {
	free(b.data);
	error(err, "Unassigned variable ${%s} at %s", varm, *pattern_p);
	free(varm);
	return NULL;
      }
      free(varm);
      while (*pattern == ':') {
	pattern ++;
#ifdef DEBUG_EXPAND
	DEBUG("mod=%s", pattern);
#endif
	switch(*pattern) {
	case '?': 
	  {
	    char *ret;
	    pattern ++;
	    ret = hexpand_shell_full(&pattern, ":}",
				     lookup, data, err);
	    if (ret == NULL) {
	      free(value);
	      free(b.data);
	      return NULL;
	    }
	    if (*value == 0) {
	      free(value);
	      value = ret;
	    }
	    break;
	}
	case '+':
	case '-':
	  {
	    long value_l = strtol(value, NULL, 10);
	    char *add = hexpand_shell_full(&pattern, ":}", lookup, data, err);
	    long add_l ;
	    char buffer[sizeof(long) * 8 + 2];
	    int r;
	    if (add == NULL)
	      return NULL;
	    add_l = strtol(add, NULL, 10);
	    free(add);
	    r = snprintf(buffer, sizeof(buffer), "%ld", 
			     value_l + add_l);
	    free(value);
	    if (r > sizeof(buffer)) {
	      error(err, "Numeric overflow at %s", *pattern_p);
	      free(b.data);
	      return NULL;
	    }
	    value = strdup(buffer);
	    if (value == NULL) {
	      free(b.data);
	      return NULL;
	    }
	    break;
	  }
	case 's': 
	  {
	    char *ret;
	    pattern ++;
	    ret = substitute(&pattern, value, lookup, data, err);
	    if (ret == NULL) {
	      free(value);
	      free(b.data);
	      return NULL;
	    }
	    if (ret != value) {
	      free(value);
	      value = ret;
	    }
	    break;
	  }
	case 'u': 
	  {
	    char *v = value;
	    pattern ++;
	    while(*v) {
	      *v = toupper(*v);
	      v++;
	    }
	    break;
	  }
	case 'l':
	  {
	    char *v = value;
	    pattern ++;
	    while(*v) {
	      *v = tolower(*v);
	      v++;
	    }
	    break;
	  }
	case 'm':
	  {
	    int ret;
	    pattern ++;
	    ret = match(&pattern, value, err);
	    if (ret == -1) {
	      free(value);
	      free(b.data);
	      return NULL;
	    }
	    if (!ret) {
	      *value = 0;
	    }
	    break;
	  }
	case 'n':
	  {
	    int ret;
	    pattern ++;
	    ret = match(&pattern, value, err);
	    if (ret == -1) {
	      free(value);
	      free(b.data);
	      return NULL;
	    }
	    if (ret) {
	      *value = 0;
	    }
	    break;
	  }
	}
#ifdef DEBUG_EXPAND
	DEBUG("value=%s", value);
#endif
      }
      if (*pattern != '}') {
	if (*pattern) {
	  error(err, "Unknown modifier '%c' at '%s'", *pattern, *pattern_p);
	} else {
	  error(err, "Unterminated expansion at '%s'", *pattern_p);
	}
	free(value);
	free(b.data);
	return NULL;
      }
      if (value) {
	if (buffer_cat(value, strlen(value), &b, err) == -1) {
	  free(value);
	  return NULL;
	}
	free(value);
      }
    } else {
      if (buffer_cat(pattern, 1, &b, err) == -1)
	return NULL;
    }
    pattern++;
  }
}

char *hexpand_string(char *pattern,
		     char *(lookup(char *name, void *data)),
		     void (error_h(char *message, void *data)),
		     void *data) {
  struct error err;
  char *ret;

  err.message[0] = 0;
  ret = hexpand_shell_full(&pattern, "", lookup, data, &err);
  if (ret == NULL) {
    if (error_h)
      error_h(err.message, data);
  }
  return ret;
}

#ifdef DEBUG_EXPAND
static void error_msg(char *message, void *data) {
    fprintf(stderr, "ERROR: %s\n", message);
}

static char *lookup(char *name, void *data) {
  char **argv  = data;
  int argc = strtoul(name, NULL, 10);
  char *ret ;
  DEBUG("name=%s (%d)", name, argc);
  DEBUG("value=%s", argv[argc]);
  ret = argv[argc];
  //  return strdup(ret ? ret : "");
  return ret ? strdup(ret) : NULL;
}

int main(int argc, char **argv) {
  //  while(1) 
  {
    char *ret = hexpand_string(argv[1], lookup, error_msg, argv+2);
    if (ret) {
      printf("%s\n", ret);
      free(ret);
    }
  }
//  return 0;
}
#endif
