/*

  Copyright 2000, 2001, 2002 Laurent Wacrenier

  This file is part of libhome

  libhome is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  libhome is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with libhome; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA

*/

#include "config.h"

static char const rcsid[] UNUSED =
"$Id: homeuser.c,v 1.18 2005/02/10 14:29:58 lwa Exp $";

#define passwd system_passwd
#include <sys/types.h>
#include <dirent.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "ascii-ctype.h"

#include <errno.h>
#undef passwd

#include "hparam.h"

extern struct param home_param;


static void lowercase(char *c) {
  while(*c) {
    if (ASCII_ISUPPER(*c))
      *c=ASCII_TOLOWER(*c);
    c++;
  }
}

static void uppercase(char *c) {
  while(*c) { 
    if (ASCII_ISLOWER(*c))
      *c=ASCII_TOUPPER(*c);
    c++;
  }
}

static int homedirexists(char *c) {
  DIR * d;

  if ((d=opendir(c))!=NULL) {
    closedir(d);
    return 0;
  }
  if (errno != ENOENT) {
    home_retry("cannot opendir(%s): %s", c, strerror(errno));
    return -1;
  }
  return -1;
}

static char *homecase(char *c) {
  if (c==NULL)
    return NULL;
  
  switch(home_param.homecase) {
  case HCASE_UPPER:
    uppercase(c);
    return c;
  case HCASE_LOWER:
    lowercase(c);
    return c;
  case HCASE_TRYLOWER:
    if (homedirexists(c) != 0) 
      lowercase(c);
    return c;
  case HCASE_TRYUPPER:
    if (homedirexists(c) != 0) 
      uppercase(c);
    return c;
  case HCASE_TRYNULL:
    if (homedirexists(c) != 0) {
      free(c);
      return NULL;
    }
    return c;
  }
  return c;
}

char *hexpand_user(char *user, struct regexp_list *rewrite) {
  char *ret;
  char *c = user;
  
  while(*c) {
    if (!ASCII_ISASCII(*c++))
      return NULL;
  }

  if (strlen(user)>MAXUSERLEN) {
    home_error("username too long: %.20s...", user);
    return NULL;
  }
  ret=hrewrite(rewrite, user, HREW_NONULL);
  if (ret==NULL)
    return NULL;

  switch(home_param.usercase) {
  case HCASE_LOWER:
    lowercase(ret);
    break;
  case HCASE_UPPER:
    uppercase(ret);
    break;
  }

#if WITH_DB
  ret=rewritedb(ret);
#endif

  return ret;
}


char *hexpand_home(char *user, char *path) {
  char *subdir;
  char *home;
  char *user_to_free=NULL;

  if (path == NULL || *path == 0) {   /* le path propos est nul */
    path="~";
  } else if (*path == '/') {
    return strdup(path);
  }

  if (*path == '~') {
    char *slash=strchr(path+1, '/');

    if (slash && *slash) {
      *slash++=0;
      subdir=slash;
    } else {
      subdir="";
    }

    if (path[1])
      user_to_free=user=hexpand_user(path+1, home_param.rewrite);
  } else {
    subdir=path;
  }


  home=hrewrite(home_param.hash, user, HREW_NONULL);

  if (user_to_free)
    free(user_to_free);

  home=homecase(home);

  if (home==NULL)
    return NULL;

  if (*subdir) {
    int home_len=strlen(home);
    int total_len=home_len+strlen(subdir)+2;
    char *epath=(char *)realloc(home, total_len);
    if (epath==NULL) {
      if (home)
	free(home);
      return NULL;
    }
    epath[home_len]='/';
    strcpy(epath+home_len+1, subdir);
    return epath;
  } else {
    return home;
  }

}
