/*

  Copyright 2000, 2001, 2002 Laurent Wacrenier

  This file is part of libhome

  libhome is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  libhome is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with libhome; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA

*/

#include "config.h"

static char const rcsid[] UNUSED =
"$Id: hsystem.c,v 1.10 2005/09/14 08:57:00 lwa Exp $";

/* redefine system struct passwd  */
#define passwd system_passwd
#include <pwd.h>
#undef passwd

#if HAVE_SHADOW_H
#include <shadow.h>
#endif

#include <string.h>
#include <stdlib.h>

#define HOME_DONT_SUBSTITUTE_SYSTEM
#include "hparam.h"

struct pwdx_s {
  struct system_passwd *pwd;
#if HAVE_GETSPNAM
  struct spwd *spwd;
#endif
};

extern int home_stayopen;

static struct pwdx_s *hsystem_query(char *rentry) {
  extern struct param home_param;
  static struct pwdx_s pwdx;
#if HAVE_SETPASSENT
  if (home_param.real_setpassent)
    home_param.real_setpassent(1);
  else
    setpassent(1);
#endif
  if (IS_UID(rentry)) {
    uid_t uid = home_calc(strtoul(GET_UID(rentry), NULL, 10),
			    home_param.uid_calc);
    pwdx.pwd = home_param.real_getpwuid ?
      home_param.real_getpwuid(uid) : getpwuid(uid);
							  
  } else {
    pwdx.pwd = home_param.real_getpwnam ? 
      home_param.real_getpwnam(rentry) : getpwnam(rentry);
  }
  if (pwdx.pwd == NULL)
    return NULL;
#if HAVE_GETSPNAM
  if (home_param.sys_shadow)
    pwdx.spwd = home_param.real_getspwnam ? 
      home_param.real_getspwnam(rentry) : getspnam(rentry);
  else 
    pwdx.spwd = NULL;
#endif
  return &pwdx;
}

static struct passwd *hsystem_store(struct pwdx_s *res, char **alias) {
  extern struct param home_param;
  struct passwd *pwd;

  if (alias)
    *alias = NULL;

  if (res->pwd == NULL)
    return NULL;

  pwd = home_getpwd();
  pwd->pw_name = strdup(res->pwd->pw_name);
#if HAVE_STRUCT_SPWD_SP_PWDP
  pwd->pw_passwd = strdup(res->spwd ?
			  res->spwd->sp_pwdp :
			  res->pwd->pw_passwd);
#else
  pwd->pw_passwd = strdup(res->pwd->pw_passwd);
#endif
  pwd->pw_uid = res->pwd->pw_uid;
  pwd->pw_gid = res->pwd->pw_gid;
  pwd->pw_dir = strdup(res->pwd->pw_dir);
#ifdef HAS_PW_CLASS
  pwd->pw_class = strdup("");
#endif
  pwd->pw_gecos = strdup(res->pwd->pw_gecos);
  pwd->pw_shell = strdup(res->pwd->pw_shell);
#ifdef HAS_PW_CHANGE
  pwd->pw_change=0;
#endif

#ifdef HAS_PW_EXPIRE
#if HAVE_STRUCT_SPWD_SP_EXPIRE
  /* TODO: sp_expire has not the same semantic in glibc or solaris
  if (res->spwd && res->spwd->sp_expire > 0) {
    pwd->pw_expire = 24 * 60 * 60 * res->spwd->sp_expire;
  } else {
    pwd->pw_expire = 0;
  }
  */
  pwd->pw_expire = 0;
#elif HAVE_STRUCT_PASSWD_PW_EXPIRE
  pwd->pw_expire = res->pwd->pw_expire;
#else
  pwd->pw_expire = 0;
#endif
#endif

#ifdef HAS_PW_FIELDS
  /* internal FreeBSD */
#endif
#ifdef HAS_PW_QUOTA
  pwd->pw_quota = home_param.sys_quota * home_param.quota_unit;
#endif
  return pwd;
}

void hsystem_clean(void) {
  extern struct param home_param;
  if (home_param.real_endpwent)
    home_param.real_endpwent();
  else 
    endpwent();
}

struct home_driver hsystem_driver = {
  (home_query_t) hsystem_query,
  (home_store_t) hsystem_store,
  (home_clean_t) hsystem_clean,
};
