/*

  Copyright 2000, 2001, 2002 Laurent Wacrenier

  This file is part of libhome

  libhome is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  libhome is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with libhome; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA

*/

#include "config.h"

static char const rcsid[] UNUSED =
"$Id: cache.c,v 1.8 2005/02/10 14:29:58 lwa Exp $";

#include <sys/types.h>
#include <fcntl.h>

#include <limits.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>

#include <db.h>

#include "hparam.h"

#define BUFCACHE 1024

extern struct param home_param;

static DB *cachedb=NULL;

static DB *opencache() {
  extern int hparam_done;
  char cachefile=NULL;
  if (cachedb)
    return cachedb;
  if (!hparam_done) {
    home_init(NULL);
  }
  if (!home_param.cachefile || !*(home_param.cachefile)) {
    return NULL;;
  }
  if (*home_param.cachefile == '/') {
    cachefile=home_param.cachefile;
  };
  return cachedb=dbopen(home_param.cachefile,
			O_NONBLOCK|O_RDWR|O_CREAT, 0600, DB_HASH, NULL);
}

void closecache(void) {
  if (cachedb) {
    cachedb->close(cachedb);
    cachedb=NULL;
  }
}

#define STORESTRING(e)      if (e) { t = e; while ((*p++ = *t++)); } else { *p++=0; }
#define STOREINT(e)      memmove(p, &(e), sizeof(e)); p+=sizeof(e);

void storecache(struct passwd *pwd) {
  char buf[BUFCACHE];
  char *t;
  time_t stamp;
  char *p=buf;
  DBT key, value;
  
  char *tmp;

  if (opencache()==NULL) {
    return;
  }

  time(&stamp);
  STOREINT(stamp);
  
  STORESTRING(pwd->pw_name);
  STORESTRING(pwd->pw_passwd);
  STOREINT(pwd->pw_uid);
  STOREINT(pwd->pw_gid);
#ifdef HAS_PW_QUOTA
  STOREINT(pwd->pw_quota);
#endif
#ifdef HAS_PW_CLASS
  STORESTRING(pwd->pw_class);
#endif
  STORESTRING(pwd->pw_gecos);
  STORESTRING(pwd->pw_dir);
  STORESTRING(pwd->pw_shell);
#ifdef HAS_PW_EXPIRE
  STOREINT(pwd->pw_expire);
#endif

  key.data=pwd->pw_name;
  key.size=strlen(key.data);
  value.data=buf;
  value.size=p-buf;

  cachedb->put(cachedb, &key, &value, 0);
  cachedb->sync(cachedb, 0);
}

#define EXPAND(e)       e = t; while ( (*t++ = *p++) );
#define SCALAR(v)       memmove(&(v), p, sizeof v); p += sizeof v


struct passwd *retrfromcache(char *name) {

  DBT key, value;
  static char *mem=NULL;
  static struct passwd pwd;
  if (opencache()==NULL) {
    return;
  }
  key.data=name;
  key.size=strlen(key.data);
  if (cachedb->get(cachedb, &key, &value, 0)==0) {
    char *p = value.data;
    char *t;
    time_t stamp;
    long timeout=home_param.cacheto;

    mem=realloc(mem, value.size);
    t=mem;

    if (mem==NULL) {
      return NULL;
    }

    SCALAR(stamp);
    if (timeout>0 && stamp<time(NULL) - timeout) {
      cachedb->del(cachedb, &key, 0);
      return NULL;
    }
      
    EXPAND(pwd.pw_name);
    EXPAND(pwd.pw_passwd);
    SCALAR(pwd.pw_uid);
    SCALAR(pwd.pw_gid);
#ifdef HAS_PW_QUOTA
    SCALAR(pwd.pw_quota);
#endif
#ifdef HAS_PW_CLASS
    EXPAND(pwd.pw_class);
#endif
    EXPAND(pwd.pw_gecos);
    EXPAND(pwd.pw_dir);
    EXPAND(pwd.pw_shell);
#ifdef HAS_PW_EXPIRE
    SCALAR(pwd.pw_expire);
#endif

    return &pwd;
  } 
  return NULL;
}
