/*

  Copyright 2000, 2001, 2002 Laurent Wacrenier

  This file is part of libhome

  libhome is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  libhome is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with libhome; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA

*/

/* $Id: hparam.h,v 1.50 2005/06/29 09:41:20 lwa Exp $ */



#ifndef _HPARAM_H_
#define _HPARAM_H_

#include "config.h"

#include <sys/types.h>
#include <sys/time.h>
#include <regex.h>
#include <hpwd.h>

#ifdef DMALLOC
#include <dmalloc.h>
#endif

#include <limits.h>

#define IS_UID(x) (*x == '\xff')
#define GET_UID(x) (x+1)

#ifndef DEFAULT_HOME_CONF
#define DEFAULT_HOME_CONF "home.conf"
#endif

struct regexp_list {
  regex_t *preg;
  char *action;
  struct regexp_list *next;
};

#define HOME_MODE_MYSQL     0
#define HOME_MODE_LDAP      1
#define HOME_MODE_SYSTEM    2
#define HOME_MODE_PAM       3
#define HOME_MODE_PGSQL     4
#define HOME_MODE_PROXY     5

#define HOME_MODE_DEFAULT HOME_MODE_MYSQL

typedef void* (*home_query_t) (char *);
typedef struct passwd* (*home_store_t) (void *, char **);
typedef void (*home_clean_t) (void);

struct home_driver {
  home_query_t query;
  home_store_t store;
  home_clean_t clean;
};


struct param {

  int mode;
  char *query;

#if WITH_MYSQL
  /* mysql stuff */
  char **my_hosts;
  char *my_database;
  char *my_user;
  char *my_passwd;
  int my_port;
  int my_connect_timeout;
#endif

#if WITH_PGSQL
  char **pg_hosts;
  char *pg_database;
  char *pg_user;
  char *pg_passwd;
#endif

#if WITH_MYSQL || WITH_PGSQL
  int sql_partial;
#endif

#if WITH_LDAP
  /* ldap stuff */
  char *ld_hosts;
  char *ld_dn;
  char *ld_passwd;
  int ld_crypt;
  int ld_version;
  struct regexp_list *ld_base;
  int ld_timeout;
  char **ld_attrs;
#endif

#if WITH_PAM
  char *pam_service;
#endif

#if WITH_PROXY
  char *proxy_socket;
  char **proxy_deny;
#endif

#if WITH_MYSQL || WITH_PGSQL || WITH_LDAP
  char *where_uid;
  char *where_nam;
#endif

  char **pwuid;

  struct home_driver *driver;

  int retries;
  int retry_delay;

  /* field retrieval */
  char *pw_name;
  char *pw_passwd;
  struct regexp_list *passwd_rew;
  int crypt_always_crypted;
  char *pw_uid;
  char *pw_gid;
  char *pw_quota;
  int quota_unit;
  char *pw_class;
  char *pw_gecos;
  char *pw_change;
  char *pw_dir;
  char *pw_shell;
  char *pw_expire;
  char *pw_alias;
  
  int backtime; /* temps pour revenir d'un serveur de backup */

  char *errmsg;
  char *expire_fmt;

  char usercase;
  char homecase;
  int log_stderr;

  char **uid_calc;

  /* regex stuff */
  struct regexp_list *rewrite;
  struct regexp_list *fallback;
  struct regexp_list *hash;

  /* settings for system method */
  int sys_quota;
  int sys_shadow; /* use shadow password infos */

  void * (*real_getpwnam)(char *login);
  void * (*real_getpwuid)(uid_t uid);
#if HAVE_GETSPNAM
  void * (*real_getspwnam)(char *login);
#endif
  void (*real_endpwent)(void);
#if HAVE_SETPASSENT
  int (*real_setpassent)(int);
#endif

  char **pures;

#if WITH_DB
  /* cache structure */
  char *cachefile;
  int cachettl;     /* timeout */
  int cacherevivettl;  /* may be used when transcient error */
  int cachesize;    /* cache size in Kb */
  int cachelockers; /* number of cache lockers */

  char **rewritedb; /* rewritting db */
#endif
};

/* cases */
#define HCASE_NONE  0
#define HCASE_UPPER 1
#define HCASE_LOWER 2

#define HCASE_TRYLOWER 3
#define HCASE_TRYUPPER 4
#define HCASE_TRYNULL  5

#define MAXUSERLEN    96   /* taille max d'un compte utilisateur */

#ifdef LINE_MAX
#  define LINEMAX LINE_MAX
#else
#  ifdef _POSIX2_LINE_MAX
#    define LINEMAX _POSIX2_LINE_MAX
#  else
#    define LINEMAX     2048   /* taille max d'une ligne de conf */
#  endif
#endif


/*
 * hparam.c
 */

struct param *home_init(char *file);
void home_clean(void);

/*
 *  tools.c
 */

struct passwd *home_getpwd(void);   /* return empty passwd structure */
time_t home_expire(char *s);

/*
 * rewrite.c 
 */


char *hrewrite(struct regexp_list *rl, char *line, const int options);

/* hrewrite options */

#define HREW_FREE    0x01         /* free() line */
#define HREW_NONULL  0x02         /* transformed line should never be empty */
#define HREW_MUST    0x04         /* all patterns must match */
#define HREW_FIRST   0x08         /* stop on first match */

unsigned long home_calc(unsigned long n, char **formula);
unsigned long home_uncalc(unsigned long n, char **formula);


/*
 * homeuser.c
 */

char *hexpand_user(char *user, struct regexp_list *rl);
char *hexpand_home(char *user, char *path);

/*
 * error.c
 */ 

int home_error(char *fmt, ...);   /* erreur anondine (pas d'utilisateur, ...)*/
int home_retry(char *fmt, ...);   /* problme de config, timeout... */
void *hmalloc_error(char *context, char *var);  /* erreur de malloc */
int home_has_transcient_condition(void);    /* check transcient condition */
void home_clear_transcient_condition(void);  /* reset transcient condition */

/* do the query and retry is needed */
void *home_query(home_query_t query, char *who);

/* retourne p ou NULL (avec errno=ENOMEM) selon que home_retry a t lanc */
struct passwd *home_getpwnam_return(struct passwd *p);


/*
 * cache.c
 */

struct passwd *retrfromcache(char *name, int ttl); /* cherche dans le cache */
void storecache(char *name, struct passwd *pwd);      /* range dans le cache */

/* 
 *  rewritedb.c
 */

#if WITH_DB
char *rewritedb(char *);        /* rcrit un compte d'aprs une base */
#endif

/*
 *  cleanup.c
 */

void home_cleanup(void);        /* nettoye ce qui est allou ou ouvert */

/*
 * wrap.c
 */

char *home_canon(char *user);            /* canonifie le compte */
char *home_crypt(char *key, char *salt); /* crypt passwd when needed */
#define HOME_CRYPTED_PLAIN 0
#define HOME_CRYPTED_CRYPT 1
#define HOME_CRYPT_CRYPT_EXTRA_CHARS 32
int home_crypted(char *passwd);          /* tell if passwd is crypted */
void home_blocsignal(int mode);     /* block/unblock signals */

/* 
 * expand.c
 */
char *hexpand_string(char *pattern,
		     char *(lookup(char *name, void *data)),
		     void (error(char *message, void *data)),
		     void *data);

#endif /* _HPARAM_H_ */
