/*
 * @file libleaftag/cache.c Cache functions
 *
 * @Copyright (C) 2005-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libleaftag/cache.h>
#include <glib.h>

G_LOCK_DEFINE_STATIC(cache_lock);
static GHashTable *sources_cache = NULL;
static GHashTable *tags_cache = NULL;

static gpointer
lt_cache_get(GHashTable *cache, const char *key)
{
	if (cache == NULL)
		return NULL;

	return g_hash_table_lookup(cache, key);
}

static void
lt_cache_add(GHashTable **cache, const char *key, gpointer data)
{
	gpointer existing_data;

	g_return_if_fail(key != NULL && *key != '\0');

	if (*cache == NULL)
	{
		*cache = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
	}

	existing_data = g_hash_table_lookup(*cache, key);

	if (existing_data != NULL)
		g_assert(existing_data == data);
	else
		g_hash_table_insert(*cache, g_strdup(key), data);
}

static void
lt_cache_remove(GHashTable **cache, const char *key, gpointer data)
{
	g_return_if_fail(key != NULL && *key != '\0');

	if (*cache == NULL)
		return;

	g_hash_table_remove(*cache, key);
}

LtSource *
lt_cache_get_source(const char *uri)
{
	g_return_val_if_fail(uri != NULL,  NULL);
	g_return_val_if_fail(*uri != '\0', NULL);

	return (LtSource *)lt_cache_get(sources_cache, uri);
}

void
lt_cache_add_source(LtSource *source)
{
	g_return_if_fail(source != NULL);

	G_LOCK(cache_lock);
	lt_cache_add(&sources_cache, lt_source_get_uri(source), source);
	G_UNLOCK(cache_lock);
}

void
lt_cache_remove_source(LtSource *source)
{
	g_return_if_fail(source != NULL);

	G_LOCK(cache_lock);
	lt_cache_remove(&sources_cache, lt_source_get_uri(source), source);
	G_UNLOCK(cache_lock);
}

LtTag *
lt_cache_get_tag(const char *name)
{
	g_return_val_if_fail(name != NULL,  NULL);
	g_return_val_if_fail(*name != '\0', NULL);

	return (LtTag *)lt_cache_get(tags_cache, name);
}

void
lt_cache_add_tag(LtTag *tag)
{
	g_return_if_fail(tag != NULL);

	G_LOCK(cache_lock);
	lt_cache_add(&tags_cache, lt_tag_get_name(tag), tag);
	G_UNLOCK(cache_lock);
}

void
lt_cache_remove_tag(LtTag *tag)
{
	g_return_if_fail(tag != NULL);

	G_LOCK(cache_lock);
	lt_cache_remove(&tags_cache, lt_tag_get_name(tag), tag);
	G_UNLOCK(cache_lock);
}
