/*
 * @file libleaftag/db.c Database functions
 *
 * @Copyright (C) 2005-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libleaftag/db.h>
#include <libleaftag/db-priv.h>
#include <glib-object.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

G_LOCK_DEFINE_STATIC(__db_filename_lock);
G_LOCK_DEFINE_STATIC(__db_handle_lock);

static char *__db_filename = NULL;
static sqlite *__db_handle = NULL;

/**
 * lt_db_set_filename
 * @filename: The filename of the database
 *
 * Set the database file.  This should only be used if your application should
 * use a database other than the primary one.
 */
void
lt_db_set_filename(const char *filename)
{
	g_return_if_fail(!lt_db_is_initted());

	G_LOCK(__db_filename_lock);

	g_free(__db_filename);
	__db_filename = (filename == NULL ? NULL : g_strdup(filename));

	G_UNLOCK(__db_filename_lock);
}

static gboolean
database_valid(LtDbResults *results)
{
	GList *row;

	if (g_list_length(results->rows) != 3)
		return FALSE;

	row = results->rows;
	if (strcmp(lt_db_row_get((LtDbRow *)row->data, "tbl_name"), "associations"))
		return FALSE;

	row = row->next;
	if (strcmp(lt_db_row_get((LtDbRow *)row->data, "tbl_name"), "sources"))
		return FALSE;

	row = row->next;
	if (strcmp(lt_db_row_get((LtDbRow *)row->data, "tbl_name"), "tags"))
		return FALSE;

	return TRUE;
}

/**
 * lt_db_init
 *
 * Initialize the database connection.
 */
void
lt_db_init(void)
{
	char *errmsg;
	char *filename;
	LtDbResults *results;

	g_return_if_fail(!lt_db_is_initted());

	if (__db_filename == NULL)
	{
		filename = g_build_filename(g_get_home_dir(), ".tags.db", NULL);
		lt_db_set_filename(filename);
		g_free(filename);
	}

	G_LOCK(__db_handle_lock);
	__db_handle = sqlite_open(__db_filename, 0666, &errmsg);
	G_UNLOCK(__db_handle_lock);

	if (__db_handle == NULL)
	{
		g_error("Unable to open the leaftag database (%s): %s",
				__db_filename, errmsg);
		sqlite_freemem(errmsg);
		return;
	}

	/* Create the schema if it doesn't exist. */
	results = lt_db_query("SELECT tbl_name FROM sqlite_master WHERE "
						  "type='table' ORDER BY tbl_name");

	if (results == NULL)
	{
		lt_db_exec("CREATE TABLE sources (\n"
				   "    id     INTEGER PRIMARY KEY,\n"
				   "    uri    TEXT    UNIQUE NOT NULL,\n"
				   "    schema TEXT,\n"
				   "    ctime  TIMESTAMP\n"
				   ")");
		lt_db_exec("CREATE TABLE tags (\n"
				   "    id          INTEGER PRIMARY KEY,\n"
				   "    name        TEXT UNIQUE NOT NULL,\n"
				   "    description TEXT,\n"
				   "    image       TEXT,\n"
				   "    hidden      BOOLEAN DEFAULT FALSE,\n"
				   "    ctime       TIMESTAMP\n"
				   ")");
		lt_db_exec("CREATE TABLE associations (\n"
				   "    source_id INTEGER,\n"
				   "    tag_id    INTEGER\n"
				   ")");
	}
	else
	{
		if (!database_valid(results))
		{
			g_error("Unknown tag database tables. The database may be "
					"corrupt or incompatible. If this is an old database, "
					"you will want to temporarily downgrade, export the "
					"old database using tagutils, and then import them in "
					"this version.");
		}

		lt_db_results_destroy(results);
	}
}

/**
 * lt_db_uninit
 *
 * Shut down the database connection.
 */
void
lt_db_uninit(void)
{
	if (lt_db_is_initted())
	{
		G_LOCK(__db_handle_lock);
		sqlite_close(__db_handle);
		__db_handle = NULL;
		G_UNLOCK(__db_handle_lock);
	}
}

/* #define DEBUG_SQL */

static void
dump_sql(const char *sql, va_list args)
{
#ifdef DEBUG_SQL
	char *query = sqlite_vmprintf(sql, args);
	printf("SQL: %s\n", query);
	sqlite_freemem(query);
#endif
}

LtDbResults *
lt_db_query(const char *sql, ...)
{
	va_list args;
	LtDbResults *results = NULL;
	int nrow, ncol;
	int row, col;
	char **resultp;
	char *errmsg;
	sqlite *db = lt_get_db();

	g_return_val_if_fail(sql != NULL, NULL);

	va_start(args, sql);

	dump_sql(sql, args);

	if (sqlite_get_table_vprintf(db, sql, &resultp, &nrow,
								 &ncol, &errmsg, args) != SQLITE_OK)
	{
		char *query = g_strdup_vprintf(sql, args);
		g_error("Unable to get results from query (%s): %s", query, errmsg);
		g_free(query);
		sqlite_freemem(errmsg);
	}
	else if (nrow > 0)
	{
		results = g_new0(LtDbResults, 1);

		for (row = 1; row <= nrow; row++)
		{
			LtDbRow *dbrow = g_new0(LtDbRow, 1);
			dbrow->cols = g_hash_table_new_full(g_str_hash, g_str_equal,
												g_free, g_free);
			results->rows = g_list_append(results->rows, dbrow);

			for (col = 0; col < ncol; col++)
			{
				g_hash_table_insert(dbrow->cols, g_strdup(resultp[col]),
									g_strdup(resultp[(row * ncol) + col]));
			}
		}

		sqlite_free_table(resultp);
	}

	va_end(args);
	return results;
}

void
lt_db_exec(const char *sql, ...)
{
	va_list args;
	sqlite *db = lt_get_db();
	char *errmsg;

	g_return_if_fail(sql != NULL);

	va_start(args, sql);
	dump_sql(sql, args);

	if (sqlite_exec_vprintf(db, sql, NULL, NULL, &errmsg, args) != SQLITE_OK)
	{
		g_error("Unable to perform query: %s", errmsg);
		sqlite_freemem(errmsg);
	}

	va_end(args);
}

static void
dump_row(const char *key, const char *value, gpointer unused)
{
	printf("%s: %s\n", key, value);
}

void
lt_db_dump_results(LtDbResults *results)
{
	GList *row;

	g_return_if_fail(results != NULL);

	for (row = results->rows; row != NULL; row = row->next)
	{
		LtDbRow *row_data;

		if (row != results->rows)
			printf("--\n");

		row_data = (LtDbRow *)row->data;
		g_hash_table_foreach(row_data->cols, (GHFunc)dump_row, NULL);
	}
}

/**
 * lt_db_is_initted
 *
 * Check whether the database is opened.
 *
 * Returns: whether or not the database connection is active.
 */
gboolean
lt_db_is_initted(void)
{
	return __db_handle != NULL;
}

sqlite *
lt_get_db(void)
{
	if (!lt_db_is_initted())
		lt_db_init();

	return __db_handle;
}

const char *
lt_db_row_get(LtDbRow *row, const char *key)
{
	g_return_val_if_fail(row != NULL,  NULL);
	g_return_val_if_fail(key != NULL,  NULL);
	g_return_val_if_fail(*key != '\0', NULL);

	return (const char *)g_hash_table_lookup(row->cols, key);
}

static void
destroy_row(LtDbRow *row)
{
	g_hash_table_destroy(row->cols);
	g_free(row);
}

void
lt_db_results_destroy(LtDbResults *results)
{
	g_return_if_fail(results != NULL);

	g_list_foreach(results->rows, (GFunc)destroy_row, NULL);
	g_list_free(results->rows);
	g_free(results);
}

char *
lt_db_build_multiple_where_clause(const char *col_prefix, const char *col_name,
								  GList *items)
{
	char *column;
	GString *str;
	GList *l;

	g_return_val_if_fail(col_name != NULL && *col_name != '\0', NULL);
	g_return_val_if_fail(items != NULL, NULL);

	str = g_string_new(NULL);

	column = lt_db_prepend_col_prefix(col_prefix, col_name);
	g_string_append_printf(str, "%s IN (", column);
	g_free(column);

	for (l = items; l != NULL; l = l->next)
	{
		char *temp;

		if (l != items)
			g_string_append(str, ", ");

		temp = sqlite_mprintf("%Q", (char *)l->data);
		g_string_append(str, temp);
		sqlite_freemem(temp);
	}

	g_string_append(str, ")");

	return g_string_free(str, FALSE);
}

char *
lt_db_build_sources_where_clause(GList *uris, const char *col_prefix)
{
	g_return_val_if_fail(uris != NULL, NULL);

	return lt_db_build_multiple_where_clause(col_prefix, "uri", uris);
}

char *
lt_db_build_tags_where_clause(GList *tags, const char *col_prefix)
{
	g_return_val_if_fail(tags != NULL, NULL);

	return lt_db_build_multiple_where_clause(col_prefix, "name", tags);
}

char *
lt_db_prepend_col_prefix(const char *prefix, const char *colname)
{
	if (prefix == NULL)
		return g_strdup(colname);

	return g_strdup_printf("%s.%s", prefix, colname);
}
