/*
 * @file libleaftag/object.c Base object
 *
 * @Copyright (C) 2005-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libleaftag/object.h>
#include <libleaftag/priv.h>

struct _LtObjectPriv
{
	guint id;
	gboolean in_db;
};

enum
{
	PROP_0,
	PROP_ID,
	PROP_IN_DB
};

enum
{
	DELETED,
	LAST_SIGNAL
};

static void lt_object_finalize(GObject *gobject);
static void lt_object_set_property(GObject *gobject, guint prop_id,
								   const GValue *value, GParamSpec *pspec);
static void lt_object_get_property(GObject *gobject, guint prop_id,
								   GValue *value, GParamSpec *pspec);

static GObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(LtObject, lt_object, G_TYPE_OBJECT);

static void
lt_object_class_init(LtObjectClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	gobject_class->finalize     = lt_object_finalize;
	gobject_class->set_property = lt_object_set_property;
	gobject_class->get_property = lt_object_get_property;

	/**
	 * LtObject::deleted
	 * @object: The object which was deleted
	 *
	 * The ::deleted signal is emitted when the object is removed from
	 * the database.
	 */
	signals[DELETED] =
		g_signal_new("deleted",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(LtObjectClass, deleted),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	g_object_class_install_property(gobject_class, PROP_ID,
		g_param_spec_uint("id", "ID",
						  "The ID of the object",
						  0, G_MAXINT, 0,
						  G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property(gobject_class, PROP_IN_DB,
		g_param_spec_boolean("in-db", "In DB",
							 "Is the object stored in the database",
							 FALSE,
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}

static void
lt_object_init(LtObject *object)
{
	object->priv = g_new0(LtObjectPriv, 1);
}

static void
lt_object_finalize(GObject *gobject)
{
	LtObject *object = LT_OBJECT(gobject);

	if (object->priv != NULL)
	{
		g_free(object->priv);
		object->priv = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->finalize != NULL)
		G_OBJECT_CLASS(parent_class)->finalize(gobject);
}

static void
lt_object_set_property(GObject *gobject, guint prop_id,
					   const GValue *value, GParamSpec *pspec)
{
	LtObject *object = LT_OBJECT(gobject);

	switch (prop_id)
	{
		case PROP_ID:
			object->priv->id = g_value_get_uint(value);
			break;

		case PROP_IN_DB:
			lt_object_set_in_db(object, g_value_get_boolean(value));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

static void
lt_object_get_property(GObject *gobject, guint prop_id, GValue *value,
					   GParamSpec *pspec)
{
	LtObject *object = LT_OBJECT(gobject);

	switch (prop_id)
	{
		case PROP_ID:
			g_value_set_uint(value, lt_object_get_id(object));
			break;

		case PROP_IN_DB:
			g_value_set_boolean(value, lt_object_get_in_db(object));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

void
lt_object_set_id(LtObject *obj, guint id)
{
	g_return_if_fail(obj != NULL);
	g_return_if_fail(LT_IS_OBJECT(obj));

	obj->priv->id = id;
}

/**
 * lt_object_get_id
 * @obj: An #LtObject
 *
 * Retreive the ID of the object.
 *
 * Returns: The ID of the object.
 */
guint
lt_object_get_id(const LtObject *obj)
{
	g_return_val_if_fail(obj != NULL, 0);
	g_return_val_if_fail(LT_IS_OBJECT(obj), 0);

	return obj->priv->id;
}

void
lt_object_set_in_db(LtObject *obj, gboolean in_db)
{
	g_return_if_fail(obj != NULL);
	g_return_if_fail(LT_IS_OBJECT(obj));

	obj->priv->in_db = in_db;
}

/**
 * lt_object_get_in_db
 * @obj: An #LtObject
 *
 * Look up whether or not the object is in the database.
 *
 * Returns: TRUE if the object exists in the database. FALSE otherwise.
 */
gboolean
lt_object_get_in_db(const LtObject *obj)
{
	g_return_val_if_fail(obj != NULL, FALSE);
	g_return_val_if_fail(LT_IS_OBJECT(obj), FALSE);

	return obj->priv->in_db;
}

/**
 * lt_free_object_list
 * @list: A list of #LtObjects
 *
 * Frees a GList and unrefs all objects inside of it.  This assumes that there
 * are only GObjects in the list, and may spew errors if anything else exists
 * inside of it.
 */
void
lt_free_object_list(GList *list)
{
	if (list == NULL)
		return;

	g_list_foreach(list, (GFunc)g_object_unref, NULL);
	g_list_free(list);
}
