/*
 * @file libleaftag/source.c Source object and functions
 *
 * @Copyright (C) 2005-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libleaftag/cache.h>
#include <libleaftag/db.h>
#include <libleaftag/priv.h>
#include <libleaftag/source.h>
#include <libleaftag/tag.h>
#include <stdlib.h>
#include <string.h>

struct _LtSourcePriv
{
	char *schema;
	char *uri;
	char *filename;
};

enum
{
	PROP_0,
	PROP_URI,
	PROP_SCHEMA
};

enum
{
	TAGS_CHANGED,
	LAST_SIGNAL
};

static void lt_source_dispose(GObject *gobject);
static void lt_source_finalize(GObject *gobject);
static void lt_source_set_property(GObject *gobject, guint prop_id,
								const GValue *value, GParamSpec *pspec);
static void lt_source_get_property(GObject *gobject, guint prop_id,
								GValue *value, GParamSpec *pspec);

static LtObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(LtSource, lt_source, LT_TYPE_OBJECT);

static void
lt_source_class_init(LtSourceClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	gobject_class->dispose      = lt_source_dispose;
	gobject_class->finalize     = lt_source_finalize;
	gobject_class->set_property = lt_source_set_property;
	gobject_class->get_property = lt_source_get_property;

	/**
	 * LtSource::tags-changed
	 * @source: An #LtSource
	 *
	 * The ::tags-changed signal is emitted when the tags associated with
	 * this source are changed.
	 */
	signals[TAGS_CHANGED] =
		g_signal_new("tags-changed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(LtSourceClass, tags_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	g_object_class_install_property(gobject_class, PROP_URI,
		g_param_spec_string("uri", "URI",
							"The URI of the source",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	g_object_class_install_property(gobject_class, PROP_SCHEMA,
		g_param_spec_string("schema", "Schema",
							"The schema of the source's URI",
							NULL,
							G_PARAM_READABLE));
}

static void
lt_source_init(LtSource *source)
{
	source->priv = g_new0(LtSourcePriv, 1);
}

static void
lt_source_dispose(GObject *gobject)
{
	LtSource *source = LT_SOURCE(gobject);

	lt_cache_remove_source(source);

	if (G_OBJECT_CLASS(parent_class)->dispose != NULL)
		G_OBJECT_CLASS(parent_class)->dispose(gobject);
}

static void
lt_source_finalize(GObject *gobject)
{
	LtSource *source = LT_SOURCE(gobject);

	if (source->priv != NULL)
	{
		g_free(source->priv->schema);
		g_free(source->priv->uri);
		g_free(source->priv->filename);

		g_free(source->priv);
		source->priv = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->finalize != NULL)
		G_OBJECT_CLASS(parent_class)->finalize(gobject);
}

static void
lt_source_set_property(GObject *gobject, guint prop_id,
					   const GValue *value, GParamSpec *pspec)
{
	LtSource *source = LT_SOURCE(gobject);

	switch (prop_id)
	{
		case PROP_URI:
			lt_source_set_uri(source, g_value_get_string(value));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

static void
lt_source_get_property(GObject *gobject, guint prop_id, GValue *value,
					   GParamSpec *pspec)
{
	LtSource *source = LT_SOURCE(gobject);

	switch (prop_id)
	{
		case PROP_URI:
			g_value_set_string(value, lt_source_get_uri(source));
			break;

		case PROP_SCHEMA:
			g_value_set_string(value, lt_source_get_schema(source));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

static char *
lt_make_abs_path(const char *path)
{
	char *result, *cwd;

	if (g_path_is_absolute(path))
		return g_strdup(path);

	cwd = g_get_current_dir();
	result = g_build_path(G_DIR_SEPARATOR_S, cwd, path, NULL);
	g_free(cwd);

	return result;
}

void
lt_uri_parse(const char *uri, char **out_schema, char **out_path)
{
	const char *c;
	char *schema = NULL;
	char *buffer;
	char *d;

	if (out_schema != NULL)
		*out_schema = NULL;

	if (out_path != NULL)
		*out_path = NULL;

	buffer = g_new0(char, strlen(uri) + 1);

	for (c = uri, d = buffer; *c != '\0';)
	{
		if (*c == ':' && schema == NULL)
		{
			schema = g_strdup(buffer);
			d = buffer;
			*d = '\0';
			c++;

			if (*c == '/' && *(c + 1) == '/')
				c += 2;
		}
		else
		{
			*d++ = *c++;
		}
	}

	if (out_schema != NULL)
	{
		if (schema == NULL)
			*out_schema = g_strdup("file");
		else
			*out_schema = schema;
	}
	else
		g_free(schema);

	if (out_path != NULL)
		*out_path = g_strdup(buffer);

	g_free(buffer);
}

char *
lt_uri_normalize(const char *uri)
{
	char *schema = NULL;
	char *path = NULL;
	char *result;
	char *c;

	g_return_val_if_fail(uri != NULL && *uri != '\0', NULL);

	lt_uri_parse(uri, &schema, &path);

	if (!strcmp(schema, "file"))
	{
		char *abspath = lt_make_abs_path(path);
		result = g_strdup_printf("file://%s", abspath);
		g_free(abspath);
	}
	else
	{
		result = g_strdup(uri);
	}

	for (c = result + strlen(result) - 1; *c == '/'; c--)
		*c = '\0';

	g_free(schema);
	g_free(path);

	return result;
}

LtSource *
lt_source_new(const char *uri)
{
	LtSource *source;

	g_return_val_if_fail(uri != NULL && *uri != '\0', NULL);

	source = g_object_new(LT_TYPE_SOURCE,
						  "uri", uri,
						  NULL);
	lt_cache_add_source(source);

	return source;
}

/**
 * lt_create_source
 * @uri: The URI of the item.
 *
 * Look up a source referring to a URI.  If there is not already a source for
 * this URI, it will be created.
 *
 * Returns: A new #LtSource
 */
LtSource *
lt_create_source(const char *uri)
{
	LtSource *source;

	g_return_val_if_fail(uri != NULL && *uri != '\0', NULL);

	source = lt_source_lookup(uri);

	if (source == NULL)
	{
		char *norm_uri = lt_uri_normalize(uri);
		source = lt_source_new(norm_uri);
		g_free(norm_uri);
	}

	return source;
}

LtSource *
lt_create_source_from_row(LtDbRow *row, const char *col_prefix)
{
	LtSource *source;
	const char *id, *uri;
	char *col_id, *col_uri;

	g_return_val_if_fail(row != NULL, NULL);

	col_id = lt_db_prepend_col_prefix(col_prefix, "id");
	id = lt_db_row_get(row, col_id);
	g_free(col_id);

	g_return_val_if_fail(id != NULL && *id != '\0', NULL);

	col_uri = lt_db_prepend_col_prefix(col_prefix, "uri");
	uri = lt_db_row_get(row, col_uri);
	g_free(col_uri);

	g_return_val_if_fail(uri != NULL && *uri != '\0', NULL);

	source = lt_cache_get_source(uri);

	if (source == NULL)
		source = lt_source_new(uri);
	else
		g_object_ref(G_OBJECT(source));

	lt_object_set_id(LT_OBJECT(source), atoi(id));
	lt_object_set_in_db(LT_OBJECT(source), TRUE);

	return source;
}

GList *
lt_gather_sources_from_results(LtDbResults *results, const char *col_prefix)
{
	GList *sources = NULL;

	if (results != NULL)
	{
		GList *l;

		for (l = results->rows; l != NULL; l = l->next)
		{
			LtDbRow *row = (LtDbRow *)l->data;
			LtSource *source = lt_create_source_from_row(row, col_prefix);
			g_assert(source != NULL);
			sources = g_list_append(sources, source);
		}

		lt_db_results_destroy(results);
	}

	return sources;
}

/**
 * lt_source_lookup
 * @uri: The URI of the item.
 *
 * Look up a source referring to a URI.
 *
 * Returns: A new #LtSource referring to the URI.  If this item has no existing
 * tags, this will return NULL.
 */
LtSource *
lt_source_lookup(const char *uri)
{
	LtSource *source = NULL;
	GList *sources = NULL;
	char *norm_uri;

	g_return_val_if_fail(uri != NULL && *uri != '\0', NULL);

	norm_uri = lt_uri_normalize(uri);

	if ((source = lt_cache_get_source(norm_uri)) == NULL)
	{
		if ((sources = lt_source_lookup_by_sql("uri=%Q", norm_uri)) != NULL)
		{
			source = LT_SOURCE(sources->data);
		}
	}

	if (source != NULL)
		g_object_ref(G_OBJECT(source));

	if (sources != NULL)
		lt_free_object_list(sources);

	g_free(norm_uri);

	return source;
}

/**
 * lt_source_lookup_many
 * @uris: A list of URIs to look up.
 *
 * Look up LtSource objects for multiple URIs.
 *
 * Returns: A list of #LtSource objects.
 */
GList *
lt_source_lookup_many(GList *uris)
{
	GList *sources = NULL;
	char *clause;

	clause = lt_db_build_sources_where_clause(uris, NULL);
	sources = lt_source_lookup_by_sql(clause);
	g_free(clause);

	return sources;
}

/**
 * lt_source_delete
 * @source: The source to delete.
 *
 * Deletes a source and any associations made to tags.
 */
void
lt_source_delete(LtSource *source)
{
	LtDbResults *results;

	g_return_if_fail(source != NULL);
	g_return_if_fail(LT_IS_SOURCE(source));
	g_return_if_fail(lt_object_get_in_db(LT_OBJECT(source)));

	lt_db_exec("DELETE FROM associations WHERE source_id=%d",
			   lt_object_get_id(LT_OBJECT(source)));
	lt_db_exec("DELETE FROM sources WHERE id=%d",
			   lt_object_get_id(LT_OBJECT(source)));

	results = lt_db_query("SELECT * FROM tags WHERE id NOT IN "
						  "(SELECT tag_id FROM associations)");

	if (results != NULL)
	{
		GList *l;

		for (l = results->rows; l != NULL; l = l->next)
		{
			LtDbRow *row = (LtDbRow *)l->data;
			LtTag *tag = lt_cache_get_tag(
				g_hash_table_lookup(row->cols, "name"));

			if (tag != NULL)
			{
				g_signal_emit_by_name(source, "deleted");
				lt_object_set_in_db(LT_OBJECT(source), FALSE);
			}
		}
	}

	lt_db_exec("DELETE FROM tags WHERE id NOT IN "
			   "(SELECT tag_id FROM associations)");

	g_signal_emit_by_name(source, "deleted");
	lt_object_set_in_db(LT_OBJECT(source), FALSE);
}

GList *
lt_source_lookup_by_sql(const char *where, ...)
{
	LtDbResults *results = NULL;

	if (where == NULL)
	{
		results = lt_db_query("SELECT * FROM sources ORDER BY uri");
	}
	else
	{
		char *str;
		va_list args;

		va_start(args, where);
		str = sqlite_vmprintf(where, args);
		va_end(args);

		results = lt_db_query("SELECT * FROM sources WHERE %s ORDER BY uri",
							  str);
		sqlite_freemem(str);
	}

	return lt_gather_sources_from_results(results, NULL);
}

void
lt_source_ensure_in_db(LtSource *source)
{
	const char *uri;
	char *schema;

	if (lt_object_get_in_db(LT_OBJECT(source)))
		return;

	uri = lt_source_get_uri(source);
	lt_uri_parse(uri, &schema, NULL);
	lt_db_exec("INSERT INTO sources (uri, schema, ctime) "
			   "VALUES(%Q, %Q, DATETIME('NOW'))",
			   uri, schema);
	g_free(schema);

	lt_object_set_id(LT_OBJECT(source), sqlite_last_insert_rowid(lt_get_db()));
	lt_object_set_in_db(LT_OBJECT(source), TRUE);
	lt_cache_add_source(source);
}

/**
 * lt_source_set_uri
 * @source: An #LtSource
 * @new_uri: The URI to set
 *
 * Set the URI for a source
 */
void
lt_source_set_uri(LtSource *source, const char *new_uri)
{
	g_return_if_fail(source != NULL);
	g_return_if_fail(LT_IS_SOURCE(source));
	g_return_if_fail(new_uri != NULL && *new_uri != '\0');

	if (source->priv->uri == new_uri)
		return;

	if (source->priv->uri != NULL)
		g_free(source->priv->uri);

	if (source->priv->filename != NULL)
	{
		g_free(source->priv->filename);
		source->priv->filename = NULL;
	}

	source->priv->uri = g_strdup(new_uri);

	g_free(source->priv->schema);

	lt_uri_parse(source->priv->uri, &source->priv->schema, NULL);

	if (lt_object_get_in_db(LT_OBJECT(source)))
	{
		lt_db_exec("UPDATE sources SET uri=%Q AND schema=%Q WHERE id=%d",
				   new_uri, source->priv->schema,
				   lt_object_get_id(LT_OBJECT(source)));
	}

	g_object_notify(G_OBJECT(source), "uri");
}

/**
 * lt_source_get_uri
 * @source: An #LtSource
 *
 * Get the URI for a source.
 *
 * Returns: The URI of the source.
 */
const char *
lt_source_get_uri(const LtSource *source)
{
	g_return_val_if_fail(source != NULL, NULL);
	g_return_val_if_fail(LT_IS_SOURCE(source), NULL);

	return source->priv->uri;
}

/**
 * lt_source_get_filename
 * @source: An #LtSource
 *
 * Get the filename for a source.  The URI schema of the source must be "file".
 *
 * Returns: The filename of the source.
 */
const char *
lt_source_get_filename(const LtSource *source)
{
	g_return_val_if_fail(source != NULL, NULL);
	g_return_val_if_fail(LT_IS_SOURCE(source), NULL);
	g_return_val_if_fail(!strcmp(lt_source_get_schema(source), "file"), NULL);

	if (source->priv->filename == NULL)
		lt_uri_parse(source->priv->uri, NULL, &source->priv->filename);

	return source->priv->filename;
}

/**
 * lt_source_get_schema
 * @source: An #LtSource
 *
 * Get the schema for a source.
 *
 * Returns: The schema.
 */
const char *
lt_source_get_schema(const LtSource *source)
{
	g_return_val_if_fail(source != NULL, NULL);
	g_return_val_if_fail(LT_IS_SOURCE(source), NULL);

	return source->priv->schema;
}

static void
add_association(LtSource *source, LtTag *tag)
{
	LtDbResults *results;

	results = lt_db_query("SELECT * FROM associations WHERE "
						  "source_id=%d AND tag_id=%d",
						  lt_object_get_id(LT_OBJECT(source)),
						  lt_object_get_id(LT_OBJECT(tag)));

	if (results == NULL)
	{
		lt_db_exec("INSERT INTO associations VALUES(%d, %d)",
				   lt_object_get_id(LT_OBJECT(source)),
				   lt_object_get_id(LT_OBJECT(tag)));
	}
	else
		lt_db_results_destroy(results);
}

static void
remove_association(LtSource *source, LtTag *tag)
{
	LtDbResults *results;
	LtDbRow *row;

	lt_db_exec("DELETE FROM associations WHERE source_id=%d AND tag_id=%d",
			   lt_object_get_id(LT_OBJECT(source)),
			   lt_object_get_id(LT_OBJECT(tag)));

	results = lt_db_query("SELECT COUNT(*) FROM associations WHERE tag_id=%d",
						  lt_object_get_id(LT_OBJECT(tag)));
	row = (LtDbRow *)results->rows->data;

	if (!strcmp(lt_db_row_get(row, "COUNT(*)"), "0"))
		lt_tag_delete(tag);

	if (lt_object_get_in_db(LT_OBJECT(source)))
	{
		results = lt_db_query("SELECT COUNT(*) FROM associations "
							  "WHERE source_id=%d",
							  lt_object_get_id(LT_OBJECT(source)));
		row = (LtDbRow *)results->rows->data;

		if (!strcmp(lt_db_row_get(row, "COUNT(*)"), "0"))
		{
			lt_source_delete(source);
		}
	}
}

/**
 * lt_source_tag
 * @source: An #LtSource
 * @tags: A list of #LtTag objects
 *
 * Tag a source with a set of tags.  This only adds associations, preserving
 * existing tags.
 */
void
lt_source_tag(LtSource *source, GList *tags)
{
	GList *l;

	g_return_if_fail(source != NULL);
	g_return_if_fail(LT_IS_SOURCE(source));
	g_return_if_fail(tags != NULL);

	lt_source_ensure_in_db(source);

	for (l = tags; l != NULL; l = l->next)
	{
		LtTag *tag = LT_TAG(l->data);
		lt_tag_ensure_in_db(tag);
		add_association(source, tag);
	}
}

static void
tag_table_foreach(const char *name, LtTag *tag, LtSource *source)
{
	add_association(source, tag);
}

/**
 * lt_source_tag_with_names
 * @source: An #LtSource
 * @tag_names: A list of tag names
 *
 * Tag a source with a set of tags.  This only adds associations, preserving
 * existing tags.
 */
void
lt_source_tag_with_names(LtSource *source, GList *tag_names)
{
	GHashTable *tag_table;
	GList *existing_tag_objs, *l;
	LtTag *tag;

	g_return_if_fail(source != NULL);
	g_return_if_fail(LT_IS_SOURCE(source));
	g_return_if_fail(tag_names != NULL);
	g_return_if_fail(lt_get_tag_names_valid(tag_names));

	lt_source_ensure_in_db(source);

	existing_tag_objs = lt_tag_lookup_many(tag_names);

	tag_table = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);

	for (l = existing_tag_objs; l != NULL; l = l->next)
	{
		tag = LT_TAG(l->data);
		g_hash_table_insert(tag_table, (gpointer)lt_tag_get_name(tag), tag);
	}

	for (l = tag_names; l != NULL; l = l->next)
	{
		const char *tag_name = (const char *)l->data;

		if (g_hash_table_lookup(tag_table, tag_name) == NULL)
		{
			tag = lt_cache_get_tag(tag_name);

			if (tag == NULL)
				tag = lt_tag_new(tag_name);

			lt_tag_ensure_in_db(tag);

			g_hash_table_insert(tag_table,
								(gpointer)lt_tag_get_name(tag), tag);
		}
	}

	g_hash_table_foreach(tag_table, (GHFunc)tag_table_foreach, source);
	g_hash_table_destroy(tag_table);
	lt_free_object_list(existing_tag_objs);
}

/**
 * lt_source_untag
 * @source: An #LtSource
 * @tags: A list of #LtTag objects
 *
 * Remove a set of associations from a source.
 */
void
lt_source_untag(LtSource *source, GList *tags)
{
	GList *l;

	g_return_if_fail(source != NULL);
	g_return_if_fail(LT_IS_SOURCE(source));
	g_return_if_fail(tags != NULL);
	g_return_if_fail(lt_object_get_in_db(LT_OBJECT(source)));

	for (l = tags; l != NULL; l = l->next)
	{
		LtTag *tag = LT_TAG(l->data);

		/*
		 * TODO: Pull from the DB, make sure there is no entry
		 *       representing this.
		 */
		if (!lt_object_get_in_db(LT_OBJECT(tag)))
			continue;

		remove_association(source, tag);
	}
}

/**
 * lt_source_untag_with_names
 * @source: An #LtSource
 * @tag_names: A list of tag names
 *
 * Remove a set of associations from a source.
 */
void
lt_source_untag_with_names(LtSource *source, GList *tag_names)
{
	GList *tags;

	g_return_if_fail(source != NULL);
	g_return_if_fail(LT_IS_SOURCE(source));
	g_return_if_fail(tag_names != NULL);
	g_return_if_fail(lt_object_get_in_db(LT_OBJECT(source)));
	g_return_if_fail(lt_get_tag_names_valid(tag_names));

	tags = lt_tag_lookup_many(tag_names);
	lt_source_untag(source, tags);
	lt_free_object_list(tags);
}

/**
 * lt_source_get_tags
 * @source: An #LtSource
 *
 * Returns a list of tags on the specified source.
 *
 * Returns: The tag objects associated with the source.
 */
GList *
lt_source_get_tags(const LtSource *source)
{
	LtDbResults *results;

	g_return_val_if_fail(source != NULL, NULL);
	g_return_val_if_fail(LT_IS_SOURCE(source), NULL);

	results = lt_db_query("SELECT tags.* FROM tags, associations WHERE "
						  "associations.tag_id=tags.id AND "
						  "associations.source_id=%d ORDER BY tags.name",
						  lt_object_get_id(LT_OBJECT(source)));

	return lt_gather_tags_from_results(results, "tags");
}

/**
 * lt_get_all_sources
 *
 * Look up all tagged sources.
 *
 * Returns: A list of all tagged sources.
 */
GList *
lt_get_all_sources(void)
{
	return lt_source_lookup_by_sql(NULL);
}

/**
 * lt_get_sources_with_schema
 * @schema: The schema.
 *
 * Look up tagged sources with the specified schema.
 *
 * Returns: The list of tagged sources.
 */
GList *
lt_get_sources_with_schema(const char *schema)
{
	g_return_val_if_fail(schema != NULL && *schema != '\0', NULL);

	return lt_source_lookup_by_sql("schema=%Q", schema);
}

/**
 * lt_get_sources_with_tags
 * @tags: The list of tag objects to match against.
 * @schema: The optional schema to match against, or NULL.
 *
 * Look up all sources tagged with the specified tags.  This optionally takes
 * a schema to match against. It can be NULL.
 *
 * Returns: The list of sources.
 */
GList *
lt_get_sources_with_tags(GList *tags, const char *schema)
{
	LtDbResults *results;
	GString *str;
	GList *l;
	char *sql;

	g_return_val_if_fail(tags != NULL, NULL);

	str = g_string_new("SELECT sources.* FROM sources, associations WHERE "
					   "associations.source_id=sources.id AND ");

	if (schema != NULL)
	{
		char *temp = sqlite_mprintf("sources.schema=%Q AND ", schema);
		g_string_append(str, temp);
		sqlite_freemem(temp);
	}

	for (l = tags; l != NULL; l = l->next)
	{
		LtTag *tag = LT_TAG(l->data);

		if (!lt_object_get_in_db(LT_OBJECT(tag)))
			continue;

		if (l == tags)
			g_string_append(str, "associations.tag_id IN (");
		else
			g_string_append(str, ", ");

		g_string_append_printf(str, "%d", lt_object_get_id(LT_OBJECT(tag)));
	}

	g_string_append(str, ") ORDER BY sources.uri");

	sql = g_string_free(str, FALSE);
	results = lt_db_query(sql);
	g_free(sql);

	return lt_gather_sources_from_results(results, "sources");
}

/**
 * lt_get_sources_with_tag_names
 * @tag_names: The list of tag names to match against.
 * @schema: The optional schema to match against, or NULL.
 *
 * Look up all sources tagged with the specified tag names.  This optionally
 * takes a schema to match against. It can be NULL.
 *
 * Returns: The list of sources.
 */
GList *
lt_get_sources_with_tag_names(GList *tag_names, const char *schema)
{
	LtDbResults *results;
	GString *str;
	GList *l;
	char *sql;

	g_return_val_if_fail(tag_names != NULL, NULL);
	g_return_val_if_fail(lt_get_tag_names_valid(tag_names), NULL);

	str = g_string_new("SELECT sources.* FROM sources, associations, tags "
					   "WHERE associations.source_id=sources.id AND "
					   "associations.tag_id=tags.id AND ");

	if (schema != NULL)
	{
		char *temp = sqlite_mprintf("sources.schema=%Q AND ", schema);
		g_string_append(str, temp);
		sqlite_freemem(temp);
	}

	for (l = tag_names; l != NULL; l = l->next)
	{
		const char *tag_name = (const char *)l->data;
		char *temp;

		if (l == tag_names)
			g_string_append(str, "tags.name IN (");
		else
			g_string_append(str, ", ");

		temp = sqlite_mprintf("%Q", tag_name);
		g_string_append(str, temp);
		sqlite_freemem(temp);
	}

	g_string_append(str, ") ORDER BY sources.uri");

	sql = g_string_free(str, FALSE);
	results = lt_db_query(sql);
	g_free(sql);

	return lt_gather_sources_from_results(results, "sources");
}
