/*
 * @file libleaftag/tag.c Tag object and functions
 *
 * @Copyright (C) 2005-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libleaftag/cache.h>
#include <libleaftag/db.h>
#include <libleaftag/priv.h>
#include <libleaftag/tag.h>
#include <stdlib.h>
#include <string.h>

struct _LtTagPriv
{
	char *name;
	char *description;
	char *image;
	gboolean hidden;
};

enum
{
	PROP_0,
	PROP_NAME,
	PROP_DESCRIPTION,
	PROP_HIDDEN,
	PROP_IMAGE
};

enum
{
	SOURCES_CHANGED,
	LAST_SIGNAL
};

static void lt_tag_dispose(GObject *gobject);
static void lt_tag_finalize(GObject *gobject);
static void lt_tag_set_property(GObject *gobject, guint prop_id,
								const GValue *value, GParamSpec *pspec);
static void lt_tag_get_property(GObject *gobject, guint prop_id,
								GValue *value, GParamSpec *pspec);

static LtObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(LtTag, lt_tag, LT_TYPE_OBJECT);

static void
lt_tag_class_init(LtTagClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	gobject_class->dispose      = lt_tag_dispose;
	gobject_class->finalize     = lt_tag_finalize;
	gobject_class->set_property = lt_tag_set_property;
	gobject_class->get_property = lt_tag_get_property;

	/**
	 * LtTag::sources-changed
	 * @tag: An #LtTag
	 *
	 * The ::sources-changed signal is emitted when sources are either
	 * tagged or untagged with this tag.
	 */
	signals[SOURCES_CHANGED] =
		g_signal_new("sources-changed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(LtTagClass, sources_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	g_object_class_install_property(gobject_class, PROP_NAME,
		g_param_spec_string("name", "Name",
							"The name of the tag",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property(gobject_class, PROP_DESCRIPTION,
		g_param_spec_string("description", "Description",
							"The tag's description",
							NULL,
							G_PARAM_READWRITE));

	g_object_class_install_property(gobject_class, PROP_HIDDEN,
		g_param_spec_boolean("hidden", "Hidden",
							 "The hidden state of the tag",
							 FALSE,
							 G_PARAM_READWRITE));

	g_object_class_install_property(gobject_class, PROP_IMAGE,
		g_param_spec_string("image", "Image",
							"The tag's image",
							NULL,
							G_PARAM_READWRITE));
}

static void
lt_tag_init(LtTag *tag)
{
	tag->priv = g_new0(LtTagPriv, 1);
}

static void
lt_tag_dispose(GObject *gobject)
{
	LtTag *tag = LT_TAG(gobject);

	lt_cache_remove_tag(tag);

	if (G_OBJECT_CLASS(parent_class)->dispose != NULL)
		G_OBJECT_CLASS(parent_class)->dispose(gobject);
}

static void
lt_tag_finalize(GObject *gobject)
{
	LtTag *tag = LT_TAG(gobject);

	if (tag->priv != NULL)
	{
		g_free(tag->priv->name);
		g_free(tag->priv->description);
		g_free(tag->priv->image);

		g_free(tag->priv);
		tag->priv = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->finalize != NULL)
		G_OBJECT_CLASS(parent_class)->finalize(gobject);
}

static void
lt_tag_set_property(GObject *gobject, guint prop_id,
					const GValue *value, GParamSpec *pspec)
{
	LtTag *tag = LT_TAG(gobject);

	switch (prop_id)
	{
		case PROP_NAME:
			tag->priv->name = g_value_dup_string(value);
			break;

		case PROP_DESCRIPTION:
			lt_tag_set_description(tag, g_value_get_string(value));
			break;

		case PROP_HIDDEN:
			lt_tag_set_hidden(tag, g_value_get_boolean(value));
			break;

		case PROP_IMAGE:
			lt_tag_set_image(tag, g_value_get_string(value));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

static void
lt_tag_get_property(GObject *gobject, guint prop_id, GValue *value,
					GParamSpec *pspec)
{
	LtTag *tag = LT_TAG(gobject);

	switch (prop_id)
	{
		case PROP_NAME:
			g_value_set_string(value, lt_tag_get_name(tag));
			break;

		case PROP_DESCRIPTION:
			g_value_set_string(value, lt_tag_get_description(tag));
			break;

		case PROP_HIDDEN:
			g_value_set_boolean(value, lt_tag_get_hidden(tag));
			break;

		case PROP_IMAGE:
			g_value_set_string(value, lt_tag_get_image(tag));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

LtTag *
lt_tag_new(const char *name)
{
	LtTag *tag;

	g_return_val_if_fail(name != NULL && *name != '\0', NULL);

	tag = g_object_new(LT_TYPE_TAG,
					   "name", name,
					   NULL);
	lt_cache_add_tag(tag);

	return tag;
}

/**
 * lt_create_tag
 * @tag_name: The tag to create
 *
 * Creates a new tag.
 *
 * Returns: The new #LtTag object.
 */
LtTag *
lt_create_tag(const char *tag_name)
{
	LtTag *tag;

	g_return_val_if_fail(lt_get_tag_name_valid(tag_name), NULL);

	tag = lt_tag_lookup(tag_name);

	if (tag == NULL)
		tag = lt_tag_new(tag_name);

	return tag;
}

LtTag *
lt_create_tag_from_row(LtDbRow *row, const char *col_prefix)
{
	LtTag *tag;
	const char *id, *name, *prop_value;
	char *col_id, *col_name, *col_prop;

	g_return_val_if_fail(row != NULL, NULL);

	col_id = lt_db_prepend_col_prefix(col_prefix, "id");
	id = lt_db_row_get(row, col_id);
	g_free(col_id);

	g_return_val_if_fail(id != NULL && *id != '\0', NULL);

	col_name = lt_db_prepend_col_prefix(col_prefix, "name");
	name = lt_db_row_get(row, col_name);
	g_free(col_name);

	g_return_val_if_fail(name != NULL && *name != '\0', NULL);

	tag = lt_cache_get_tag(name);

	if (tag == NULL)
		tag = lt_tag_new(name);
	else
		g_object_ref(G_OBJECT(tag));

#define CHECK_AND_SET_PROP(propname) \
	col_prop = lt_db_prepend_col_prefix(col_prefix, (propname)); \
	prop_value = lt_db_row_get(row, col_prop); \
	g_free(col_prop); \
	g_object_set(G_OBJECT(tag), (propname), prop_value, NULL)

	CHECK_AND_SET_PROP("description");
	CHECK_AND_SET_PROP("image");

	col_prop = lt_db_prepend_col_prefix(col_prefix, "hidden");
	prop_value = lt_db_row_get(row, col_prop);
	g_free(col_prop);
	g_object_set(G_OBJECT(tag), "hidden", !strcmp(prop_value, "TRUE"), NULL);

#undef CHECK_AND_SET_PROP

	lt_object_set_id(LT_OBJECT(tag), atoi(id));
	lt_object_set_in_db(LT_OBJECT(tag), TRUE);

	return tag;
}

GList *
lt_gather_tags_from_results(LtDbResults *results, const char *col_prefix)
{
	GList *tags = NULL;

	if (results != NULL)
	{
		GList *l;

		for (l = results->rows; l != NULL; l = l->next)
		{
			LtDbRow *row = (LtDbRow *)l->data;
			LtTag *tag = lt_create_tag_from_row(row, col_prefix);
			g_assert(tag != NULL);
			tags = g_list_append(tags, tag);
		}

		lt_db_results_destroy(results);
	}

	return tags;
}

/**
 * lt_tag_lookup
 * @tag_name: The tag to look up
 *
 * Look up an existing tag.
 *
 * Returns: An #LtTag object
 */
LtTag *
lt_tag_lookup(const char *tag_name)
{
	LtTag *tag = NULL;
	GList *tags = NULL;

	g_return_val_if_fail(lt_get_tag_name_valid(tag_name), NULL);

	if ((tag = lt_cache_get_tag(tag_name)) == NULL)
	{
		if ((tags = lt_tag_lookup_by_sql("name=%Q", tag_name)) != NULL)
		{
			tag = LT_TAG(tags->data);
		}
	}

	if (tag != NULL)
		g_object_ref(G_OBJECT(tag));

	if (tags != NULL)
		lt_free_object_list(tags);

	return tag;
}

/**
 * lt_tag_lookup_many
 * @tag_names: A list of tag names to look up
 *
 * Look up a set of existing tags.
 *
 * Returns: A list of #LtTag objects.
 */
GList *
lt_tag_lookup_many(GList *tag_names)
{
	GList *tags = NULL;
	char *clause;

	g_return_val_if_fail(lt_get_tag_names_valid(tag_names), NULL);

	clause = lt_db_build_tags_where_clause(tag_names, NULL);
	tags = lt_tag_lookup_by_sql(clause);
	g_free(clause);

	return tags;
}

/**
 * lt_tag_delete
 * @tag: An #LtTag.
 *
 * Deletes a tag and any associations made to sources.
 */
void
lt_tag_delete(LtTag *tag)
{
	LtDbResults *results;

	g_return_if_fail(tag != NULL);
	g_return_if_fail(LT_IS_TAG(tag));
	g_return_if_fail(lt_object_get_in_db(LT_OBJECT(tag)));

	lt_db_exec("DELETE FROM associations WHERE tag_id=%d",
			   lt_object_get_id(LT_OBJECT(tag)));
	lt_db_exec("DELETE FROM tags WHERE id=%d",
			   lt_object_get_id(LT_OBJECT(tag)));

	results = lt_db_query("SELECT * FROM sources WHERE id NOT IN "
						  "(SELECT source_id FROM associations)");

	if (results != NULL)
	{
		GList *l;

		for (l = results->rows; l != NULL; l = l->next)
		{
			LtDbRow *row = (LtDbRow *)l->data;
			LtSource *source = lt_cache_get_source(
				g_hash_table_lookup(row->cols, "uri"));

			if (source != NULL)
			{
				g_signal_emit_by_name(source, "deleted");
				lt_object_set_in_db(LT_OBJECT(source), FALSE);
			}
		}
	}

	lt_db_exec("DELETE FROM sources WHERE id NOT IN "
			   "(SELECT source_id FROM associations)");

	g_signal_emit_by_name(tag, "deleted");
	lt_object_set_in_db(LT_OBJECT(tag), FALSE);
}

GList *
lt_tag_lookup_by_sql(const char *where, ...)
{
	LtDbResults *results = NULL;

	if (where == NULL)
	{
		results = lt_db_query("SELECT * FROM tags ORDER BY name");
	}
	else
	{
		char *str;
		va_list args;

		va_start(args, where);
		str = sqlite_vmprintf(where, args);
		va_end(args);

		results = lt_db_query("SELECT * FROM tags WHERE %s ORDER BY name",
							  str);
		sqlite_freemem(str);
	}

	return lt_gather_tags_from_results(results, NULL);
}

void
lt_tag_ensure_in_db(LtTag *tag)
{
	g_return_if_fail(tag != NULL);

	if (lt_object_get_in_db(LT_OBJECT(tag)))
		return;

	lt_db_exec("INSERT INTO tags (name, description, image, hidden, ctime) "
			   "VALUES(%Q, %Q, %Q, %Q, DATETIME('NOW'))",
			   lt_tag_get_name(tag),
			   lt_tag_get_description(tag),
			   lt_tag_get_image(tag),
			   lt_tag_get_hidden(tag) ? "TRUE" : "FALSE");

	lt_object_set_id(LT_OBJECT(tag), sqlite_last_insert_rowid(lt_get_db()));
	lt_object_set_in_db(LT_OBJECT(tag), TRUE);
	lt_cache_add_tag(tag);
}

/**
 * lt_tag_get_name
 * @tag: An #LtTag
 *
 * Retrieve the name of a tag.
 *
 * Returns: the name of the tag.
 */
const char *
lt_tag_get_name(const LtTag *tag)
{
	g_return_val_if_fail(tag != NULL, NULL);
	g_return_val_if_fail(LT_IS_TAG(tag), NULL);

	return tag->priv->name;
}

/**
 * lt_tag_set_hidden
 * @tag: An #LtTag
 * @hidden: Whether the tag should be hidden
 *
 * Set the hidden status of a tag.  Hidden tags will generally not be shown in
 * lists of tags.
 */
void
lt_tag_set_hidden(LtTag *tag, gboolean hidden)
{
	gboolean old_hidden;

	g_return_if_fail(tag != NULL);
	g_return_if_fail(LT_IS_TAG(tag));

	old_hidden = tag->priv->hidden;
	tag->priv->hidden = hidden;

	if (lt_object_get_in_db(LT_OBJECT(tag)))
	{
		lt_db_exec("UPDATE tags SET hidden=%Q WHERE id=%d",
				   (hidden ? "TRUE" : "FALSE"),
				   lt_object_get_id(LT_OBJECT(tag)));
	}

	if (hidden != old_hidden)
		g_object_notify(G_OBJECT(tag), "hidden");
}

/**
 * lt_tag_get_hidden
 * @tag: An #LtTag
 *
 * Look up whether a tag is marked as hidden.
 *
 * Returns: TRUE if the tag is hidden.
 */
gboolean
lt_tag_get_hidden(const LtTag *tag)
{
	g_return_val_if_fail(tag != NULL, FALSE);
	g_return_val_if_fail(LT_IS_TAG(tag), FALSE);

	return tag->priv->hidden;
}

/**
 * lt_tag_set_description
 * @tag: An #LtTag
 * @description: The new description
 *
 * Set the description for a tag.
 */
void
lt_tag_set_description(LtTag *tag, const char *description)
{
	g_return_if_fail(tag != NULL);
	g_return_if_fail(LT_IS_TAG(tag));

	if (tag->priv->description != NULL)
		g_free(tag->priv->description);

	tag->priv->description =
		(description == NULL ? NULL : g_strdup(description));

	if (lt_object_get_in_db(LT_OBJECT(tag)))
	{
		lt_db_exec("UPDATE tags SET description=%Q WHERE id=%d",
				   description, lt_object_get_id(LT_OBJECT(tag)));
	}

	g_object_notify(G_OBJECT(tag), "description");
}

/**
 * lt_tag_get_description
 * @tag: An #LtTag
 *
 * Look up the description for a tag.
 *
 * Returns: The description as previously set.
 */
const char *
lt_tag_get_description(const LtTag *tag)
{
	g_return_val_if_fail(tag != NULL, NULL);
	g_return_val_if_fail(LT_IS_TAG(tag), NULL);

	return tag->priv->description;
}

/**
 * lt_tag_set_image
 * @tag: An #LtTag
 * @image_path: A string describing an image.  This can be either a filename or
 *         a stock ID.
 *
 * Sets the image for a tag.
 */
void
lt_tag_set_image(LtTag *tag, const char *image)
{
	g_return_if_fail(tag != NULL);
	g_return_if_fail(LT_IS_TAG(tag));

	if (tag->priv->image != NULL)
		g_free(tag->priv->image);

	tag->priv->image =
		(image == NULL ? NULL : g_strdup(image));

	if (lt_object_get_in_db(LT_OBJECT(tag)))
	{
		lt_db_exec("UPDATE tags SET image=%Q WHERE id=%d",
				   image, lt_object_get_id(LT_OBJECT(tag)));
	}

	g_object_notify(G_OBJECT(tag), "image");
}

/**
 * lt_tag_get_image
 * @tag: An #LtTag
 *
 * Look up the image for a tag.
 *
 * Returns: A string describing an image.  This can be either a filename or a
 * stock ID.
 */
const char *
lt_tag_get_image(const LtTag *tag)
{
	g_return_val_if_fail(tag != NULL, NULL);
	g_return_val_if_fail(LT_IS_TAG(tag), NULL);

	return tag->priv->image;
}

/**
 * lt_tag_get_sources:
 * @tag: An #LtTag
 * @schema: A schema to filter by.  If this is NULL, sources for this tag in
 *          all schemas will be returned.
 *
 * Look up sources for a tag.
 *
 * Returns: A list of LtSource objects.
 */
GList *
lt_tag_get_sources(const LtTag *tag, const char *schema)
{
	LtDbResults *results;
	char *schema_clause = NULL;

	g_return_val_if_fail(tag != NULL, NULL);
	g_return_val_if_fail(LT_IS_TAG(tag), NULL);

	if (schema != NULL)
		schema_clause = sqlite_mprintf("AND sources.schema=%Q", schema);

	results = lt_db_query("SELECT sources.* FROM sources, associations "
						  "WHERE associations.source_id=sources.id AND "
						  "associations.tag_id=%d %s ORDER BY sources.uri",
						  lt_object_get_id(LT_OBJECT(tag)),
						  schema_clause == NULL ? "" : schema_clause);

	if (schema_clause != NULL)
		sqlite_freemem(schema_clause);

	return lt_gather_sources_from_results(results, "sources");
}

/**
 * lt_get_all_tags
 *
 * Retrieve all tags on the system.
 *
 * Returns: A list of #LtTag objects.
 */
GList *
lt_get_all_tags(void)
{
	return lt_tag_lookup_by_sql(NULL);
}

/**
 * lt_get_visible_tags
 *
 * Retrieve all tags which are not marked as hidden.
 *
 * Returns: A list of #LtTag objects.
 */
GList *
lt_get_visible_tags(void)
{
	return lt_tag_lookup_by_sql("hidden != 'TRUE'");
}

/**
 * lt_get_tag_name_valid
 * @tag_name: A tag name
 *
 * Check a tag name to see if it is valid.  Valid names do not contain any of
 * the following set of characters: \t\n\r()[]<>+,'"
 *
 * Returns: TRUE if the tag name is valid.
 */
gboolean
lt_get_tag_name_valid(const char *tag)
{
	if (tag == NULL || *tag == '\0')
		return FALSE;

	return strcspn(tag, "\t\n\r()[]<>+,'\"") == strlen(tag);
}

/**
 * lt_get_tag_names_valid
 * @tag_names: A list of tag names
 *
 * Check a set of tag names to see if they are valid.  Valid names do not
 * contain any of the following set of characters: \t\n\r()[]<>+,'"
 *
 * Returns: TRUE if the all of the tag names are valid.
 */
gboolean
lt_get_tag_names_valid(GList *tag_names)
{
	GList *l;

	if (tag_names == NULL)
		return FALSE;

	for (l = tag_names; l != NULL; l = l->next)
	{
		if (!lt_get_tag_name_valid((char *)l->data))
			return FALSE;
	}

	return TRUE;
}
