/*
 * x86 spinlock operations
 * 
 * Some codes from the PM2 (Parallel Multithreaded Machine) software suite.
 * 
 * $Id: spinlock.h,v 1.2 2004/01/18 19:41:44 mbuna Exp $
 */
#ifndef INCLUDED_PEAK_SPINLOCK_H_
#define INCLUDED_PEAK_SPINLOCK_H_

#define PEAK_SPINLOCK_ITERATIONS 1000

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <pthread.h>
#include <unistd.h>
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/* PEAK INTERNAL SPIN LOCK (x86 processors)
 * 
 * _peak_spinlock_lock(lockp)
 * _peak_spinlock_lock_try(lockp) - returns 0 (busy) or 1 (got the lock)
 * _peak_spinlock_unlock(lockp)
 * 
 */

extern int _peak_is_threaded;

typedef volatile unsigned peak_spinlock_t;
#define PEAK_SPINLOCK_INITIALIZER (0)

#define LOCK_PREFIX "lock; "

/* set *spinlock to 1 and returns previous value */
static inline unsigned
__peak_spinlock_testandset(peak_spinlock_t *lockp)
  {
  unsigned ret;
  
  __asm__ __volatile__(
    LOCK_PREFIX "xchgl %0, %1"
    : "=q"(ret), "=m"(*lockp)
    : "0"(1), "m"(*lockp)
    : "memory");
  
  return ret;
  }
  
static inline void
_peak_spinlock_lock(peak_spinlock_t *lockp)
  {
  unsigned int tries = PEAK_SPINLOCK_ITERATIONS;
  
  if (!_peak_is_threaded) /* set only if peak uses several threads */
    return;
  
  while (__peak_spinlock_testandset(lockp))
    {
    if (--tries > 0)
      {
      sched_yield();
      tries = PEAK_SPINLOCK_ITERATIONS;
      }
    }
  }

static inline int
_peak_spinlock_lock_try(peak_spinlock_t *lockp)
  {
  if (!_peak_is_threaded) /* set only if peak uses several threads */
    return 1; /* always succeed */
  
  return !__peak_spinlock_testandset(lockp);
  }

static inline void
_peak_spinlock_unlock(peak_spinlock_t *lockp)
  {
  if (!_peak_is_threaded) /* set only if peak uses several threads */
    return;
  
  *lockp = 0;
  }

#if defined(__cplusplus)
}
#endif

#endif /* INCLUDED_PEAK_SPINLOCK_H_ */
