/* PEAK Library
 *
 * Copyright (c) 2003
 *      Stephane Thiell <mbuna@bugged.org>. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * $Id: engine_client.h,v 1.2 2005/01/27 16:31:50 mbuna Exp $
 */
#ifndef INCLUDED_PEAK_ENGINE_CLIENT_H_
#define INCLUDED_PEAK_ENGINE_CLIENT_H_

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "engine.h"
#include "spinlock.h"
#include "runtime.h"
#include <peak/task.h>

typedef void (*peak_engine_client_event_process_func)(peak_engine_client c,
                                                      int event, int info);

/* Engine IO client object structure */
struct __peak_engine_client
  {
  PEAK_STRUCT_RT_HEADER;
  peak_task _task;                           /* associated task */
  peak_engine _engine;                       /* associated engine */
  peak_spinlock_t _lock;
  int _ident;                                /* identifier: file descriptor,
                                              * signum, etc.
                                              */
  /* Optional engine dependent field.
   */
#if defined(PEAK_ENGINE_POLL)
  int _index;
#elif defined(PEAK_ENGINE_EPOLL)
  uint32_t _events;
#endif
  
  uint16_t _state;                           /* object state */
  uint16_t _sstate;                          /* saved object state (sorf of
                                              * cache)
                                              */
  peak_engine_client_event_process_func _ep; /* call out function ptr */
  };

/* Bah. :)
 */
#define PEAK_STRUCT_ENGINE_CLIENT_HEADER struct __peak_engine_client _c


#if defined(__cplusplus)
extern "C" {
#endif

/* Base class engine_client
 */
PEAK_CLASS_BASE_DECLARE_VIRTUAL(engine_client);

__private_extern__ void _peak_engine_client_configure
(
    peak_engine_client c,
    int fd,
    uint32_t state_init,
    peak_engine_client_event_process_func ep
);


/* Lock a client to control data structure access.
 */
static inline void
_peak_engine_client_lock(peak_engine_client c)
  {
  _peak_spinlock_lock(&c->_lock);
  }

/* Unlock a client.
 */
static inline void
_peak_engine_client_unlock(peak_engine_client c)
  {
  _peak_spinlock_unlock(&c->_lock);
  }

/* Accept was performed.
 */
static inline void
_peak_engine_client_did_accept(peak_engine_client c)
  {
  _peak_engine_client_lock(c);
  
  c->_state |= CS_ACCEPTING;
  
  /* When a client's event is currently being handled (probably by ourself,
   * but it might be another task's thread too), we do nothing as changes 
   * will automagically be commited. Otherwise, we need to edit the client
   * explicitely.
   * Summary:
   *   1. We are processing an event generated by this client... ok, we
   *      don't edit it immediately, as all changes will be commited in
   *      peak_engine_event_postprocess().
   *   2. Events of two clients are being processed simultaneously in two
   *      task's threads. Thread of first client do an operation (accept,
   *      read or write) on second client. Probably rare, but we don't edit
   *      it immediately neither.
   *   3. We are processing an event generated by this client, and are doing
   *      an operation on another "inactive" client (he didn't generate any
   *      event simultaneously)... oh, he needs to know! edit it now. Note that
   *      the other client could have been triggered at the same time for
   *      parallel events but have already finished his processing, then
   *      it won't commit things anymore and we need to edit it explicitely.
   * 
   * Sure we could ignore this and edit the client every time, however
   * some event engines like kqueue use a syscall for client's edition...
   */
  if (c->_engine && !(c->_state & CS_HANDLED))
    _peak_engine_edit_client(c->_engine, c);
  
  _peak_engine_client_unlock(c);
  }

/* Read was performed.
 */
static inline void
_peak_engine_client_did_read(peak_engine_client c)
  {
  _peak_engine_client_lock(c);
  
  c->_state |= CS_READING;
  
  if (c->_engine && !(c->_state & CS_HANDLED))
    _peak_engine_edit_client(c->_engine, c);
  
  _peak_engine_client_unlock(c);
  }

/* Write was performed.
 */
static inline void
_peak_engine_client_did_write(peak_engine_client c)
  {
  _peak_engine_client_lock(c);
  
  c->_state |= CS_WRITING;
  
  if (c->_engine && !(c->_state & CS_HANDLED))
    _peak_engine_edit_client(c->_engine, c);
  
  _peak_engine_client_unlock(c);
  }

#if defined(__cplusplus)
}
#endif

#endif /* INCLUDED_PEAK_ENGINE_CLIENT_H_ */
