/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <glib/gi18n.h>
#include <tomoe.h>

#include "tomoe-details.h"
#include "tomoe-canvas.h"
#include "tomoe-edit-char.h"
#include "tomoe-edit-strokes.h"
#include "tomoe-edit-meta.h"

enum {
    PROPERTY_EDITABLE_COLUMN,
    PROPERTY_NAME_COLUMN,
    PROPERTY_VALUE_COLUMN,
    COLUMN_COUNT
};

enum {
    CHARACTER_PROPERTY,
    STROKE_PROPERTY,
    READINGS_PROPERTY,
    N_PROPERTIES 
};

enum
{
    PROP_0,
    PROP_TOMOE_CHAR,
    PROP_TOMOE_DICT
};


typedef struct _TomoeDetailsPrivate	TomoeDetailsPrivate;
struct _TomoeDetailsPrivate
{
    GtkListStore *basic_prop;
    GtkListStore *sub_prop;
    GtkWidget    *meta_view;
    GtkWidget    *canvas;
    GtkWidget    *edit_char_button;
    GtkWidget    *edit_strokes_button;
    GtkWidget    *edit_meta_button;
    TomoeChar    *character;
    TomoeDict    *dict;
};

#define TOMOE_DETAILS_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_DETAILS, TomoeDetailsPrivate))

G_DEFINE_TYPE (TomoeDetails, tomoe_details, GTK_TYPE_DIALOG)

static void     dispose      (GObject              *object);
static GObject *constructor  (GType                  type,
                              guint                  n_props,
                              GObjectConstructParam *props);
static void     set_property (GObject               *object,
                              guint                  prop_id,
                              const GValue          *value,
                              GParamSpec            *pspec);
static void     get_property (GObject               *object,
                              guint                  prop_id,
                              GValue                *value,
                              GParamSpec            *pspec);

static void _show_details                  (TomoeDetails         *dialog);

static void on_close_button_clicked        (GtkButton            *button,
                                            gpointer              user_data);
static void on_edit_char_button_clicked    (GtkButton            *button,
                                            gpointer              user_data);
static void on_edit_strokes_button_clicked (GtkButton            *button,
                                            gpointer              user_data);
static void on_edit_meta_button_clicked    (GtkButton            *button,
                                            gpointer              user_data);
static void on_property_value_edited       (GtkCellRendererText  *renderer,
                                            const gchar          *path_str,
                                            const gchar          *new_text,
                                            gpointer              user_data);


static void
tomoe_details_class_init (TomoeDetailsClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose      = dispose;
    gobject_class->constructor  = constructor;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;
    g_object_class_install_property (gobject_class,
                                     PROP_TOMOE_CHAR,
                                     g_param_spec_object ("tomoe-char",
                                         N_("TomoeChar object"),
                                         N_("The Object of TomoeChar"),
                                         TOMOE_TYPE_CHAR,
                                         G_PARAM_READWRITE |
                                         G_PARAM_CONSTRUCT_ONLY));
    g_object_class_install_property (gobject_class,
                                     PROP_TOMOE_DICT,
                                     g_param_spec_object ("tomoe-dict",
                                         N_("TomoeDict object"),
                                         N_("The Object of TomoeDict"),
                                         TOMOE_TYPE_DICT,
                                         G_PARAM_READWRITE |
                                         G_PARAM_CONSTRUCT_ONLY));
    g_type_class_add_private (gobject_class, sizeof (TomoeDetailsPrivate));
}

static GObject *
constructor (GType type, guint n_props,
             GObjectConstructParam *props)
{

    GObject *object;
    GObjectClass *klass = G_OBJECT_CLASS (tomoe_details_parent_class);
    GtkWidget *hbox, *main_vbox, *vbox, *alignment, *button;
    GtkWidget *label, *canvas, *meta, *frame, *tree_view;
    GtkListStore *list_store;
    GtkTreeViewColumn *column;
    GtkTreeIter iter;
    TomoeDetails *window;
    GtkCellRenderer *renderer;

    object = klass->constructor (type, n_props, props);

    window = TOMOE_DETAILS (object);
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (object);

    gtk_window_set_title (GTK_WINDOW (window),
                          _("Character Details"));

    gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (window), 5);
    gtk_dialog_set_default_response (GTK_DIALOG (window), GTK_RESPONSE_DELETE_EVENT);
    gtk_dialog_set_has_separator (GTK_DIALOG (window), FALSE);
    /*gtk_dialog_add_button (GTK_DIALOG (window), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);*/

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (window)->vbox), hbox);
    gtk_widget_set_size_request (hbox, 400, 320);
    gtk_widget_show (hbox);

    main_vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), main_vbox,
                        TRUE, TRUE, 0);
    gtk_widget_show (main_vbox);

    /* button area */
    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 8, 0);
    gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (alignment), vbox);
    gtk_widget_show (vbox);

    button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_close_button_clicked),
                      (gpointer) window);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (_("Edit Character"));
    priv->edit_char_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_edit_char_button_clicked),
                      (gpointer) window);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (_("Edit Strokes"));
    priv->edit_strokes_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_edit_strokes_button_clicked),
                      (gpointer) window);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (_("Edit Meta"));
    priv->edit_meta_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_edit_meta_button_clicked),
                      (gpointer) window);
    gtk_widget_show (button);

    /* character / strokes area */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show (hbox);
    frame = gtk_frame_new (NULL);
    gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 4);
    gtk_widget_show (frame);

    canvas = tomoe_canvas_new ();
    priv->canvas = canvas;
    tomoe_canvas_lock (TOMOE_CANVAS (canvas), TRUE);
    gtk_container_add (GTK_CONTAINER (frame), canvas);
    gtk_widget_set_size_request (canvas, 100, 100);
    /*
    g_signal_connect (G_OBJECT (canvas), "stroke-added",
                      G_CALLBACK (on_canvas_stroke_added), (gpointer) page);
    */
    gtk_widget_show (canvas);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
    gtk_widget_show (vbox);

    label = gtk_label_new (_("Basic information"));
    gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 2);
    gtk_widget_show (label);

    /* basic informations */
    list_store = gtk_list_store_new (COLUMN_COUNT, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING);
    priv->basic_prop = list_store;
    tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list_store));
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tree_view), FALSE);
    /* gtk_tree_view_set_grid_lines (GTK_TREE_VIEW (tree_view), GTK_TREE_VIEW_GRID_LINES_VERTICAL); */
    gtk_box_pack_end (GTK_BOX (vbox), tree_view, FALSE, FALSE, 0);
    gtk_widget_show (tree_view);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Property",
                                                       renderer,
                                                       "text", PROPERTY_NAME_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

    renderer = gtk_cell_renderer_text_new ();
    g_signal_connect (G_OBJECT (renderer), "edited",
                      G_CALLBACK (on_property_value_edited), window);
    column = gtk_tree_view_column_new_with_attributes ("Value",
                                                       renderer,
                                                       "editable", PROPERTY_EDITABLE_COLUMN,
                                                       "text", PROPERTY_VALUE_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

    /* set initial values to list store */
    gtk_list_store_append (list_store, &iter);
    gtk_list_store_set (list_store, &iter,
                        PROPERTY_EDITABLE_COLUMN, TRUE,
                        PROPERTY_NAME_COLUMN, _("Character"),
                        PROPERTY_VALUE_COLUMN, NULL,
                        -1);
    gtk_list_store_append (list_store, &iter);
    gtk_list_store_set (list_store, &iter,
                        PROPERTY_EDITABLE_COLUMN, FALSE,
                        PROPERTY_NAME_COLUMN, _("Strokes"),
                        PROPERTY_VALUE_COLUMN, _("No stroke information"),
                        -1);
    gtk_list_store_append (list_store, &iter);
    gtk_list_store_set (list_store, &iter,
                        PROPERTY_EDITABLE_COLUMN, TRUE,
                        PROPERTY_NAME_COLUMN, _("Readings"),
                        PROPERTY_VALUE_COLUMN, NULL,
                        -1);
    g_object_unref (list_store);

    /* meta data area */
    meta = gtk_text_view_new ();
    priv->meta_view = meta;
    gtk_text_view_set_editable (GTK_TEXT_VIEW (meta), FALSE);
    gtk_box_pack_start (GTK_BOX (main_vbox), meta, TRUE, TRUE, 4);
    gtk_widget_show (meta);
    
#if 0
    list_store = gtk_list_store_new (COLUMN_COUNT, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING);
    window->sub_prop = list_store;
    tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list_store));
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tree_view), FALSE);
    /* gtk_tree_view_set_grid_lines (GTK_TREE_VIEW (tree_view), GTK_TREE_VIEW_GRID_LINES_VERTICAL); */
    gtk_box_pack_start (GTK_BOX (main_vbox), tree_view, FALSE, FALSE, 4);
    gtk_widget_show (tree_view);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Property",
                                                       renderer,
                                                       "text", PROPERTY_NAME_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

    renderer = gtk_cell_renderer_text_new ();
    g_signal_connect (G_OBJECT (renderer), "edited",
                      G_CALLBACK (on_property_value_edited), window);
    column = gtk_tree_view_column_new_with_attributes ("Value",
                                                       renderer,
                                                       "editable", PROPERTY_EDITABLE_COLUMN,
                                                       "text", PROPERTY_VALUE_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
#endif

    _show_details (window);

    return object;
}

static void
tomoe_details_init (TomoeDetails *window)
{
}

static void
dispose (GObject *object)
{
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (object);

    if (priv->character)
        g_object_unref (priv->character);
    if (priv->dict)
        g_object_unref (priv->dict);

    priv->character = NULL;
    priv->dict      = NULL;

    if (G_OBJECT_CLASS(tomoe_details_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_details_parent_class)->dispose(object);
}

static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (object);

    switch (prop_id) {
        case PROP_TOMOE_CHAR:
            priv->character = g_object_ref (g_value_get_object (value));
            break;
        case PROP_TOMOE_DICT:
            priv->dict = g_object_ref (g_value_get_object (value));
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
get_property (GObject    *object,
              guint       prop_id,
              GValue     *value,
              GParamSpec *pspec)
{
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (object);

    switch (prop_id) {
        case PROP_TOMOE_CHAR:
            g_value_set_object (value, priv->character);
            break;
        case PROP_TOMOE_DICT:
            g_value_set_object (value, priv->dict);
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

GtkWidget *
tomoe_details_new (TomoeChar *chr, TomoeDict *dict)
{
    GtkWidget *w = GTK_WIDGET(g_object_new (TOMOE_TYPE_DETAILS,
                                            "tomoe-char", chr,
                                            "tomoe-dict", dict,
                                            NULL));
    return w;
}

static void
each_meta_data_func (gpointer key, gpointer value, gpointer user_data)
{
    /* not implemented yet */
}

static void
_show_details (TomoeDetails *dialog)
{
    GtkTextBuffer *buf = gtk_text_buffer_new (NULL);
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (dialog);
    TomoeChar *c = priv->character;
    TomoeWriting *writing;
    const GList *readings = tomoe_char_get_readings (c);
    char *chr;
    GtkTreeIter iter;

    gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (priv->basic_prop),
                                   &iter, NULL, CHARACTER_PROPERTY);
    gtk_list_store_set (priv->basic_prop, &iter,
                        PROPERTY_VALUE_COLUMN, tomoe_char_get_utf8 (c),
                        -1);
    writing = tomoe_char_get_writing (c);
    if (writing) {
        tomoe_canvas_set_writing (TOMOE_CANVAS (priv->canvas), 
                                  writing);
        chr = g_markup_printf_escaped 
            ("%d", tomoe_writing_get_n_strokes (tomoe_char_get_writing (c)));
        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (priv->basic_prop),
                                       &iter, NULL, STROKE_PROPERTY);
        gtk_list_store_set (priv->basic_prop, &iter,
                            PROPERTY_VALUE_COLUMN, chr,
                            -1);
        g_free (chr);
    }

    if (readings) {
        int j;
        const GList *list;
        gint reading_num = g_list_length ((GList *) readings);
        gchar **str_array = g_new0 (gchar *, reading_num + 1);
        gchar *readings_text;
        str_array[reading_num] = NULL;
        for (list = readings, j = 0; list; list = g_list_next (list), j++) {
	    TomoeReading *reading = TOMOE_READING (list->data);
            str_array[j]  = (gchar *) tomoe_reading_get_reading (reading);
        }
        readings_text = g_strjoinv (" ", str_array);
        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (priv->basic_prop),
                                       &iter, NULL, READINGS_PROPERTY);
        gtk_list_store_set (priv->basic_prop, &iter,
                            PROPERTY_VALUE_COLUMN, readings_text,
                            -1);
        g_free (str_array);
        g_free (readings_text);
    }

    gtk_text_view_set_buffer (GTK_TEXT_VIEW (priv->meta_view), buf);

    /* set meta data */
    tomoe_char_meta_data_foreach (c, (GHFunc) each_meta_data_func, priv);
}

static void
on_close_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog *dlg = GTK_DIALOG (user_data);

    g_return_if_fail (GTK_IS_DIALOG (dlg));

    gtk_dialog_response (dlg, GTK_RESPONSE_CLOSE);
}

static void
on_edit_char_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeDetails *dialog = TOMOE_DETAILS (user_data);
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (dialog);
    GtkWidget    *wnd = tomoe_edit_char_new (priv->character, NULL); /*FIXME*/
    gint result;

    result = gtk_dialog_run (GTK_DIALOG (wnd));
    gtk_widget_destroy (wnd);
    switch (result) {
        case GTK_RESPONSE_APPLY:
            _show_details (dialog);
	    break;
        case GTK_RESPONSE_CANCEL:
	default:
	    break;
    }
}

static void
on_edit_strokes_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeDetails *dialog = TOMOE_DETAILS (user_data);
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (dialog);
    GtkWidget    *wnd    = tomoe_edit_strokes_new (priv->character);
    gint result;

    result = gtk_dialog_run (GTK_DIALOG (wnd));
    gtk_widget_destroy (wnd);
    if (result)
        _show_details (dialog);
}

static void
on_edit_meta_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeDetails *dialog = TOMOE_DETAILS (user_data);
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (dialog);
    GtkWidget    *wnd = tomoe_edit_meta_new (priv->character);
    gint result;

    result = gtk_dialog_run (GTK_DIALOG (wnd));
    gtk_widget_destroy (wnd);
    if (result)
        _show_details (dialog);
}
static void
on_property_value_edited (GtkCellRendererText  *renderer,
                          const gchar          *path_str,
                          const gchar          *new_text,
                          gpointer              user_data)
{
    GtkTreePath *path;
    GtkTreeIter iter;
    gint index;
    gchar **readings = NULL;
    TomoeDetails *dialog = TOMOE_DETAILS (user_data);
    TomoeDetailsPrivate *priv = TOMOE_DETAILS_GET_PRIVATE (dialog);

    path = gtk_tree_path_new_from_string (path_str);
    index = gtk_tree_path_get_indices (path)[0];
    gtk_tree_path_free (path);

    gtk_tree_model_get_iter_from_string (GTK_TREE_MODEL (priv->basic_prop), &iter, path_str);

    switch (index) {
        case CHARACTER_PROPERTY:
            tomoe_char_set_utf8 (priv->character, new_text);
            gtk_list_store_set (priv->basic_prop, &iter,
                                PROPERTY_VALUE_COLUMN, new_text,
                                -1);
            break;
        case STROKE_PROPERTY: /* stroke number should not modify directly*/
            break;
        case READINGS_PROPERTY:
            readings = g_strsplit (new_text, " ", -1);
            if (readings) {
                guint i;
                GPtrArray *reading_array = NULL;
                reading_array = g_ptr_array_new ();
                for (i = 0; i < g_strv_length (readings); i++) {
                    g_ptr_array_add (reading_array, g_strdup (readings[i]));
                }
                g_strfreev (readings);
                /* FIXME ! */
                /* tomoe_char_set_readings (priv->character, reading_array);*/
                g_ptr_array_free (reading_array, TRUE);
            } else {
                /* tomoe_char_set_readings (priv->character, NULL); */
            }
            gtk_list_store_set (priv->basic_prop, &iter,
                                PROPERTY_VALUE_COLUMN, new_text,
                                -1);
            break;
        default:
            break;
    }
}

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
