/****************************************************************************
**
** $Id: directory.cpp,v 1.227 2004/05/04 13:18:56 hemer Exp $
**
** Copyright (C) 1999-2004 The LinCVS development team.
**    Tilo Riemer <riemer@lincvs.org>
**    Falk Brettschneider <gigafalk@yahoo.com>
**    Frank Hemer <frank@hemer.org>
**    Wim Delvaux <wim.delvaux@chello.be>
**    Jose Hernandez <joseh@tesco.net>
**    Helmut Koll <HelmutKoll@web.de>
**    Tom Mishima <tmishima@mail.at-m.or.jp>
**    Joerg Preiss <auba@auba.de>
**    Sven Trogisch <trogisch@iapp.de>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** LinCVS is available under two different licenses:
**
** If LinCVS is linked against the GPLed version of Qt 
** LinCVS is released under the terms of GPL also.
**
** If LinCVS is linked against a nonGPLed version of Qt 
** LinCVS is released under the terms of the 
** LinCVS License for non-Unix platforms (LLNU)
**
**
** LinCVS License for non-Unix platforms (LLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than LinCVS.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** LinCVS License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation; either version 2 of the License, or 
** (at your  option) any later version.  This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/


#include "config.h"

//only needed for cout...
#include <iostream>

//----------------------------------------------------------------------------

#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qinputdialog.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qpixmap.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <qpalette.h> 
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qvaluelist.h>
#include <qdialog.h>
#include <qpushbutton.h>
#include <qdict.h>
#include <qregexp.h>
#include <qtextcodec.h>
#include <qsignal.h>

#ifndef Q_WS_WIN
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#endif


//----------------------------------------------------------------------------

#include "ccvscommand.h"
#include "cvslistview.h"
#include "noncvslistview.h"
#include "cvsignorelistview.h"
#include "directory.h"
#include "globals.h"
#include "pixmapcache.h"
#include "login.h"
#include "CCvsOutput.h"
#include "LinCVSLog.h"
#include "TextDecoder.h"

#ifdef QT_NEED_PWDIALOGPATCH
#include "PasswordDialogImpl.h"
#endif

#define VIRTUAL_IN_FILEVIEW

#define DIRECTORY 123456

//----------------------------------------------------------------------------

// helps creating fast hash-tables
static const unsigned int nextPrime( unsigned int i) {
  switch( i) {
     case 0: return 19;
     case 19: return 41;
     case 41: return 79;
     case 79: return 157;
     case 157: return 311;
     case 311: return 619;
     case 619: return 1249;
     case 1249: return 2503;
     case 2503: return 4999;
     case 4999: return 9973;
     default: return i*2+1;
  }
}

//----------------------------------------------------------------------------

void Entry::setState( EntryStates newState ) {
//   printf("\nfile: %s, state: %i, newstate: %i\n",m_fileName.ascii(),m_state,newState);
  if (m_state==newState) return;
  m_state = newState;

  if (newState>myDirectory->get_ES_Max()) myDirectory->set_ES_Max(newState);
  else if (newState<myDirectory->get_ES_Max()) myDirectory->calc_ES_Max();

  // bubble up
  myDirectory->setState( newState );
}

//----------------------------------------------------------------------------

const bool Directory::VIRTUAL = true;

//----------------------------------------------------------------------------

//top level dir
Directory::Directory( QListView * parent,
		      QStringList subProjectList,
		      const QString& fileName,
		      CvsListView * CvsFileListView,
		      NonCvsListView * NonCvsFileListView,
		      CvsIgnoreListView * CvsIgnoreFileListView )
  : QListViewItem( parent ),
    m_subProjectList(subProjectList)
{
   m_topDir = this;
   m_isSubProjectRoot = true;
   m_topView = parent;
   m_virtual = false;
   m_pDirectorySuicide = new QSignal();
   parent->setUpdatesEnabled(FALSE);
   init(fileName, CvsFileListView, NonCvsFileListView, CvsIgnoreFileListView );
   parent->setUpdatesEnabled(TRUE);
   parent->triggerUpdate();
}

//----------------------------------------------------------------------------

//subdir
Directory::Directory( QListViewItem * parent, 
		      const QString& fileName, 
		      CvsListView * CvsFileListView,
		      NonCvsListView * NonCvsFileListView,
		      CvsIgnoreListView * CvsIgnoreFileListView,
		      bool virtualDir)
  : QListViewItem( parent ),
    m_subProjectList( NULL),
    m_pDirectorySuicide( NULL)
{
   m_topDir = ((Directory*)parent)->topDir();
   m_isSubProjectRoot = false;
   m_topView = m_topDir->m_topView;
   m_virtual = virtualDir;
   bool update;
   if (update = m_topView->isUpdatesEnabled()) {
     m_topView->setUpdatesEnabled(FALSE);
   }
   init(fileName, CvsFileListView, NonCvsFileListView, CvsIgnoreFileListView );
   if (update) {
     m_topView->setUpdatesEnabled(TRUE);
     m_topView->triggerUpdate();
   }
}

//----------------------------------------------------------------------------

Directory::~Directory() {
  releaseMonitoredDir(this);
}

//----------------------------------------------------------------------------

int Directory::compare ( QListViewItem * item, int row, bool ascending) const {
  int ret = QString::localeAwareCompare(text(row).lower(),item->text(row).lower());
  if (ret != 0) return ret;
  else return QListViewItem::compare(item,row,ascending);
}

//----------------------------------------------------------------------------

int Directory::rtti()const {
  return DIRECTORY;
}

//----------------------------------------------------------------------------

Directory * Directory::parent() {
  if (QListViewItem::parent() && (QListViewItem::parent()->rtti() == DIRECTORY)) {
    return static_cast<Directory*>(QListViewItem::parent());
  } else return NULL;
}

//----------------------------------------------------------------------------

void Directory::connect( QObject * receiver, const char * member ) {
  m_pDirectorySuicide->connect( receiver, member );
}

//----------------------------------------------------------------------------

void Directory::activateItem(bool resetCache)
{
//    qDebug("activateItem("+QString(resetCache ? "true" : "false")+")");
   m_pCvsFileListView->clear();
   m_pCvsFileListView->setColumnWidth(0,100);
   m_pCvsFileListView->setColumnWidth(1,100);
   m_pCvsFileListView->setColumnWidth(2,50);
   m_pCvsFileListView->setColumnWidth(3,50);
   m_pCvsFileListView->setColumnWidth(4,50);
   m_pCvsFileListView->setColumnWidth(5,100);
   m_pCvsFileListView->setColumnWidth(6,100);
   m_pCvsFileListView->setCurAbsPath(m_fullName);

   m_pNonCvsFileListView->clear();
   m_pNonCvsFileListView->setColumnWidth(0,100);
   m_pNonCvsFileListView->setColumnWidth(1,100);
   m_pNonCvsFileListView->setCurAbsPath(m_fullName);

   m_pCvsIgnoreFileListView->clear();
   m_pCvsIgnoreFileListView->setColumnWidth(0,100);
   m_pCvsIgnoreFileListView->setColumnWidth(1,100);
   m_pCvsIgnoreFileListView->setCurAbsPath(m_fullName);
   
   if (!m_disabled) {//don't activate if disabled
     qApp->setOverrideCursor(waitCursor); // takes some time on huge dirs
     checkAndShowStatus(NULL,FALSE,resetCache);
     m_pNonCvsFileListView->signalState();
     qApp->restoreOverrideCursor();
   }
}

//----------------------------------------------------------------------------

void Directory::init(const QString& fileName, 
		     CvsListView * CvsFileListView, 
		     NonCvsListView * NonCvsFileListView,
		     CvsIgnoreListView * CvsIgnoreFileListView )
{
   m_haveCvsDir = false;
   m_subDirHasCvsDir = false;
   m_isAnalyzedDir = false;
   m_entriesCached = false;
   m_tmpEntries.clear();
   m_activatingIsNecessary = true;

   m_pCvsFileListView = CvsFileListView;
   m_pNonCvsFileListView = NonCvsFileListView;
   m_pCvsIgnoreFileListView = CvsIgnoreFileListView;
   m_fullName = fileName;
   m_readable = QDir(m_fullName).isReadable();
   m_entries.setAutoDelete(true);
   m_entries = QDict<Entry>(nextPrime(0));
   m_CVSEntries.setAutoDelete(true);
   m_CVSEntries = QDict<CVSEntries>(nextPrime(0));
   m_ignoreFiles = QDict<bool>(nextPrime(0));
   m_ignoreFiles.setAutoDelete(true);
   m_nonCvsFiles = QDict<bool>(nextPrime(0));
   m_nonCvsFiles.setAutoDelete(true);
   m_curCvsState = ES_unknown;
   m_port = -1;
   m_checkedForIgnoresFile = FALSE;
   m_isAutoUpdate = FALSE;

   setStatusText(fileName);

   if(m_fullName.length() == 1) {//root
     setText(0, m_fullName);
   } else {
     setText(0, QFileInfo( m_fullName ).fileName() );
   }   

   // init global state variables
   m_curCvsStateMax = ES_probably_up_to_date;
   m_curCvsDirStateMax = ES_probably_up_to_date;

   //check for CVS/ directory
   QDir myDir(m_fullName);
   myDir.setFilter( QDir::Dirs | QDir::Hidden);
   QStringList subDirList = myDir.entryList();
   QStringList::Iterator it = subDirList.find("CVS");
   if (it != subDirList.end()) {//Found the CVS dir: read it!
     m_haveCvsDir = checkCvsDir();
   }

   //check for being a sub project and adapt disabledList
   if (m_isSubProjectRoot) {
     int au = 0;
     QString subProjectName = relativeName();
     projectSettings->get(subProjectName,DISABLED,m_disabledList);
     projectSettings->get(subProjectName,PAUTOUPDATE,au);
     if (AUTOUPDATE && (au>0) ) {
       m_isAutoUpdate = true;
       m_topDir->setAutoUpdate(true);
     }
   }

   // clear directory
   setState( ES_probably_up_to_date );

   bool dontAnalyze = false;
   if (m_virtual) {
     m_disabled = false;
     setState(ES_missing_dir);
     dontAnalyze = true;
   } else if (!m_readable ) {
     m_disabled = true;
     dontAnalyze = true;
   } else if ( getDisabled(m_fullName) ) {
     m_disabled = true;
     Directory *hlp = parent();
     if (hlp) hlp->setState(ES_unknown,true); // reset state in dir-hierarchy
     dontAnalyze = true;
   }
   if (dontAnalyze) {
     setDirIcon();
     return;
   }
   QFileInfo fInfo(m_fullName);
   if (fInfo.exists()) {
     m_lastModTimeOfDir = fInfo.lastModified();
     m_lastCvsCallTime = m_lastModTimeOfDir;
   }
   m_disabled = false;
   m_hasDSTBug = hasDSTBug(&m_fullName);
   qApp->processEvents(1);
   if(!globalStopAction){
     if ( !((depth() >= 1) && (ONTHEFLYSCANNING)) ) {
       analyzeDirs(ONTHEFLYSCANNING);
     }
     if (m_haveCvsDir) {
       addMonitoredDir(this);
     }
   }
   setDirIcon();
}

//----------------------------------------------------------------------------

void Directory::setOpen( bool open) {
  QString removed;
  setOpen(open,removed);
}

void Directory::setOpen( bool open, QString &removed) {

  bool update;
  if (update = m_topView->isUpdatesEnabled()) {
    m_topView->setUpdatesEnabled(FALSE);
  }

  removed = "";
  if (ONTHEFLYSCANNING && open) {
    QApplication::setOverrideCursor(waitCursor);
    Directory * C = (Directory *)firstChild();
    while( C ) {
      if ( (!C->m_isAnalyzedDir) && (!C->m_disabled)) {
	C->analyzeDirs(false);
	if (!C->haveCvsDir() && !C->isVirtual() && (C->childCount() == 0) ) {
	  Directory * tmp = (Directory *)C->nextSibling();
	  delete C;
	  C = tmp;
	  continue;
	}
      }
      C = (Directory *)C->nextSibling();
    }

    if (!haveCvsDir() && !isVirtual() && (childCount() == 0)) {//can't kill myself -- so call cvscontrol to sync with the event loop
      Directory * tmp = this;
      while (tmp->parent() && !(tmp->parent()->haveCvsDir()) && (tmp->parent()->childCount() == 1) && (tmp->parent() != m_topDir) ) {
	tmp = tmp->parent();
      }
      m_topDir->m_pDirectorySuicide->setValue(tmp->fullName());
      m_topDir->m_pDirectorySuicide->activate();
      removed = tmp->fullName();
    }

    setStatusText("");
    QApplication::restoreOverrideCursor();
  }
  QListViewItem::setOpen(open);
  if (update) {
    m_topView->setUpdatesEnabled(TRUE);
    m_topView->triggerUpdate();
  }
}

//----------------------------------------------------------------------------

void Directory::analyzeAll() {
  
  bool tmpScanMode = ONTHEFLYSCANNING;

  ONTHEFLYSCANNING = false;

  bool update;
  if (update = m_topView->isUpdatesEnabled()) {
    m_topView->setUpdatesEnabled(FALSE);
  }

  recAnalyzeAll();

  if (update) {
    m_topView->setUpdatesEnabled(TRUE);
    m_topView->triggerUpdate();
  }

  ONTHEFLYSCANNING = tmpScanMode;
}

//----------------------------------------------------------------------------

//This must only be called if not in onTheFlyScanning mode!
bool Directory::recAnalyzeAll() {

  Directory * C = (Directory *)firstChild();
  while( C ) {
    bool remove = false;
    if ( !C->m_disabled) {
      if (!C->m_isAnalyzedDir) {
	C->analyzeDirs(false);
	if (!C->haveCvsDir()) {
	  remove = true;
	}
      } else if (!C->recAnalyzeAll()) {
	remove = true;
      }
    }
    if (remove) {
	Directory * tmp = C;
	C = (Directory *)C->nextSibling();
	delete tmp;
	continue;
    } else C = (Directory *)C->nextSibling();
  }

  if (!haveCvsDir()) return false;
  else return true;
}

//----------------------------------------------------------------------------

//used from disableProjectSlot, enableProjectSlot, and rereadProjectOfDir
Directory * Directory::reloadChild(QString childDir) {

  bool update;
  if (update = m_topView->isUpdatesEnabled()) {
    m_topView->setUpdatesEnabled(FALSE);
  }
  Directory * C = (Directory *)firstChild();

  while( C ) {
    if( C->fullName().find(childDir)==0) {
      delete(C);
      break;
    }
    C = (Directory *)C->nextSibling();
  }

  setState( ES_unknown, true);//reset state hierarchy
  C = new Directory(this, childDir, 
		    m_pCvsFileListView,
		    m_pNonCvsFileListView,
		    m_pCvsIgnoreFileListView );
  if (ONTHEFLYSCANNING && (!C->m_disabled)) {//won't scan otherwise
    C->analyzeDirs(false);
  }
  if (update) {
    m_topView->setUpdatesEnabled(TRUE);
    m_topView->triggerUpdate();
  }
  return C;
}

//----------------------------------------------------------------------------

void Directory::removeChild(QString childDir) {

  Directory * C = (Directory *)firstChild();
  while( C ) {
    if( C->fullName().find(childDir)==0) {
      delete(C);
      break;
    }
    C = (Directory *)C->nextSibling();
  }
  setState( ES_unknown, true);//reset state hierarchy
}

//----------------------------------------------------------------------------

void Directory::removeChild(Directory * childDir) {
  delete(childDir);
  setState( ES_unknown, true);//reset state hierarchy
}

//----------------------------------------------------------------------------

QString Directory::shortName( void ) {

        QString R;
        int pos = m_fullName.findRev( '/' );

        R = m_fullName.mid( pos+1 );
        return R;
}

//----------------------------------------------------------------------------

QString Directory::relativeName() {

  QString rel;
  int pos = m_topDir->m_fullName.findRev( '/');
  rel = m_fullName.mid( pos+1);
  return rel;
}

/*---------------------------------------------------------------------------*/
/*!
\fn			void Directory::paintCell( QPainter *p, const QColorGroup &cg,
                                 int column, int width, int alignment )
\brief		Paints a dedicated directory entry with a different color.

\param		*p		
\param		&cg		
\param		column		
\param		width		
\param		alignment		

<BR><HR>*/
/*---------------------------------------------------------------------------*/

void Directory::paintCell( QPainter *p, const QColorGroup &cg,
                           int column, int width, int alignment )
{
    QColorGroup colGrp( cg );
    QColor c = colGrp.text();
 
    if (m_disabled)
    {
      colGrp.setColor( QColorGroup::Text, Qt::gray );
    }
    
    QListViewItem::paintCell( p, colGrp, column, width, alignment );
    colGrp.setColor( QColorGroup::Text, c );
}                                                                           


//----------------------------------------------------------------------------

//called after checkout, if it's a part of an existing project, and after UPDATE_DIR_CMD and QUERY_UPDATE_CMD
//returns the first not jet analyzed directory in the hierarchy
//if addVirtual==true, add as virtual dir, itter, don't analyze!
Directory * Directory::addDir(QString newDir, bool addVirtual)
{
   if (!m_isAnalyzedDir && !m_virtual ) {//don't add if scanning hasn't reached this depth jet!
     return this;
   }

   if (m_fullName.compare(newDir) == 0) {//I am it ;-)

     if (addVirtual) return NULL;
     resetState();
     Directory *myChild = (Directory*)firstChild();
     while(myChild) {
       Directory *myTmpChild = myChild;
       myChild = (Directory*)myChild->nextSibling();
       delete (myTmpChild);
     }
     m_isAnalyzedDir = false;//need to rescan!
     analyzeDirs(false);
     return NULL;

   } else {//It's in one of the subdir's path

     Directory *myChild = (Directory*)firstChild();
     while(myChild) {
       if(newDir.find(myChild->fullName()+"/" ) == 0) {
         return myChild->addDir(newDir, addVirtual);
       }
       myChild = (Directory*)myChild->nextSibling();
     }
   }

   //Not found yet? So it's a new subdir
   Directory * item = NULL;
   if (!addVirtual) {
     item = new Directory(this, newDir, m_pCvsFileListView,
			  m_pNonCvsFileListView,
			  m_pCvsIgnoreFileListView );
     
     if(item->haveCvsDir()) { // fails if dir is disabled
       m_subDirHasCvsDir = true;
     }
     else if (!item->m_disabled) delete item;
     calcDirChilds_ES_Max();
     if (item && (!item->m_isAnalyzedDir)) {
       return item;
     } else return NULL;

   } else {//virtual dir (query update)

     QString newName = newDir;
     int pos = newName.find( "/", m_fullName.length()+1);
     if (pos > -1) {
       newName = newDir.left(pos);
     }
     item = new Directory(this, newName, m_pCvsFileListView,
			  m_pNonCvsFileListView,
			  m_pCvsIgnoreFileListView, true );
     item->addDir( newDir, true);//wouldn't itter otherwise cause not analyzed
     return item;//item must never be analyzed, it is virtual!

   }

}

//----------------------------------------------------------------------------

bool Directory::analyzeDirs(bool)
{
   if (m_isAnalyzedDir) {//don't analyze twice on setOpen()
     return m_haveCvsDir || m_subDirHasCvsDir;
   }

//    qDebug("analyzing: "+m_fullName+", depth: "+QString::number(depth()));
   QDir myDir(m_fullName);
   myDir.setFilter( QDir::Dirs | QDir::Hidden);
   QStringList subDirList = myDir.entryList();
   QString path = m_fullName;
   QString curSubDirName;
   QFileInfo fileInfo;
   Directory *item;

   //is the current dir or the parent dir -- ignore it
   subDirList.remove(".");
   subDirList.remove("..");
   subDirList.remove("CVS");

   if(!subDirList.isEmpty()) {//has subdirs
      if(m_fullName.length() > 1){//is not root
         path += "/";
      }
   
      for(unsigned int i = 0; i < subDirList.count(); i++) {
         if(globalStopAction) break;   //cancel this action

         curSubDirName = subDirList[i];
         fileInfo.setFile(path + curSubDirName);

	 //ignore symlinks
         if( fileInfo.isSymLink()) continue;

	 //ignore non-cvs dirs
	 if ( (!bSCANNONCVS) && (!QFileInfo(path+curSubDirName+"/CVS").exists()) ) continue;

         if(fileInfo.isReadable() && fileInfo.isExecutable()) {//is a dir and readable

	   //at the moment lacks a test for enough mem   
	   m_isAnalyzedDir = true;//no need to analyze again in onthefly mode
	   item = new Directory(this, path + curSubDirName, 
				m_pCvsFileListView,
				m_pNonCvsFileListView,
				m_pCvsIgnoreFileListView );
	   
	   if(item->haveCvsDir()) { // fails if dir is disabled and returns true even if
	                            // there are dirs in between that have no CVS dir
	     m_subDirHasCvsDir = true;
	   }
	   else if (item->m_disabled) continue;
	   else if (ONTHEFLYSCANNING && !item->isAnalyzed()) continue;//don't know jet if there are cvs dirs somewhere in depth
	   else delete item;
	 }
      }

      calcDirChilds_ES_Max();

   } else {//has no subdirs
     m_isAnalyzedDir = true;//no need to analyze again in onthefly mode
   }

   // check and set FolderState and therewith initialize caches
   checkAndShowStatus(NULL,TRUE);

   return m_haveCvsDir || m_subDirHasCvsDir;
}

//----------------------------------------------------------------------------

bool Directory::checkCvsDir( )
{
   QString path = m_fullName;

   if(path.length() > 1){//is not root
         path += "/";
      }
      
   QFile f;
   QString line;
   
   //read Root
   f.setName(path + "CVS/Root");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      line = textStream.readLine();
      f.close();

      m_cvsRoot = line.stripWhiteSpace();
      if (!extractCVSROOT(line,
			  m_connectMethod,
			  m_userName,
			  m_passwd,
			  m_host,
			  m_port,
			  m_rootDir)) {
	return false;
      }

      if ((m_connectMethod == "local")
        || (m_connectMethod == "fork"))
      { //local
         m_userName = getenv("USER");
         m_host = "localhost";
         m_connectMethod = "local";
      }
   }
   else return false;

   //read Repository
   f.setName(path + "CVS/Repository");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      m_repository = textStream.readLine().stripWhiteSpace();
      f.close();
   }
   else return false;

   // check/set sub-project properties
   QString name = shortName();
   Directory * p = parent();
   if ( p) {
     bool addSubProject = FALSE;
     if ( p->isCvsControlled()) {
       if (m_repository != p->repository()+"/"+name) {//maybe sub project
	 //test for existence in parent entries file
	 addSubProject = TRUE;
	 f.setName(parent()->fullName() + "/CVS/Entries");
	 if(f.open(IO_ReadOnly)) {//file is readable
	   QTextStream textStream(&f); 
           textStream.setCodec(I18n::g_pTextDecoder->pTextCodecPool()->pCodecForName(I18n::g_nameOfDecoder));
	   while(!textStream.eof()) {
	     line = textStream.readLine();
	     if(line.at(0) == 'D') {//entry is a directory
	       line = line.mid(2);
	       line.truncate(line.find("/"));
	       if (line == name) {//no sub project
		 addSubProject = FALSE;
		 break;
	       }
	     }
	   }
	   f.close();
	 }
       }
     } else {
       addSubProject = TRUE;
     }

     if (addSubProject) {
       m_isSubProjectRoot = true;
       QStringList subList;
       QString subProjectName = relativeName();
       projectSettings->getSubProjects(m_topDir->shortName(),subList);
       if ( subList.find(subProjectName) == subList.end()) {
	 projectSettings->addSubProject(m_topDir->shortName(),subProjectName);
	 projectSettings->set(subProjectName,WORKDIR,m_fullName);
	 m_topDir->m_subProjectList.append(subProjectName);
       }
     }
   }

   //workaround for old cvs version
   if(m_repository.find(m_rootDir) == 0){//old cvs version detected
      m_repository = m_repository.mid(m_rootDir.length() + 1);
   }
   
   return true;
}

//----------------------------------------------------------------------------

void Directory::activate()
{
  /* Due to problems with activation, e.g.
   * expanding a dir without clicking on the
   * folder would not call activate()
   * this function body remains empty,
   * and its functionality was subsequently
   * moved to activateItem()
   */
}

//----------------------------------------------------------------------------

Directory* Directory::searchDirOfRepository(QString repository)
{
   if(repository.compare(m_rootDir + "/" + m_repository) == 0)
      return this;   //its my repository :-)

   bool isSubDir = false;
   if((m_repository.find(".")) == 0 && (m_repository.length() == 1)) {// . detected
      if(repository.find(m_rootDir + "/") == 0) {//is subdir
         isSubDir = true;
      }
   }
   else {
      if(repository.find(m_rootDir + "/" + m_repository) == 0) {//is subdir
         isSubDir = true;
      }
   }

   if(isSubDir) {//is subdir
      Directory *myChild = (Directory*)firstChild();
      Directory *result;
      while(myChild) {
         if( (result = myChild->searchDirOfRepository(repository)) ) {
            return result;
         }
         else {
            myChild = (Directory*)myChild->nextSibling();
         }
      }
   }
   
   return 0;
}

//----------------------------------------------------------------------------

Directory* Directory::searchDirOfPath(QString path, bool findVirtual) {
  if (!findVirtual && m_virtual) {
    return NULL;
  }
  if (path.compare(m_fullName) == 0) {
    return this;   //its my path :-)
  }
  
  if (path.find(m_fullName+"/") == 0) {//is subdir
    Directory *myChild = (Directory*)firstChild();
    Directory *result;
    while(myChild) {
      if ( (result = myChild->searchDirOfPath(path, findVirtual)) ) {
	return result;
      }
      else {
	myChild = (Directory*)myChild->nextSibling();
      }
    }
  }
  return NULL;
}

//----------------------------------------------------------------------------

//return the deepest valid (not disabled, not virtual) dir in hierarchy
Directory* Directory::searchLastValidDirOfPath(QString path) {
  if (m_disabled || m_virtual) {
    return NULL;
  }
  if (path.compare(m_fullName) == 0) {
    return this;   //its my path :-)
  }
  if (path.find(m_fullName+"/") == 0) {//is subdir
    Directory *myChild = (Directory*)firstChild();
    Directory *result;
    while(myChild) {
      if ( (result = myChild->searchLastValidDirOfPath(path)) ) {
	return result;
      }
      else {
	myChild = (Directory*)myChild->nextSibling();
      }
    }
    return this;
  }
  return 0;
}

//----------------------------------------------------------------------------

void Directory::recResetState( ) {
  if ( m_disabled ) return; // we don't want to scan and change disabled dirs
  Directory * D;
  m_curCvsState = ES_probably_up_to_date;
  setDirIcon();
  D = (Directory *)firstChild();
  while( D ) {
    D->recResetState();
    D = (Directory *)D->nextSibling();
  }
}

//----------------------------------------------------------------------------

void Directory::resetState( ) {
     // reset from top down
     m_topDir->recResetState();
}

Directory * Directory::topDir() {
  return m_topDir;
}

Directory * Directory::topCvsDir() {
  if (!m_isSubProjectRoot) {
    Directory * dir = parent();
    Q_CHECK_PTR(dir);
    return dir->topCvsDir();
  } else return this;
}

//----------------------------------------------------------------------------

void Directory::setState( EntryStates newState, bool fromChild) {

//   printf("\n%s:\nm_curCvsState: %i, newState: %i, m_curCvsStateMax: %i, m_curCvsDirStateMax: %i\n",fullName().ascii(),m_curCvsState,newState,m_curCvsStateMax,m_curCvsDirStateMax);

  if ( newState == m_curCvsState ) return;

  EntryStates nextState;

  if( newState > m_curCvsState ) {
    // forced or more severe
    nextState = newState;
    if (fromChild) m_curCvsDirStateMax = newState;
  }
  else { // still warnlevel required
    if (fromChild) calcDirChilds_ES_Max();
    EntryStates dirMaxState = m_curCvsDirStateMax;
    if ( m_curCvsStateMax > dirMaxState) dirMaxState = m_curCvsStateMax;
    nextState = dirMaxState; // set new state
  }

//   printf("m_curCvsState: %i, newState: %i, m_curCvsStateMax: %i, m_curCvsDirStateMax: %i\n",m_curCvsState,newState,m_curCvsStateMax,m_curCvsDirStateMax);

  // only change pix if required
  if (m_curCvsState<=ES_up_to_date && nextState<=ES_up_to_date) {
    m_curCvsState = nextState;
    return; // same level (ok), nothing to be done
  }
  else if (m_curCvsState>=ES_conflict && nextState>=ES_conflict) {
      m_curCvsState = nextState;
      return; // same level (problem) nothing to be done
  }
  else if (m_curCvsState>ES_modified && nextState>ES_modified && m_curCvsState<ES_needs_patch && nextState<ES_needs_patch) {
      m_curCvsState = nextState;
      return; // same level (uncommitted) nothing to be done
  }
  else if (m_curCvsState>=ES_needs_patch && nextState>=ES_needs_patch && m_curCvsState<ES_conflict && nextState<ES_conflict) {
      m_curCvsState = nextState;
      return; // same level (warn) nothing to be done
  }
  else m_curCvsState = nextState;

  // now set the pixmap
  setDirIcon();

  if( depth() > 0 ) {
    // non root -> has directory parent
    ((Directory *)parent())->setState( nextState, true);
  }
}

//----------------------------------------------------------------------------

void Directory::setDirIcon() {

  const char * PMName;
  if ( m_disabled) {
    PMName = "FolderDisabled16x16";
  } else if ( m_virtual) {//virtual folder
    PMName = "FolderMissing16x16";
  } else if ( m_curCvsState <= ES_up_to_date ) {// all is OK
    PMName = (m_haveCvsDir) ?
      (m_readable) ?
      (m_isAutoUpdate) ? "FolderClosedAuto26x16" : "FolderClosed16x16"
      : "FolderClosedLocked16x16"
      : (m_isAutoUpdate) ? "FolderStatusAuto26x16" : "FolderStatus16x16";
  } else if ( m_curCvsState == ES_noncvs) { // noncvs
    PMName = (m_haveCvsDir) ?
      (m_readable) ?
      (m_isAutoUpdate) ? "FolderClosedNonCVSAuto26x16" :  "FolderClosedNonCVS16x16"
      : "FolderClosedLockedNonCVS16x16"
      : (m_isAutoUpdate) ? "FolderStatusAuto26x16" : "FolderStatus16x16";
  } else if ( m_curCvsState == ES_modified) { // modified
    PMName = (m_haveCvsDir) ?
      (m_readable) ?
      (m_isAutoUpdate) ? "FolderClosedModifiedAuto26x16" : "FolderClosedModified16x16"
      : "FolderClosedLockedModified16x16"
      : (m_isAutoUpdate) ? "FolderStatusAuto26x16" : "FolderStatus16x16";
  } else if( m_curCvsState < ES_needs_patch ) { // uncommitted
    PMName = (m_haveCvsDir) ?
      (m_readable) ?
      (m_isAutoUpdate) ? "FolderClosedUncommittedAuto26x16" : "FolderClosedUncommitted16x16"
      : "FolderClosedLockedUncommitted16x16"
      : (m_isAutoUpdate) ? "FolderStatusAuto26x16" : "FolderStatus16x16";
  } else if( m_curCvsState < ES_conflict ) { // warn
    PMName = (m_haveCvsDir) ?
      (m_readable) ?
      (m_isAutoUpdate) ? "FolderClosedWarnAuto26x16" : "FolderClosedWarn16x16"
      : "FolderClosedLockedWarn16x16"
      : (m_isAutoUpdate) ? "FolderStatusAuto26x16" : "FolderStatus16x16";
  } else { // problem
    PMName = (m_haveCvsDir) ?
      (m_readable) ?
      (m_isAutoUpdate) ? "FolderClosedProblemAuto26x16" : "FolderClosedProblem16x16"
      : "FolderClosedLockedProblem16x16"
      : (m_isAutoUpdate) ? "FolderStatusAuto26x16" : "FolderStatus16x16";
  }
  setPixmap( 0, findEmbeddedPixmap (PMName));
}

//----------------------------------------------------------------------------

void Directory::setAutoUpdate(bool state) {
  m_isAutoUpdate = state;
  setDirIcon();
}

//----------------------------------------------------------------------------

EntryStates Directory::getState(QString fileName) {
  return m_entries.find(fileName.stripWhiteSpace())->state();
}

//----------------------------------------------------------------------------

void Directory::setFilesToUpToDate(QStringList& updateFileList)
{
   QString tmp;
   while (!updateFileList.isEmpty()) {
      tmp = updateFileList.first();
      setEntry(tmp, ES_up_to_date);
      updateFileList.remove(tmp);
   }
}

//----------------------------------------------------------------------------

void Directory::setCvsCallTime(QDateTime & dateTime) {
  m_lastCvsCallTime = dateTime;
  Directory * D;
  D = (Directory *)firstChild();
  while( D ) {
    D->setCvsCallTime(dateTime);
    D = (Directory *)D->nextSibling();
  }
}

//----------------------------------------------------------------------------

bool Directory::entriesFileModified() { // returns true if modified or m_modTimeOfEntries not set( so, nothing cached yet)
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFileInfo fInfo(path + "CVS/Entries");
   if (!fInfo.exists()) return false;
      
   if (m_modTimeOfEntries == fInfo.lastModified()) {
      return false;
   } else if (m_disabled) {
     return false;
   } else {
//      qDebug("CVS modified: "+m_fullName);
     return true;
   }
}

//----------------------------------------------------------------------------

bool Directory::isModified() {
  QString path = m_fullName;
  if(path.length() > 1){//is not root
    path += "/";
  }
  
  QFileInfo fInfo(path);
  if (!fInfo.exists()) return false;
  
  if (m_lastModTimeOfDir == fInfo.lastModified()) {
    return entriesFileModified();
  } else if (fInfo.lastModified() > QDateTime::currentDateTime().addSecs(-1)) {//causes problems on mounted dirs with non-synched timestamp
    //fileInfo doesn't contain msec info, so new changes are not
    //detected if they are within the same second
    //we ignore changes here, so the first run with latest changes
    //being in past more than 1 second will return true
    return entriesFileModified();
  } else {
//     qDebug("Dir modified: "+m_fullName);
    m_lastModTimeOfDir = fInfo.lastModified();
    return true;
  }
}

//----------------------------------------------------------------------------

bool Directory::entriesLogFileModified() { // returns true if modified or m_modTimeOfEntriesLog not set( so, nothing cached yet)
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFileInfo fInfo(path + "CVS/Entries.Log");
   if (!fInfo.exists()) return false;
      
   if (m_modTimeOfEntriesLog == fInfo.lastModified()) {
      return false;
   } else if (m_disabled) {
     return false;
   } else {
//      qDebug("entriesLogFileModified");
     return true;
   }
}

//----------------------------------------------------------------------------

// This method is a fix for showing dirs and files detected to be missing by QUERY_UPDATE_CMD
// For displaying, these have to be added to Entries. We don't modify CVS/Entries, but
// add temporarily until next modification of CVS/Entries.
bool Directory::setAndAppendTmpEntryQueryUpdate(QString m_curCvsFileName) {
  if (getCVSEntries()->find(m_curCvsFileName)) return false; //entry found
  QFileInfo fInfo(m_fullName+"/"+m_curCvsFileName);
  if (fInfo.exists()) return false;//don't add existing dirs/files as tmp, they might allready be in one of the other tabs

  CVSEntries *cvsEntries = new CVSEntries();
  cvsEntries->name = m_curCvsFileName;
  cvsEntries->rev.string = "locally missing";
  cvsEntries->rev.missing = false;
  cvsEntries->date.string = "unknown";
  cvsEntries->date.dummy = false;
  cvsEntries->date.newDummy = false;
  cvsEntries->date.conflict = false;
  cvsEntries->date.datetime = QDateTime::currentDateTime();
  cvsEntries->date.date = cvsEntries->date.datetime.date();
  cvsEntries->date.time = cvsEntries->date.datetime.time();
  cvsEntries->option.string = "";
  cvsEntries->option.b = false;
  cvsEntries->option.o = false;
  cvsEntries->option.v = false;
  cvsEntries->sticky.string = "";
  cvsEntries->sticky.D = false;
  cvsEntries->sticky.T = false;

  m_CVSEntries.insert(cvsEntries->name,cvsEntries);
  m_tmpEntries.append(m_curCvsFileName);
  return true;
}

//----------------------------------------------------------------------------

void Directory::removeTmpEntries(QString name) {
  QStringList::Iterator it;
  if (name.isNull()) {
    for ( it = m_tmpEntries.begin(); it != m_tmpEntries.end(); it++ ) {
      setEntry(*it,ES_unknown);//adapt the warn level before removal
      m_CVSEntries.remove(*it);
      removeEntry(*it);
    }
    m_tmpEntries.clear();
  } else if (setEntry(name,ES_unknown)) {//adapt the warn level before removal
    m_CVSEntries.remove(name);
    removeEntry(name);
    it = m_tmpEntries.find(name);
    if (it != m_tmpEntries.end()) {
      m_tmpEntries.remove(it);
    }
  }
  //remove virtual dirs
  Directory *myChild = (Directory*)firstChild();
  while(myChild) {
    if (myChild->isVirtual()) {
       Directory *myTmpChild = myChild;
       myChild = (Directory*)myChild->nextSibling();
       if (name.isNull() || (name == myTmpChild->shortName()) ) {
	 delete (myTmpChild);
	 setState(ES_unknown,true);//reset state
       }
    } else {
      myChild = (Directory*)myChild->nextSibling();
    }
  }
}

//----------------------------------------------------------------------------

void Directory::recRemoveTmpEntries(bool allProjects) {
  if (m_disabled) return;
  Directory * C = (Directory *)firstChild();
  while( C ) {
    if (allProjects || (C->topCvsDir() == topCvsDir()) ) C->recRemoveTmpEntries(allProjects);
    C = (Directory *)C->nextSibling();
  }
  removeTmpEntries();
}

//----------------------------------------------------------------------------

void Directory::recGetOpenDirsStringList(QStringList& list) {
  if (!isOpen()) return;
  else list.append(m_fullName);
  Directory * C = (Directory *)firstChild();
  while( C ) {
    C->recGetOpenDirsStringList(list);
    C = (Directory *)C->nextSibling();
  }
}

//----------------------------------------------------------------------------

//should be the only method to read CVS/Entries for proper cache state
QDict<CVSEntries>* Directory::getCVSEntries( bool forceCacheReset /*= false*/) {

  bool modified = FALSE;
  if ( !m_entriesCached
       || (modified = entriesFileModified())
       || (modified = entriesLogFileModified())
       || forceCacheReset) { // not initialized or cached jet

    QFile f;
    QString line;
    QString logMsg;

    QString path = m_fullName;
    if(path.length() > 1){//is not root
      path += "/";
    }

    m_CVSEntries.clear();
    QStringList oldTmpEntries = m_tmpEntries;
    m_tmpEntries.clear();

    if (Debug::g_logLevel == Debug::LL_GOSSIP_MONGER) {
      logMsg = m_fullName+" reading CVS dir:\n";
    }

    //read Tag
    f.setName( path + "CVS/Tag");
    if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      while(!textStream.atEnd()) {
	line = textStream.readLine();
	if ( (line.at(0)=='T') || (line.at(0)=='N') || (line.at(0)=='D') ) {
	  m_dirTag = line.stripWhiteSpace();
	  break;
	} else {
	  qDebug(m_fullName+"/CVS/Tag contains unknown entrie: "+line);
	}
      }
      if (Debug::g_logLevel == Debug::LL_GOSSIP_MONGER) {
	f.at(0);
	QTextStream s(&f);
	logMsg += "CVS/Tag:\n"+s.read();
      }
      f.close();
    } else {
      m_dirTag = "";
    }

    //read Entries
    QStringList entrieList;
    f.setName(path + "CVS/Entries");
    if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      textStream.setCodec(I18n::g_pTextDecoder->pTextCodecPool()->pCodecForName(I18n::g_nameOfDecoder));
      QFileInfo fInfo(f);
      m_modTimeOfEntries = fInfo.lastModified();

      if (modified && (m_lastCvsCallTime < m_modTimeOfEntries) ) {//clear align-cache, only in case of an external modification of CVS dir
	m_entries.clear();
	m_lastCvsCallTime = m_modTimeOfEntries;
      }
      
      // read CVS/Entries and initialize cache
      while(!textStream.eof()) {
	line = textStream.readLine();
	if(line.at(0) == 'D') {
	  continue;   //entry is a directory
	}
	if(line.at(0)=='/') {//entry seems to be correct
	  entrieList.append(line);
	}
      }
      if (Debug::g_logLevel == Debug::LL_GOSSIP_MONGER) {
	f.at(0);
	QTextStream s(&f);
	logMsg += "CVS/Entries:\n"+s.read();
      }
      f.close();
    } else {
      //qDebug("file: "+path+"CVS/Entries not readable");//will give output on each checkAndShowStatus if no CVS/Entries file
    }

    //read Entries.log
    f.setName(path + "CVS/Entries.Log");
    if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      QFileInfo fInfo(f);
      m_modTimeOfEntriesLog = fInfo.lastModified();
      
      if (modified && (m_lastCvsCallTime < m_modTimeOfEntriesLog) ) {//clear align-cache, only in case of an external modification of CVS dir
	m_entries.clear();
	m_lastCvsCallTime = m_modTimeOfEntriesLog;
      }

      // read CVS/Entries and initialize cache
      while(!textStream.eof()) {
	line = textStream.readLine();
	if(line.at(2) == 'D') {
	  continue;   //entry is a directory
	}
	if(line.at(0)=='A') {//add entry
	  line = line.mid(2);
	  entrieList.append(line);
	} else if (line.at(0)=='R') {//remove entry
	  line = line.mid(2);
	  QStringList::Iterator test = entrieList.find(line);
	  if (test != entrieList.end()) {//found such entry
	    entrieList.remove(test);
	  }
	} else {
	  qDebug("unknown line in: "+m_fullName+"/CVS/Entries.log: "+line);
	}
      }
      if (Debug::g_logLevel == Debug::LL_GOSSIP_MONGER) {
	f.at(0);
	QTextStream s(&f);
	logMsg += "CVS/Entries.log:\n"+s.read();
      }
      f.close();
    } else {
      //qDebug("file: "+path+"CVS/Entries.Log not readable");//will give output on each checkAndShowStatus if no CVS/Entries.Log file
    }

    QStringList::Iterator it;
    for (it = entrieList.begin(); it != entrieList.end(); it++) {

      int posLeft,posRight;
      CVSEntries *cvsEntries = new CVSEntries();
      line = (*it);
	  
      if((posLeft = line.find("/", 1)) > 1) {
        //cvsEntries->name = line.mid(1, posLeft - 1);
        cvsEntries->name = line.section("/", 1,1);
	if((posRight = line.find("/", posLeft + 1)) > 1) {
	  if (line.find("-",posLeft + 1) == (posLeft + 1)) {
	    cvsEntries->rev.missing = true;
	    posLeft++;
	  } else {
	    cvsEntries->rev.missing = false;
	  }
	  cvsEntries->rev.string  = line.mid(posLeft + 1, posRight - 1 - posLeft);
	  posLeft = posRight;
	  if((posRight = line.find("/", posLeft + 1)) > 1) {
	    cvsEntries->date.string = line.mid(posLeft + 1, posRight - 1 - posLeft).simplifyWhiteSpace();
	    if (cvsEntries->date.string.find("dummy timestamp") > -1) {
	      cvsEntries->date.dummy = true;
	      if(cvsEntries->date.string.find("from new-entry",16) > -1) {//modified file after switching to a different branch where this file doesn't exist ...
		cvsEntries->date.newDummy = true;
	      } else {
		cvsEntries->date.newDummy = false;
	      }
	    } else if(cvsEntries->date.string.find("Initial ") > -1) {//same as befor but in :local: mode
	      cvsEntries->date.dummy = true;
	      cvsEntries->date.newDummy = false;
	    } else {
	      cvsEntries->date.dummy = false;
	      cvsEntries->date.newDummy = false;
	    }
	    QDateTime tmpTime = QDateTime::fromString(cvsEntries->date.string);
	    cvsEntries->date.datetime = tmpTime;
	    bool invalid = false;
	    if (!tmpTime.isValid()) {//no date in entries but "Result of merge" or alike
	      cvsEntries->date.localdatetime = QDateTime::currentDateTime();//has to be marked as modified
	      cvsEntries->date.datetime = getAsUTC(cvsEntries->date.localdatetime,m_hasDSTBug);
	      invalid = true;
	    }
	    if (WINVERSION > 0) {//fix for cvs windows version that doesn't consider daylight settings
	      //seems to be a w2k bug: create a file, note timestamp and change system date from
	      //winter to summer. Surprised?
	      if (CVSVERSION == "cvs") {
		if (winOnly_isDayLight) {
		  cvsEntries->date.datetime = cvsEntries->date.datetime.addSecs(3600);
		}
		if (!invalid) {//date.string might hold merge or dummy timestamp marker
		  cvsEntries->date.localdatetime = getAsLocal(cvsEntries->date.datetime,m_hasDSTBug);
		  cvsEntries->date.string = cvsEntries->date.localdatetime.toString(LookAndFeel::g_dateTimeFormat);
		}
	      } else if (!invalid) {
		cvsEntries->date.localdatetime = getAsLocal(tmpTime,m_hasDSTBug);//convert to local time
		cvsEntries->date.string = cvsEntries->date.localdatetime.toString(LookAndFeel::g_dateTimeFormat);
	      }
	    } else if (!invalid) {
	      cvsEntries->date.localdatetime = getAsLocal(tmpTime,m_hasDSTBug);//convert to local time
	      cvsEntries->date.string = cvsEntries->date.localdatetime.toString(LookAndFeel::g_dateTimeFormat);
	    }
	    cvsEntries->date.date = cvsEntries->date.datetime.date();
	    cvsEntries->date.time = cvsEntries->date.datetime.time();
	    if (cvsEntries->date.string.find("Result of merge+")> -1) {
	      cvsEntries->date.conflict = true;
	    } else {
	      cvsEntries->date.conflict = false;
	    }
	  } else {
	    cvsEntries->date.string = "";
	  }
	  posLeft = posRight;
	  if((posRight = line.find("/", posLeft + 1)) > 1) {
	    if ( (posRight-posLeft)>1) {
	      cvsEntries->option.string = line.mid(posLeft + 1, posRight - 1 - posLeft).stripWhiteSpace();
	      switch (cvsEntries->option.string.at(2).latin1()) {
		case 'b': {
		  cvsEntries->option.b = true;
		  cvsEntries->option.o = false;
		  cvsEntries->option.v = false;
		  break;
		}
		case 'o': {
		  cvsEntries->option.b = false;
		  cvsEntries->option.o = true;
		  cvsEntries->option.v = false;
		  break;
		}
		case 'v': {
		  cvsEntries->option.b = false;
		  cvsEntries->option.o = false;
		  cvsEntries->option.v = true;
		  break;
		}
		default: {
		  cvsEntries->option.b = false;
		  cvsEntries->option.o = false;
		  cvsEntries->option.v = false;
		}
	      }
	    } else {
	      cvsEntries->option.string = "";
	      cvsEntries->option.b = false;
	      cvsEntries->option.o = false;
	      cvsEntries->option.v = false;
	    }
	  }
	  posLeft = line.findRev("/");
	  cvsEntries->sticky.string = line.mid(posLeft+2);
	  if( line.find("T",posLeft) > 0) {   //sticky tag
	    cvsEntries->sticky.T = true;
	  } else {
	    cvsEntries->sticky.T = false;
	  }
	  if( line.find("D",posLeft) > 0) {   //sticky date
	    cvsEntries->sticky.D = true;
	  } else {
	    cvsEntries->sticky.D = false;
	  }
	} else {
	  delete(cvsEntries);
	  continue;
	}
	if (m_CVSEntries.size() <= m_CVSEntries.count()) m_CVSEntries.resize( nextPrime(m_CVSEntries.size()));
	m_CVSEntries.insert(cvsEntries->name,cvsEntries);
      } else {
	delete(cvsEntries);
	continue;
      }
    }      

    //readd previous tmp entries
    QStringList::Iterator iter;
    for ( iter = oldTmpEntries.begin(); iter != oldTmpEntries.end(); iter++ ) {
      setAndAppendTmpEntryQueryUpdate(*iter);
    }
    if ( Debug::g_logLevel == Debug::LL_GOSSIP_MONGER) {
      if ( !m_tmpEntries.isEmpty()) {
	logMsg += "tmp entries:\n\""+m_tmpEntries.join(" ")+"\"";
      }
      Debug::g_pLog->log(Debug::LL_GOSSIP_MONGER, logMsg);
    }

    //and set cache flag
    m_entriesCached = TRUE;

//     QDictIterator<CVSEntries>iterator(m_CVSEntries);
//     while(iterator.current()) {
//       qDebug( "checking name: " + iterator.current()->name );
//       qDebug( "rev: " + iterator.current()->rev.string );
//       qDebug( "rev-missing: " + QString(iterator.current()->rev.missing ? "true" : "false") );
//       qDebug( "datestring: " + iterator.current()->date.string );
//       qDebug( "datetime: " + iterator.current()->date.datetime.toString() );
//       qDebug( "date: " + iterator.current()->date.date.toString() );
//       qDebug( "date-dummy: " + QString(iterator.current()->date.dummy ? "true" : "false") );
//       qDebug( "date-missing: " + QString(iterator.current()->date.conflict ? "true" : "false") );
//       qDebug( "sticky-T: " + QString(iterator.current()->sticky.T ? "true" : "false") );
//       qDebug( "sticky-D: " + QString(iterator.current()->sticky.D ? "true" : "false") );
//       qDebug( "==============================================================");
//       ++iterator;
//     }

  }
  return &m_CVSEntries;
}

//----------------------------------------------------------------------------

void Directory::fillFileListWithEntries(QStringList& fileList) {
  QDictIterator<CVSEntries> it(*getCVSEntries());
  while(it.current()) {
    fileList.append(it.current()->name);
    ++it;
  }
}

//----------------------------------------------------------------------------

void Directory::fillFileListWithFullNameEntries(QStringList& fileList) {
  QDictIterator<CVSEntries> it(*getCVSEntries());
  while(it.current()) {
    fileList.append(m_fullName+"/"+it.current()->name);
    ++it;
  }
}

//----------------------------------------------------------------------------

void Directory::setAllToUpToDate() {
  //alle entries im eignen verzeichnis auf uptodate setzen
  QDictIterator<CVSEntries> it(*getCVSEntries());
  while(it.current()) {
    setAndAppendEntry(it.current()->name, ES_up_to_date);
    ++it;
  }
  //jetzt die subdirs...
  Directory *myChild = (Directory*)firstChild();
  while(myChild) {
    myChild->setAllToUpToDate();
    myChild = (Directory*)myChild->nextSibling();
  }
}

//----------------------------------------------------------------------------

bool Directory::setEntry(QString &fileName, EntryStates stateId) {
  Entry *entry;
  if (!(entry = m_entries.find(fileName))) return false;
  entry->setState(stateId);
  if (Debug::g_pLog) {
    QString msg = m_fullName+" changing Entry: ";
    msg += fileName;
    msg += ", state: "+QString::number(stateId);
    Debug::g_pLog->log(Debug::LL_GOSSIP_MONGER, msg);
  }
  return true;
}

//----------------------------------------------------------------------------

void Directory::removeEntry(QString &fileName) {
  m_entries.remove(fileName);
  if (Debug::g_pLog) {
    QString msg = m_fullName+" removing Entry: ";
    msg += fileName;
    Debug::g_pLog->log(Debug::LL_GOSSIP_MONGER, msg);
  }
}

//----------------------------------------------------------------------------

void Directory::appendEntry(QString &key, Entry *entry) {
  if (m_entries.size() <= m_entries.count()) m_entries.resize( nextPrime(m_entries.size()));
  m_entries.insert(key,entry);
  if (entry->state()>m_curCvsStateMax) m_curCvsStateMax=entry->state();
  if (Debug::g_pLog) {
    QString msg = m_fullName+" appending Entry: ";
    msg += key;
    msg += ", state: "+QString::number(entry->state());
    Debug::g_pLog->log(Debug::LL_GOSSIP_MONGER, msg);
  }
}

//----------------------------------------------------------------------------

void Directory::set_ES_Max(EntryStates e_state) {
  m_curCvsStateMax = e_state;
}

//----------------------------------------------------------------------------

EntryStates Directory::get_ES_Max() {
  return m_curCvsStateMax;
}

//----------------------------------------------------------------------------

void Directory::calc_ES_Max() {
  if (m_nonCvsFiles.isEmpty()) m_curCvsStateMax = ES_unknown;//set modified state if there are non-cvs-files
  else m_curCvsStateMax = ES_noncvs;
  QDictIterator<Entry>it(m_entries);
  while(it.current()) {
    if (it.current()->state() > m_curCvsStateMax) m_curCvsStateMax = it.current()->state();
    ++it;
  }
}

//----------------------------------------------------------------------------

void Directory::calcDirChilds_ES_Max() {
  m_curCvsDirStateMax = ES_unknown;
  Directory * C = (Directory *)firstChild();
  while( C ) {
    if( C->m_curCvsState > m_curCvsDirStateMax) m_curCvsDirStateMax = C->m_curCvsState;
      C = (Directory *)C->nextSibling();
  }
}

//----------------------------------------------------------------------------

void Directory::setAndAppendEntry(QString &fileName, EntryStates stateId)
{
   if(!setEntry(fileName, stateId)) {
     appendEntry(fileName, new Entry(fileName, stateId, this ));
     return;
   }
}

//----------------------------------------------------------------------------

bool Directory::recCopyTree(QString src,QString dst, bool deleteSource /*=TRUE*/) {

  QDir newDir;
  QString dstDir = dst+"/"+src.mid(src.findRev("/"));
  if (!newDir.mkdir(dstDir) ) {
    qDebug("can't create dir: "+dstDir );
    return FALSE;
  }

  //dirs first
  QStringList AllSubDirs;
  QDir D( src);
  D.setFilter(QDir::Dirs);
  AllSubDirs = D.entryList();

  // remove . and ..
  AllSubDirs.remove( "." );
  AllSubDirs.remove( ".." );
  AllSubDirs.remove( "CVS" );
  
  QStringList::Iterator fileit;  //VC6 does _not_ like it...
  for (fileit = AllSubDirs.begin(); fileit != AllSubDirs.end(); fileit++) {
    recCopyTree(src+"/"+(*fileit),dstDir,deleteSource);
  }
  
  //now the files
  QStringList AllFilesInDirectory;
  D.setFilter( QDir::Files | QDir::Hidden);
  AllFilesInDirectory = D.entryList();

  for (fileit = AllFilesInDirectory.begin(); fileit != AllFilesInDirectory.end(); fileit++) {
    QString srcName = src+"/"+(*fileit);
    QString dstName = dstDir+"/"+(*fileit);
    QFileInfo fInfo(srcName);
    int permission = 0;
    if (fInfo.isReadable()) permission = (permission | READABLE);
    if (fInfo.isWritable()) permission = (permission | WRITEABLE);
    if (fInfo.isExecutable()) permission = (permission | EXECUTABLE);
    if (!copyFile(srcName, dstName, permission, deleteSource) ) {
      qDebug("can't copy file: "+srcName+" -> "+dstName);
      return FALSE;
    }
  }
  return TRUE;
}

//----------------------------------------------------------------------------

bool Directory::recCopyCvsFiles(QString nextDir) {

  QDir newDir(nextDir);
  if(!newDir.mkdir(nextDir)) {
    qDebug("can't create dir: "+nextDir);
    return false;
  }

  QDict<CVSEntries> *cvsEntries = getCVSEntries();
  QDictIterator<CVSEntries> it(*cvsEntries);
  while(it.current()) {
    QString source = m_fullName+"/"+it.current()->name;
    QString target = nextDir+"/"+it.current()->name;

    if (QFileInfo(source).exists()) {
      if (!copyFile(source,target,READABLE)) {
	qDebug("can't copy file: "+source+" -> "+target);
	return false;
      }
    }
    ++it;
  }

  Directory * child = (Directory*)firstChild();
  while( child) {
    if ( !(child->m_disabled || child->m_virtual) ) {
      if (!child->recCopyCvsFiles(nextDir+"/"+child->shortName())) {
	return false;
      }
    }
    child = (Directory*)child->nextSibling();
  }
  return true;
}

//----------------------------------------------------------------------------

bool Directory::copyFile(QString src, QString dst, int permission, bool deleteSource /*=FALSE*/) {

  QFile srcFile( src );
  if (srcFile.open( IO_ReadOnly )) {
    unsigned int size = srcFile.size();
    char * buff = new char[size];
    QDataStream srcStream(&srcFile);
    srcStream.readRawBytes(buff,size);
    
    QFile dstFile( dst);
    if (dstFile.open( IO_WriteOnly )) {
      QDataStream dstStream(&dstFile);
      dstStream.writeRawBytes(buff,size);
      dstFile.close();
    } else {
      delete buff;
      return false;
    }
    setPermission(dstFile,permission);
    delete buff;
    srcFile.close();
    if (deleteSource) {
      setPermission(srcFile,READABLE | WRITEABLE);
      return srcFile.remove();
    } else {
      return true;
    }
  } else return false;
}

//----------------------------------------------------------------------------

bool Directory::recRemoveUnregisteredFiles(QString &filter) {

  QDir D( m_fullName);
  D.setFilter( QDir::Files | QDir::Hidden);
  D.setNameFilter( filter);
  QStringList fileList = D.entryList();

  QDict<CVSEntries> *cvsEntries = getCVSEntries();
  for (QStringList::Iterator fileit = fileList.begin(); fileit != fileList.end(); fileit++) {
    CVSEntries *cvsEntry = cvsEntries->find(*fileit);
    if (!cvsEntry) {
      QString fileName = m_fullName + "/" + (*fileit);
      QFile file(fileName);
      setPermission(file, READABLE | WRITEABLE);
      if (!file.remove()) {
	filter = fileName;
	return false;
      }
    }
  }

  Directory * child = (Directory*)firstChild();
  while( child) {
    if ( !(child->m_disabled || child->m_virtual) ) {
      if (!child->recRemoveUnregisteredFiles(filter)) {
	return false;
      }
    }
    child = (Directory*)child->nextSibling();
  }

  return true;
}

//----------------------------------------------------------------------------

//Check and modify the dir state according to non-cvs files,
//use !!!only!!! for this!!!
void Directory::checkNonCvsFilesDirState() {
  if (m_nonCvsFiles.isEmpty()) {
    if (m_curCvsStateMax == ES_noncvs) {
      calc_ES_Max();
      setState( m_curCvsStateMax);
    }
  } else if (m_curCvsStateMax < ES_noncvs) {
    m_curCvsStateMax = ES_noncvs;
    setState( ES_noncvs);
  }
}

//----------------------------------------------------------------------------

EntryStates Directory::alignWithEntries(QString name, EntryStates stateId) {

   Entry *entry = m_entries.find(name);

   if(entry) {//file allready registered
     //get and set state
     switch(stateId)
       {
       case ES_modified:
	 if((entry->state() == ES_needs_patch)
	    || (entry->state() == ES_needs_merge)) {
	   entry->setState(ES_needs_merge);
	   stateId = ES_needs_merge;
	 }
	 else{
	   if(entry->state() == ES_added) {
	     stateId = ES_added;
	   }
	   else
	     entry->setState(ES_modified);
	 }
	 break;
       case ES_missing:
	 if(entry->state() == ES_needs_checkout) {
	   stateId = ES_needs_checkout;
	 } else if (entry->state() == ES_missing_dir) {
	   stateId = ES_missing_dir;
	   break;
	 } else {
	   entry->setState(ES_needs_checkout);
	   stateId = ES_needs_checkout;
	 }
	 break;
       case ES_added://special for merge, added files are displayed as U
	 if(entry->state() == ES_up_to_date) {
	   entry->setState(ES_added);
	 }
	 break;
       case ES_probably_up_to_date:
	 if((entry->state() == ES_added) || (entry->state() == ES_modified)) {
	   entry->setState(ES_probably_up_to_date);
	 } else {
	   stateId = entry->state();
	 }
	 break;
       case ES_removed://special if files are removed from outside lincvs
	   entry->setState(ES_removed);
	   break;
       case ES_probably_up_to_date_and_timezone_incorrect:
	 if((entry->state() == ES_added) || (entry->state() == ES_modified)) {
	   entry->setState(ES_probably_up_to_date);
	 }
	 stateId = entry->state();
	 break;
       default:
	 stateId = entry->state();
	 break;
       }            
   }
   else {
      appendEntry(name, new Entry(name, stateId, this));
   }
   return stateId;
}

//----------------------------------------------------------------------------

//recursive check for modifications, only detected by a timestamp
//change of the dir or its CVS/Entries file
//if checkCvsFiles is true, also check for a change in each files timestamp
//if modifications are detected, check and set the new dir state
void Directory::recCheckForModifications(bool checkCvsFiles) {

  if ( (m_disabled) || (m_virtual) ) return;

  Directory * C = (Directory *)firstChild();
  while( C ) {
    C->recCheckForModifications(checkCvsFiles);
    C = (Directory *)C->nextSibling();
  }
  if ( isModified() || (checkCvsFiles && isFilesModified()) ) {
    checkAndShowStatus(NULL,TRUE,TRUE);
    if ( Debug::g_logLevel == Debug::LL_GOSSIP_MONGER) {
      Debug::g_pLog->log(Debug::LL_GOSSIP_MONGER, "Detected changes: "+m_fullName);
    }
  }
}

//----------------------------------------------------------------------------

//check all files for differences between timestamp of CVS/Entries
//returns true if modified files are detected
bool Directory::isFilesModified() {

  QDictIterator<CVSEntries> it(*getCVSEntries());
  QString path = m_fullName+"/";
  while(it.current()) {

    QFileInfo finfo(path + it.current()->name);
    if (!finfo.exists()) {
//       qDebug("missing file: "+path+it.current()->name);
      return TRUE;
    }
    QDateTime localTime = finfo.lastModified();
    localTime = getAsUTC(localTime,m_hasDSTBug);
    EntryStates es = getState(it.current()->name);
    if ( (es <= ES_up_to_date)
	 && (it.current()->date.datetime != localTime)
	 && (es != ES_probably_up_to_date_and_timezone_incorrect) ) {
//       qDebug("File modified: "+path+it.current()->name);
      return TRUE;
    }
    ++it;
  }
  return FALSE;
}

//----------------------------------------------------------------------------

void Directory::validateCvsFilesStatus( bool recursive /* = FALSE*/,
					bool forceCacheReset /* = FALSE*/,
					bool forceEntriesReset /* = FALSE*/) {

  if (forceEntriesReset) {
    m_entries.clear();
  }

  // now check and set state
  QDictIterator<CVSEntries> it(*getCVSEntries(forceCacheReset));
  while(it.current()) {
    setStatusInFileListView(NULL, it.current(),TRUE);
    ++it;
  }

  if (recursive) {
    Directory * C = (Directory *)firstChild();
    while( C ) {
      C->validateCvsFilesStatus(recursive,forceCacheReset,forceEntriesReset);
      C = (Directory *)C->nextSibling();
    }
  }
}

//----------------------------------------------------------------------------

//Add a single file, only known by its name, and display it
void Directory::checkAndUpdateFileCache(QString& file) {

  // first read .cvsignore
  bool homeIgnoresChanged;
  bool dirIgnoresChanged;
  const QString homeFilesToIgnore = getHomeFilesToIgnore(m_lastTimeHomeIgnoresChanged,homeIgnoresChanged);
  const QString filesToIgnore = getDirFilesToIgnore(homeFilesToIgnore,dirIgnoresChanged);
  
  bool ignoreCacheReseted = false;
  if (dirIgnoresChanged || homeIgnoresChanged) {
    if (isSelected()) {
      activateItem(TRUE);
      return;
    } else {
      m_ignoreFiles.clear();
      m_nonCvsFiles.clear();
      ignoreCacheReseted = true;
    }
  }

  QDict<CVSEntries> *cvsEntries = getCVSEntries();
  CVSEntries *cvsEntry = cvsEntries->find(file);
  if (cvsEntry) {
    QListViewItem * item = NULL;
    if (isSelected()) {
      item = m_pCvsFileListView->firstChild();
      while (item) {
	if (item->text(0) == file) {
	  break;
	} else item = item->nextSibling();
      }
    }
    setStatusInFileListView( static_cast<FileListViewItem*>(item), cvsEntry, !isSelected());
  } else {
    bool found;
    bool doIgnore = updateFileCacheAndCheckForIgnores(filesToIgnore,file,ignoreCacheReseted,found);
    if (isSelected() && (!found) ) {
      FileListViewItem * item;

      bool IsDir = false;
      bool isWritable = false;
      bool isExecutable = false;
      
      QFileInfo info(m_fullName+"/"+file);
      IsDir = info.isDir();
      isWritable = info.isWritable();
      isExecutable = info.isExecutable();
      
      if (doIgnore) {
       // show cvsignore entries
       item = new FileListViewItem(m_pCvsIgnoreFileListView,IsDir);
       item->setPixmap(0, findEmbeddedPixmap((IsDir) ?
					     (isExecutable) ?  "FolderClosed16x16" : "FolderClosedLocked16x16" :
					     (isWritable) ? "FileStatus16x16" : "FileLocked16x16"));
      } else {
       //show non-cvs entries
       item = new FileListViewItem(m_pNonCvsFileListView,IsDir);
       item->setPixmap(0, findEmbeddedPixmap((IsDir) ?
					     (isExecutable) ?  "FolderStatus16x16" : "FolderClosedLocked16x16" :
					     (isWritable) ? "FileUnknown16x16" : "FileLocked16x16"));
       m_pNonCvsFileListView->signalState();
      }
      
      item->setText(0, file);
      QDateTime localDate = info.lastModified();
      item->setModifiedDate(1, &localDate, localDate.toString(LookAndFeel::g_dateTimeFormat));
    }
  }
}

//----------------------------------------------------------------------------

//Remove a single file, only known by its name, and remove from display
bool Directory::removeFromFileCache(QString& file) {
  if (m_ignoreFiles.remove(file)) {
    QListViewItem * item = m_pCvsIgnoreFileListView->firstChild();
    while (item) {
      if (item->text(0) == file) {
       QListViewItem * hlp = item;
       item = item->nextSibling();
       delete hlp;
       continue;
      } else item = item->nextSibling();
    }
    return TRUE;
  } else if (m_nonCvsFiles.remove(file)) {
    QListViewItem * item = m_pNonCvsFileListView->firstChild();
    while (item) {
      if (item->text(0) == file) {
       QListViewItem * hlp = item;
       item = item->nextSibling();
       delete hlp;
       continue;
      } else item = item->nextSibling();
    }
    checkNonCvsFilesDirState();
    if (isSelected()) {
      m_pNonCvsFileListView->signalState();
    }
    return TRUE;
  }
  return FALSE;
}

//----------------------------------------------------------------------------

//Main function for checking file state and displaying, used by DirWatch,AnalyzeDir,cvscontrol::checkStatus ...
void Directory::checkAndShowStatus(FileListViewItem *item /* = NULL*/, bool checkOnly /* = FALSE*/, bool resetCache /* = FALSE*/) {

  QDict<CVSEntries> *cvsEntries = getCVSEntries();

  if(item) {//only one item
    QString fileName = item->text(0);
    CVSEntries *cvsEntry = cvsEntries->find(fileName);
    if (cvsEntry) {
      setStatusInFileListView(item, cvsEntry);
    }
  } else { // read entries in directory
    QStringList AllFilesInDirectory;
    
    QFile f;
    QString line;
    QString path = m_fullName;
    if(path.length() > 1){//is not root
      path += "/";
    }

    QDir D( path);
    D.setFilter( QDir::Files | QDir::Dirs | QDir::Hidden | QDir::System);
    AllFilesInDirectory = D.entryList();

    // remove . and ..
    AllFilesInDirectory.remove( "." );
    AllFilesInDirectory.remove( ".." );
    AllFilesInDirectory.remove( "CVS" );

    // now check and set state
    QDictIterator<CVSEntries> it(*cvsEntries);
    while(it.current()) {
      item = setStatusInFileListView(0, it.current(), checkOnly);
      AllFilesInDirectory.remove(it.current()->name);
      ++it;
    }

    // show 'non-cvs' and 'ignore files' entries

    // first read .cvsignore
    bool homeIgnoresChanged;
    bool dirIgnoresChanged;
    const QString homeFilesToIgnore = getHomeFilesToIgnore(m_lastTimeHomeIgnoresChanged,homeIgnoresChanged);
    const QString filesToIgnore = getDirFilesToIgnore(homeFilesToIgnore,dirIgnoresChanged);

    //reset cache if changes occured
    bool ignoreCacheReseted = false;
    if (resetCache || dirIgnoresChanged || homeIgnoresChanged) {
	m_ignoreFiles.clear();
	m_nonCvsFiles.clear();
	ignoreCacheReseted = true;
    }

    // now check the files
    bool IsDir = false;
    bool isWritable = false;
    bool isExecutable = false;

    for (QStringList::Iterator fileit = AllFilesInDirectory.begin(); fileit != AllFilesInDirectory.end(); fileit++) {
      QFileInfo info(path + (*fileit));

      IsDir = info.isDir();
      isWritable = info.isWritable();
      isExecutable = info.isExecutable();

      if (IsDir) {
          QFile F(path + (*fileit) + "/CVS/Root");
          if (F.exists())	{
              // CVS directory -> do NOT show
              continue;
          }
      }

      //check wether file belongs to .cvsignore or is not registered
      bool doIgnore = false;
      bool found;
      doIgnore = updateFileCacheAndCheckForIgnores(filesToIgnore,*fileit,ignoreCacheReseted, found);

      if (doIgnore) {
	// continue here if checkOnly, don't show
	if (checkOnly) continue;
	// show cvsignore entries
	item = new FileListViewItem(m_pCvsIgnoreFileListView,IsDir);
	item->setPixmap(0, findEmbeddedPixmap((IsDir) ?
					      (isExecutable) ?  "FolderClosed16x16" : "FolderClosedLocked16x16" :
					      (isWritable) ? "FileStatus16x16" : "FileLocked16x16"));
      } else {
	// stop here if checkOnly because now we know the non-cvs state (dir state was allready set in updateFileCacheAndCheckForIgnores
	if (checkOnly) return;
	//show non-cvs entries
	item = new FileListViewItem(m_pNonCvsFileListView,IsDir);
	item->setPixmap(0, findEmbeddedPixmap((IsDir) ?
					      (isExecutable) ?  "FolderStatus16x16" : "FolderClosedLocked16x16" :
					      (isWritable) ? "FileUnknown16x16" : "FileLocked16x16"));
      }
      
      item->setText(0, *fileit);
      QDateTime localDate = info.lastModified();
      item->setModifiedDate(1, &localDate, localDate.toString(LookAndFeel::g_dateTimeFormat));
    }
    checkNonCvsFilesDirState();
  }
}

//----------------------------------------------------------------------------

bool Directory::updateFileCacheAndCheckForIgnores(const QString filesToIgnore, QString& file, bool ignoreCacheReseted, bool& found) {

  bool doIgnore = FALSE;
  found = FALSE;
  if (!ignoreCacheReseted) {
    if (m_ignoreFiles.find(file)) {
      doIgnore = true;
      found = true;
    } else if (m_nonCvsFiles.find(file)) {
      found = true;
    }
  }
  if (!found) {
    //reimplemented old version of ignore scan,
    //now using wildcard mode
    //this is at about factor 2 faster on lin!!!
    QStringList ignf = QStringList::split(" ",filesToIgnore);
    QRegExp exp;
    exp.setWildcard(true);

    for (QStringList::Iterator it = ignf.begin(); it != ignf.end(); it++) {
      exp.setPattern(*it);
      if ( exp.exactMatch(file) ) {
	doIgnore = true;
	break;
      }
    }
    if (doIgnore) {
      if (m_ignoreFiles.size()<=m_ignoreFiles.count()) {
	m_ignoreFiles.resize(nextPrime(m_ignoreFiles.size()));
      }
      m_ignoreFiles.insert(file,new bool(true));
    } else {
      if (m_nonCvsFiles.size()<=m_nonCvsFiles.count()) {
	m_nonCvsFiles.resize(nextPrime(m_nonCvsFiles.size()));
      }
      m_nonCvsFiles.insert(file,new bool(true));
      checkNonCvsFilesDirState();
    }
  }
  return doIgnore;
}

//----------------------------------------------------------------------------

FileListViewItem * Directory::setStatusInFileListView(FileListViewItem *item, CVSEntries *cvsEntry, bool checkOnly /* = FALSE*/) {
  EntryStates stateId = ES_unknown;
  QString state, modDateLocal;
  bool sticky = false;
  char *pix;
  QDateTime localDate, localDateUtc;
  QString path = m_fullName;
  if(path.length() > 1){//is not root
    path += "/";
  }

  if (cvsEntry->sticky.T || cvsEntry->sticky.D) sticky = true;

  //status?
  QFileInfo info(path + cvsEntry->name);
  if(info.exists()){
    localDate = info.lastModified();
    localDateUtc = getAsUTC(localDate,m_hasDSTBug);

    if(localDateUtc == cvsEntry->date.datetime){
      stateId = ES_probably_up_to_date;
    }
    else {
      if(cvsEntry->date.dummy) {
	if (cvsEntry->rev.missing) {
	  stateId = ES_removed;
	} else if (cvsEntry->date.newDummy) {
	  stateId = ES_modified;
	} else stateId = ES_added;
      }
      else if (cvsEntry->date.conflict) stateId = ES_conflict;
      else {
	if ( (localDateUtc.time().second() == cvsEntry->date.time.second()) &&
	     (localDateUtc.time().minute() == cvsEntry->date.time.minute())) {
//   	  cout << "timezone: " << cvsEntry->name.ascii() <<
// 	    ", utc: ->" << modDateUtc << "<- entry: ->" <<
// 	    cvsEntry->date.string << "<-!!!\n";
	  stateId = ES_probably_up_to_date_and_timezone_incorrect;
	}
	else stateId = ES_modified;
      }
    }
  }
  else if (cvsEntry->rev.missing) {
    stateId = ES_removed;
  }
  else stateId = ES_missing;

  stateId = alignWithEntries(cvsEntry->name, stateId);

  if (checkOnly) return NULL; // we only check and set the states, displaying is timeconsuming

  switch(stateId)
    {
    case ES_probably_up_to_date:
      state = QObject::tr("seems up to date");
      pix = getUnchangedPix(info);
      break;
    case ES_probably_up_to_date_and_timezone_incorrect:
      state = QObject::tr("!!check timezone: incorrect!!");
      //pix = &pixUnchangedTimezoneConflict;
      pix = getUnchangedPix(info);
      break;
    case ES_up_to_date:
      state = QObject::tr("up to date");
      pix = getUnchangedPix(info);
      break;
    case ES_modified:
      state = QObject::tr("modified");
      pix = "FileModified16x16";
      modDateLocal = localDate.toString(LookAndFeel::g_dateTimeFormat);
      break;
    case ES_needs_patch:
      state = QObject::tr("needs patch");
      pix = "FileNeedsPatch16x16";
      break;
    case ES_needs_merge:
      state = QObject::tr("needs merge");
      pix = "FileNeedsMerge16x16";
      modDateLocal = localDate.toString(LookAndFeel::g_dateTimeFormat);
      break;
    case ES_needs_checkout:
      state = QObject::tr("needs checkout");
      pix = "FileNeedsCheckout16x16";
      break;
    case ES_missing:
      state = QObject::tr("missing");
      pix = "FileRemoved16x16";
      break;
    case ES_missing_dir:
      state = QObject::tr("needs checkout");
      pix = "FolderMissing16x16";
      break;
    case ES_conflict:
      state = QObject::tr("conflict");
      pix = "FileConflict16x16";
      break;
    case ES_added:
      //welche zeit anzeigen???
      state = QObject::tr("commit to add");
      pix = "FileAdded16x16";
      modDateLocal = localDate.toString(LookAndFeel::g_dateTimeFormat);
      break;
    case ES_removed:
      //welche zeit anzeigen???
      state = QObject::tr("commit to remove");
      pix = "FileRemoved16x16";
      break;
    default:
      state = QObject::tr("unknown");
      pix = getUnchangedPix(info);
    }
            
  if( item == 0 ) {
    item = new FileListViewItem(m_pCvsFileListView );
  }

  item->setText(0, cvsEntry->name );
  item->setText(1, cvsEntry->rev.string );
  item->setText(2, cvsEntry->sticky.string );
  item->setText(3, cvsEntry->option.string ); 
  item->setText(4, state); 
  item->setDate(5, &cvsEntry->date.localdatetime, cvsEntry->date.string);
  item->setModifiedDate(6, &localDate, modDateLocal);

  item->setPixmap(0, findEmbeddedPixmap (pix));
  if (sticky) {
    item->setPixmap(2, findEmbeddedPixmap ("Tag16x16"));
  }  else {
    item->setPixmap(2, 0);
  }
  if (cvsEntry->option.b) {
    item->setPixmap(3, findEmbeddedPixmap ("FileBinary16x16"));
  } else {
    item->setPixmap(3, 0);
  }
  return item;
}

//----------------------------------------------------------------------------

char * Directory::getUnchangedPix(QFileInfo & info)                             
{                                                                               
   if(info.isWritable())                                                        
   {                                                                            
      return "FileWriteable16x16";                                            
   }                                                                            
   else                                                                         
   {                                                                            
      return "FileUnchanged16x16";
   }                                                                            
}                                                                               

//----------------------------------------------------------------------------
                      
bool Directory::loginOk(CCvsOutput *pMessages, bool showMessage)
{
   if(!m_haveCvsDir) {
      QMessageBox::warning(qApp->mainWidget(), QObject::tr("Warning"), QObject::tr("Is not a CVS directory."), 0);
      return false;
   }

   if(m_connectMethod.find("pserver") == -1) {
      if(showMessage) {
         QMessageBox::information(qApp->mainWidget(), QObject::tr("Information"), 
                                  QObject::tr("No login necessary. It's a local CVS repository"), 0);
      }

      return true;
   }
	
   QString cvsRoot = ":" + m_connectMethod + ":" + m_userName + 
                     "@" + m_host + ":" + QString::number(m_port) + m_rootDir;

	if (!isInCvsPass(cvsRoot))
	{

	        bool ok = false;

#ifdef QT_NEED_PWDIALOGPATCH
		QString pwd;
		PasswordDialogImpl *pwDialog = new PasswordDialogImpl(0);
		pwDialog->Label->setText( m_cvsRoot);
		if (pwDialog->exec()) {
		  pwd = pwDialog->TextField->text();
		  ok = true;
		}
		delete pwDialog;
#else
		QString pwd = QInputDialog::getText("Enter CVS password",
			m_cvsRoot, QLineEdit::Password, QString::null,
			&ok, 0/*dynamic_cast<QWidget *>(this)*/, "Password Dialog");
#endif

		if (!ok)
			return false;

		Login *l = new Login( pMessages, m_userName, m_host, m_port, m_rootDir);
		l->doLogin(pwd);
		delete l;
            
		if (!isInCvsPass(cvsRoot))
		{
			QMessageBox::warning(qApp->mainWidget(), 
				QObject::tr("Warning"), QObject::tr("Login failed."), 0);

			return false;
		}

		if (showMessage)
		{
			QMessageBox::information(qApp->mainWidget(), 
				QObject::tr("Information"),
				QObject::tr("Login successfully."), 0);
		}

		return true;
	}
   
   if(showMessage) {
     QMessageBox::information(qApp->mainWidget(), 
			      QObject::tr("Information"), QObject::tr("You are already logged in."), 0);
   }
   
   return true;
}

//----------------------------------------------------------------------------

void Directory::removeLogin(CCvsOutput *pMessages) {
  if(!m_haveCvsDir) {
    QMessageBox::warning(0, QObject::tr("Warning"), QObject::tr("Is not a CVS directory."), 0);
    return;
  }
  if(m_connectMethod.find("pserver") == -1) {
    QMessageBox::information(0, QObject::tr("Information"), 
			     QObject::tr("Logout impossible. It's a local CVS repository"), 0);
    return;
  }

  Login *l = new Login( pMessages, m_userName, m_host, m_port, m_rootDir);
  if (!(l->removeCvsPassEntry())) {
    QMessageBox::information(qApp->mainWidget(), QObject::tr("Information"), 
			     QObject::tr("Logout impossible. You have not been logged in."), 0);
  }
  delete l;
}

//---------------------------------------------------------------------------

void Directory::updateDisabledList(QStringList& list) {
  m_disabledList = list;
}

void Directory::updateSubProjectList(QStringList& list) {
  m_subProjectList = list;
}

bool Directory::getDisabled(QString filename)
{
  QStringList* list = topCvsDir()->getDisabledList();
  QStringList::iterator it = list->find( filename);
  if (it != list->end()) {
    return true;
  } else {
    return false;
  }
}

void Directory::setAllOpen( ) {

        Directory * D = this;
        do {
          D->setOpen( TRUE );
          if( D->depth() == 0 ) 
            // root item 
            break;
          D = (Directory *)D->parent();
        } while ( 1 );
}

void Directory::expandAllDir() {

  bool update;
  if (update = m_topView->isUpdatesEnabled()) {
    m_topView->setUpdatesEnabled(FALSE);
  }

  if (!isOpen()) {
    setOpen(true);
  }
  Directory *myChild = (Directory *)firstChild();
  while(myChild) {
    myChild->expandAllDir();
    myChild = (Directory *)myChild->nextSibling();
  }

  if (update) {
    m_topView->setUpdatesEnabled(TRUE);
    m_topView->triggerUpdate();
  }
}

void Directory::collapsAllDir() {
  Directory *myChild = (Directory *)firstChild();
  while(myChild) {
    myChild->collapsAllDir();
    myChild = (Directory *)myChild->nextSibling();
  }
  if (isOpen()) {
    setOpen(false);
  }
}

int Directory::isTreeEmpty() {

  if (getCVSEntries()->count()) return 0;

  // seems to contain NO regular CVS files
  
  Directory * C = (Directory *)firstChild();
  while( C ) {
    if( ! C->isTreeEmpty() ) 
      return 0;
    C = (Directory *)C->nextSibling();
  }
  // no entry found
  return 1;
}


bool Directory::parseCvsCallResult(CvsBuffer *output, int icmd, QStringList *stringList) {

  bool retVal = TRUE;
  m_lastErrorMsg = "";

  Directory *pCurCvsDir = NULL;
  Directory *pCurCvsLastUpdateDir = NULL;

  QStringList curCvsUpdateFileList;
  QStringList addedDirList;
  QString curCvsFileName;
  EntryStates locCurCvsState = ES_unknown ;
  unsigned int len = (*output).numLines();
  int pos = 0;
  EntryStates_E stateToSet = ES_unknown;//init to prevent compiler warning
  QString tmp, fullPath, prevFullPath, path, repository;
  prevFullPath = "";
  fullPath = "";
   
  QString line;
  for (unsigned int i = 0; i<len;i++) {
    line = (*output).textLine(i);
    if (line.isEmpty()) continue;

    switch(icmd) {
       case COMMIT_CMD:
	 if(line.find("Removing ") == 0) {// commit after remove
	   curCvsFileName = line.mid(9, line.length() - 10);   //truncate ";"
	   stateToSet=ES_unknown; // we have to set a state to adapt the warnlevel
	 }
	 else if(line.find("Checking in ") == 0) {//followed of filename
	   curCvsFileName = line.mid(12, line.length() - 13);   //truncate ";"
	   stateToSet=ES_up_to_date;
	 }
	 else if(line.compare("done") == 0){//commit completed successfully
	   if(pCurCvsDir && curCvsFileName.length()) {
	     pCurCvsDir->setEntry(curCvsFileName, stateToSet);
	     if (stateToSet==ES_unknown) pCurCvsDir->removeEntry(curCvsFileName); // must not appear in entries any more
	   }
	   curCvsFileName="";
	   break;
	 }
	 else continue; //nothing found
	 pos = curCvsFileName.findRev("/");
	 if (pos > -1) { //located in subdir
	   fullPath = m_fullName + "/" + curCvsFileName.mid(0, pos);
	   removeDoubleSlashes(fullPath);
	   curCvsFileName = curCvsFileName.mid(pos + 1);
	   if(fullPath.compare(prevFullPath)) { // change subdir
	     pCurCvsDir = searchDirOfPath(fullPath);
	     prevFullPath = fullPath;
	   }
	 } else {
	   pCurCvsDir = this;   //no subdir
	   fullPath = "";
	   prevFullPath = "";
	 }
	 break;
       case STATUS_CMD:
	 if((pos = line.find("Examining ")) > -1) {
	   path = line.mid(pos + 10);
	   path = path.stripWhiteSpace();
	   
	   //is a subdir?
	   if (path.compare(".") == 0) pCurCvsDir = this;
	   else {
	     fullPath = m_fullName + "/" + path;
	     removeDoubleSlashes(fullPath);
	   
	     //locate dir
	     pCurCvsDir = searchDirOfPath(fullPath);//search only for non-virtual dirs
	   }
	 } else if(line.find("File:") == 0){//status output of next file begins
	   curCvsFileName = line.mid(6, line.find("\t", 6) - 6);
	   curCvsFileName = curCvsFileName.stripWhiteSpace();
	   if(curCvsFileName.find("no File") == 0) {
	     curCvsFileName = curCvsFileName.mid(8);
	   }
	   
	   if(line.find("Up-to-date") > -1){
	     locCurCvsState = ES_up_to_date;
	   } else if(line.find("Locally Modified") > -1){
	     locCurCvsState = ES_modified;
	   } else if(line.find("Needs Patch") > -1){
	     locCurCvsState = ES_needs_patch;
	   } else if(line.find("Needs Merge") > -1){
	     locCurCvsState = ES_needs_merge;
	   } else if(line.find("Needs Checkout") > -1){
	     locCurCvsState = ES_needs_checkout;
	   } else if(line.find("File had conflicts on merge") > -1){
	     locCurCvsState = ES_conflict;
	   } else if(line.find("Locally Added") > -1){
	     locCurCvsState = ES_added;
	   } else if(line.find("Locally Removed") > -1){
	     locCurCvsState = ES_removed;
	   }

	   //set state
	   if(pCurCvsDir) {
	     pCurCvsDir->setAndAppendEntry(curCvsFileName, locCurCvsState);
	   }

	 }
	 break;
       case STATUS_FILES_CMD:
	 if(line.find("File:") == 0){//status output of next file begins
	   curCvsFileName = line.mid(6, line.find("\t", 6) - 6);
	   curCvsFileName = curCvsFileName.stripWhiteSpace();
	   if(curCvsFileName.find("no File") == 0) {
	     curCvsFileName = curCvsFileName.mid(8);
	   }
	   
	   if(line.find("Up-to-date") > -1){
	     locCurCvsState = ES_up_to_date;
	   } else if(line.find("Locally Modified") > -1){
	     locCurCvsState = ES_modified;
	   } else if(line.find("Needs Patch") > -1){
	     locCurCvsState = ES_needs_patch;
	   } else if(line.find("Needs Merge") > -1){
	     locCurCvsState = ES_needs_merge;
	   } else if(line.find("Needs Checkout") > -1){
	     locCurCvsState = ES_needs_checkout;
	   } else if(line.find("File had conflicts on merge") > -1){
	     locCurCvsState = ES_conflict;
	   } else if(line.find("Locally Added") > -1){
	     locCurCvsState = ES_added;
	   } else if(line.find("Locally Removed") > -1){
	     locCurCvsState = ES_removed;
	   }
	 } else if(line.find("Repository revision:") > -1) {//repository of fileName
	   //repository extrahieren
	   if((m_repository.find(".")) == 0 && (m_repository.length() == 1)) {// . detected
	     pos = line.find(m_rootDir);
	   }
	   else {
	     pos = line.find(m_rootDir + "/" + m_repository);
	   }
	   repository = line.mid(pos, line.findRev("/") - pos);
	   if(repository.compare(m_curCvsRepository)) {//different dir than lasttime?
	     //zugehoeriges dir finden
	     pCurCvsDir = searchDirOfRepository(repository);
	     if (!pCurCvsDir) { // check for dead trunk
	       repository = repository.remove(repository.find("/Attic"),6);
	       pCurCvsDir = searchDirOfRepository(repository);
	     }
	     m_curCvsRepository = repository;
	   } else { //not a subdir
	     if (!pCurCvsDir) {//NULL on first loop
	       pCurCvsDir = this;
	     }
	   }
	   //set state
	   if(pCurCvsDir) {
	     pCurCvsDir->setAndAppendEntry(curCvsFileName, locCurCvsState);
	   }
	 }
	 break; 
       case GET_TAG_INFO_PROJECT_CMD:
       case GET_TAG_INFO_FILES_CMD:
	 if (pos == 0) {
	   if (line.find("Existing Tags:") > -1) pos = 1;
	 }
	 else if ( (line.find("No Tags Exist") > -1)
		   || (line.startsWith("cvs "))
		   || (line.startsWith("===================================================================")) ) {
	   pos = 0;
	 } else {
	   QRegExp rx( "(\\S+)(?:\\s*\\()(\\w+):" );
	   rx.search(line);
	   //qDebug("cap(0): ->"+rx.cap(0)+"<- cap(1): ->"+rx.cap(1)+"<- cap(2): ->"+rx.cap(2)+"<-");
	   QString txt;
	   if (rx.cap(2).startsWith("branch")) {
	     txt = "B: "+rx.cap(1);
	   } else if (rx.cap(2).startsWith("revision")) {
	     txt = "T: "+rx.cap(1);
	   } else {
	     qDebug("unknown line on GET_TAG_INFO: "+line);
	     continue;
	   }
	   if (stringList->find(txt) == stringList->end()) {
	     stringList->append(txt);
	   }
	 }
	 break;
       case UPDATE_DIR_CMD:

	 //updating of dir ... welches verzeichnis?
	 if((pos = line.find("Updating ")) > -1) {
	   //cvs changed dir, put last scanned dir to up-to-date or remove if not physically available
	   if (pCurCvsLastUpdateDir) {
	     QDir dir(pCurCvsLastUpdateDir->fullName());
	     if (dir.exists()) {
	       pCurCvsLastUpdateDir->setFilesToUpToDate(curCvsUpdateFileList);
	     } else {
	       ((Directory*)pCurCvsLastUpdateDir->parent())->removeChild(pCurCvsLastUpdateDir->fullName());
	     }
	   }

	   path = line.mid(pos + 9);
	   path = path.stripWhiteSpace();
	   
	   //is a subdir?
	   if (path.compare(".") == 0) fullPath = m_fullName;
	   else fullPath = m_fullName + "/" + path;
	   removeDoubleSlashes(fullPath);
	   
	   //locate dir
	   pCurCvsLastUpdateDir = searchDirOfPath(fullPath);//search only for non-virtual dirs

	   if (!pCurCvsLastUpdateDir) {//not scanned or just newly added?
	     addedDirList.append(fullPath);
	   }
	   
	   //fill filelist with entries of located dir
	   curCvsUpdateFileList.clear();   
	   if (pCurCvsLastUpdateDir) pCurCvsLastUpdateDir->fillFileListWithEntries(curCvsUpdateFileList);
	 }

	 locCurCvsState = ES_unknown;
	 //filestate...
	 if((line.find("P ") == 0) 
	    || (line.find("U ") == 0)){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_up_to_date;
	 } else if(line.find("M ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_modified;
	 } else if(line.find("C ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_conflict;
	 } else if(line.find("A ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_added;
	 } else if(line.find("R ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_removed;
	 } else if(line.find("Merging differences between ") == 0){
	   curCvsFileName = line.mid(line.find("into ")+5);
	   locCurCvsState = ES_modified;
	 } else if (line.find("warning: conflicts during merge") > -1) {//filename allready extracted at <Merging differences between ...>
	   locCurCvsState = ES_conflict;
	 } else {
	   int pos1,pos2;
	   if (((pos1=line.find("warning: ")) > -1) && ((pos2=line.find(" is not (any longer) pertinent")) > -1)) {
	     curCvsFileName = line.mid(pos1+9,pos2-pos1-9);
	   } else if (((pos1=line.find("warning: new-born ")) > -1) && ((pos2=line.find(" has disappeared")) > -1)) {
	     curCvsFileName = line.mid(pos1+18,pos2-pos1-18);
	   } else if (((pos1=line.find("conflict: ")) > -1) && ((pos2=line.find(" but no longer in the repository")) > -1)) {
	     int eofileName = line.findRev(" is ",pos2-1);
	     QString state = line.mid(eofileName,pos2-eofileName);
	     curCvsFileName = line.mid(pos1+10,eofileName-pos1-10);
	     m_lastErrorMsg += line+"\n";
	     retVal = FALSE;
	     i++;//this is dirty but easier than using a separate check for skipping the next line.
	         //the next line needs to be skipped to not parse the conflict info given by cvs since it differs from CVS/Entries
	     if (state == " is modified") {//removing the entry doesn't matter since CVSEntries will be reread anyway
	       locCurCvsState = ES_modified;
	     } else {//maybe other entries, didn't find any jet
	       break;
	     }
	   } else {
	     break;
	   }
	   if((pos1 = curCvsFileName.findRev("/")) > -1) //is a subdir?
	     curCvsFileName = curCvsFileName.mid(pos1 + 1);
	   if(pCurCvsLastUpdateDir) {
	     pCurCvsLastUpdateDir->setAndAppendEntry(curCvsFileName, locCurCvsState);//we have to set a state to adapt the warnlevel
	     pCurCvsLastUpdateDir->removeEntry(curCvsFileName);
	     curCvsUpdateFileList.remove(curCvsFileName);
	   }
	   break;
	 }
	 
	 if(locCurCvsState != ES_unknown) {
	   curCvsFileName = curCvsFileName.stripWhiteSpace();
	   
	   //is file located in subdir?
	   if((pos = curCvsFileName.findRev("/")) > -1) 
	     curCvsFileName = curCvsFileName.mid(pos + 1);
	   
	   if(pCurCvsLastUpdateDir) {
	     pCurCvsLastUpdateDir->setAndAppendEntry(curCvsFileName, locCurCvsState);
	     curCvsUpdateFileList.remove(curCvsFileName);
	   }
	 }      
	 break;
       case UPDATE_FILES_CMD:

	 locCurCvsState = ES_unknown;
	 //filestate...
	 if((line.find("P ") == 0) 
	    || (line.find("U ") == 0)){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_up_to_date;
	 } else if(line.find("M ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_modified;
	 } else if(line.find("C ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_conflict;
	 } else if(line.find("A ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_added;
	 } else if(line.find("R ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_removed;
	 } else if(line.find("Merging differences between ") == 0){
	   curCvsFileName = line.mid(line.find("into ")+5);
	   locCurCvsState = ES_modified;
	 } else if (line.find("warning: conflicts during merge") > -1) {//filename allready extracted at <Merging differences between ...>
	   locCurCvsState = ES_conflict;
	 } else {
	   int pos1,pos2;
	   if (((pos1=line.find("warning: ")) > -1) && ((pos2=line.find(" is not (any longer) pertinent")) > -1)) {
	     curCvsFileName = line.mid(pos1+9,pos2-pos1-9);
	   } else if (((pos1=line.find("warning: new-born ")) > -1) && ((pos2=line.find(" has disappeared")) > -1)) {
	     curCvsFileName = line.mid(pos1+18,pos2-pos1-18);
	   } else if (((pos1=line.find("conflict: ")) > -1) && ((pos2=line.find(" but no longer in the repository")) > -1)) {
	     int eofileName = line.findRev(" is ",pos2-1);
	     QString state = line.mid(eofileName,pos2-eofileName);
	     curCvsFileName = line.mid(pos1+10,eofileName-pos1-10);
	     m_lastErrorMsg += line+"\n";
	     retVal = FALSE;
	     i++;//this is dirty but easier than using a separate check for skipping the next line.
	         //the next line needs to be skipped to not parse the conflict info given by cvs since it differs from CVS/Entries
	     if (state == " is modified") {//removing the entry doesn't matter since CVSEntries will be reread anyway
	       locCurCvsState = ES_modified;
	     } else {//maybe other entries, didn't find any jet
	       break;
	     }
	   } else {
	     break;
	   }
	   setAndAppendEntry(curCvsFileName, locCurCvsState);//we have to set a state to adapt the warnlevel
	   removeEntry(curCvsFileName);
	   break;
	 }
	 
	 if(locCurCvsState != ES_unknown ) {
	   curCvsFileName = curCvsFileName.stripWhiteSpace();
	   setAndAppendEntry(curCvsFileName, locCurCvsState);
	 }
	 
	 break;
       case QUERY_UPDATE_ALL_CMD:
       case QUERY_UPDATE_CMD:

	 locCurCvsState = ES_unknown;
	 if((pos = line.find("Updating ")) > -1) {
	   path = line.mid(pos + 9);
	   path = path.stripWhiteSpace();
	   //is a subdir?
	   if (path.compare(".") == 0) fullPath = m_fullName;
	   else fullPath = m_fullName + "/" + path;
	   removeDoubleSlashes(fullPath);
	   //locate dir
	   pCurCvsDir = searchDirOfPath(fullPath,VIRTUAL);

	   if (!pCurCvsDir) {
	     Directory * tmpDir = searchLastValidDirOfPath(fullPath);
	     if (tmpDir) addDir(fullPath, VIRTUAL);//add as virtual
	     pCurCvsDir = searchDirOfPath(fullPath,VIRTUAL);//let's see if it's there
	   }
	   //put all files of dir into list
	   //and don't clear list because buggy cvs output returns filenames in wrong order
	   //and so we set untouched files up-to-date after we parsed the whole output
	   if (pCurCvsDir) {
	     pCurCvsDir->fillFileListWithFullNameEntries(curCvsUpdateFileList);
	   }
	   break;
	 }
	 
	 //filestate... differs from UPDATE_DIR_CMD 'cause it's just a query
	 else if((line.find("M ") == 0)){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_modified;
	 } else if((line.find("U ") == 0)){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_needs_checkout;
	 } else if((line.find("P ") == 0)){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_needs_patch;
	 } else if(line.find("C ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_needs_merge;
	 } else if(line.find("A ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_added;
	 } else if(line.find("R ") == 0){
	   curCvsFileName = line.mid(2);
	   locCurCvsState = ES_removed;
	 } else if(line.find("Merging differences between ") == 0){//special case here so we break after handling it
	   curCvsFileName = line.mid(line.find("into ")+5);
	   if(pCurCvsDir) { //subdir
	     curCvsUpdateFileList.remove(fullPath+"/"+curCvsFileName);
	     pCurCvsDir->setAndAppendEntry(curCvsFileName, ES_needs_merge);
	   } else {
	     curCvsUpdateFileList.remove(m_fullName+"/"+curCvsFileName);
	     setAndAppendEntry(curCvsFileName, ES_needs_merge);
	   }
	   i++;//this is dirty but easier than using a separate check for skipping the next line.
	       //the next line needs to be skipped to not parse the file modified info given by cvs since it would hide the ES_needs_merge
	   break;
	 } else {
	   int pos1,pos2;
	   if (((pos1=line.find("warning: ")) > -1) && ((pos2=line.find(" is not (any longer) pertinent")) > -1)) {
	     line = line.mid(pos1+7,pos2-pos1-7); //modify line to not need double code
	     curCvsFileName = line.mid(2);
	     locCurCvsState = ES_needs_checkout;
	   } else if ( ((pos1=line.find("New directory `")) > -1) && ((pos2=line.find("' -- ignored" )) > -1) ) {
	     pos1 += 15;
	     curCvsFileName = line.mid(pos1,pos2-pos1);
	     locCurCvsState = ES_missing_dir;
	   }
	 }
	 
	 if(locCurCvsState != ES_unknown) {
	   curCvsFileName = curCvsFileName.stripWhiteSpace();

	   curCvsUpdateFileList.remove(m_fullName+"/"+curCvsFileName); //don't set to up-to-date

	   //now set new state
	   //is file located in subdir?
	   if((pos = curCvsFileName.findRev("/")) > -1) {
	     path = curCvsFileName.left(pos);
	     path = path.stripWhiteSpace();
	     curCvsFileName = curCvsFileName.mid(pos + 1);
	     //is a subdir?
	     fullPath = m_fullName + "/" + path;
	     removeDoubleSlashes(fullPath);
	     //locate dir
	     pCurCvsDir = searchDirOfPath(fullPath,VIRTUAL);
	     if(pCurCvsDir) { //subdir
	       switch (locCurCvsState) {
		 case ES_needs_checkout: {
		   //if file doesn't exists in entries, we have to add it to be displayable in cvs-views
		   pCurCvsDir->setAndAppendTmpEntryQueryUpdate(curCvsFileName);
		   pCurCvsDir->setAndAppendEntry(curCvsFileName, locCurCvsState);
		   break;
		 }
		 case ES_missing_dir: {
		   if (CvsOptions::g_bShowVirtualInFiletab) {
		     //if file doesn't exists in entries, we have to add it to be displayable in cvs-views
		     if (pCurCvsDir->setAndAppendTmpEntryQueryUpdate(curCvsFileName)) {
		       pCurCvsDir->setAndAppendEntry(curCvsFileName, locCurCvsState);
		     }
		   } else {
		   if (!searchDirOfPath(fullPath+"/"+curCvsFileName, VIRTUAL)) {//don't ever add twice
		     addDir(fullPath+"/"+curCvsFileName, VIRTUAL);//add as virtual
		   }
		   }
		   break;
		 }
		 default: {
		   pCurCvsDir->setAndAppendEntry(curCvsFileName, locCurCvsState);
		   break;
		 }
	       }
	     }
	   }
	   else {
	     switch (locCurCvsState) {
	       case ES_needs_checkout: {
		 //if file doesn't exists in entries, we have to add it to be displayable in cvs-views
		 setAndAppendTmpEntryQueryUpdate(curCvsFileName);
		 setAndAppendEntry(curCvsFileName, locCurCvsState); // local dir
		 break;
	       }
	       case ES_missing_dir: {
		 if (CvsOptions::g_bShowVirtualInFiletab) {
		   //if file doesn't exists in entries, we have to add it to be displayable in cvs-views
		   if (setAndAppendTmpEntryQueryUpdate(curCvsFileName)) {
		     setAndAppendEntry(curCvsFileName, locCurCvsState); // local dir
		   }
		 } else {
		 if (!searchDirOfPath(m_fullName+"/"+curCvsFileName, VIRTUAL)) {//don't ever add twice
		     addDir(m_fullName+"/"+curCvsFileName, VIRTUAL);//add as virtual
		   }
		 }
		 break;
	       }
	       default: {
		 setAndAppendEntry(curCvsFileName, locCurCvsState); // local dir
		 break;
	       }
	     }
	   }
	 }
	 break;
       case ADD_CMD:
	 if (stringList) {
	   for(unsigned int i = 0; i < (*stringList).count(); i++) {
	     QString fileName = (*stringList)[i];
	     fileName = fileName.stripWhiteSpace();
	     QString MsgToSearch = "scheduling file `" + fileName + "' for addition";
	     if(line.find(MsgToSearch) >= 0){
	       setAndAppendEntry(fileName, ES_added);
	     }
	   }
	 }
	 break;
       case REMOVE_CMD:
	 if (stringList) {
	   for(unsigned int i = 0; i < (*stringList).count(); i++) {
	     QString fileName = (*stringList)[i];
	     fileName = fileName.stripWhiteSpace();
	     QString MsgToSearch = "scheduling `" + fileName + "' for removal";
	     if(line.find(MsgToSearch) >= 0){
	       setAndAppendEntry(fileName, ES_removed);
	     }
	   }
	 }
	 break;
       case REMOVEDIR_CMD:
	 //removing of dir ...
	 if((pos = line.find("Removing ")) > -1) {
	   path = line.mid(pos + 9);
	   path = path.stripWhiteSpace();

	   //is a subdir?
	   if (path.compare(".") == 0) fullPath = m_fullName;
	   else fullPath = m_fullName + "/" + path;
	   removeDoubleSlashes(fullPath);
	   //locate dir
	   pCurCvsLastUpdateDir = searchDirOfPath(fullPath);
	 } else if ( (pos=line.find("scheduling `")) > -1) {
	   pos += 12;
	   int endPos;
	   if ( (endPos=line.find("' for removal",pos)) > -1) {
	     QString curCvsFileName = line.mid(pos,endPos-pos);
	     //is file located in subdir?
	     if((pos = curCvsFileName.findRev("/")) > -1) {
	       curCvsFileName = curCvsFileName.mid(pos + 1);
	     }
	     if(pCurCvsLastUpdateDir) {
	       pCurCvsLastUpdateDir->setAndAppendEntry(curCvsFileName, ES_removed);
	     }
	   }
	 }
	 break;
       case MERGE_REV_INTO_CMD:
	 if(line.startsWith("Merging differences between ")){//get the filename
	   curCvsFileName = line.mid(line.find("into",28)+5);
	   curCvsFileName = curCvsFileName.stripWhiteSpace();
	 } else	if (line.find("rcsmerge: warning: conflicts during merge")==0) {//finished, set the state
	   setAndAppendEntry(curCvsFileName, ES_conflict);
	 }
	 break;
       case RESURRECT_CMD:
	 if(line.find("U ")==0){//get the filename
	   curCvsFileName = line.mid(2);
	   curCvsFileName = curCvsFileName.stripWhiteSpace();
	 } else	if (line.findRev("resurrected")>=0) {//finished, set the state
	   setAndAppendEntry(curCvsFileName, ES_up_to_date);
	 }
	 break;
    }
  }
  //some things have to be done when parsing has finished
  switch(icmd) {
     case QUERY_UPDATE_ALL_CMD:
     case QUERY_UPDATE_CMD: { //set all file not mentioned to up-to-date
       QString tmp,file;
       while (!curCvsUpdateFileList.isEmpty()) {
	 file = tmp = curCvsUpdateFileList.first();
	 //is file located in subdir?
	 if((pos = tmp.findRev("/")) > -1) {
	   path = tmp.left(pos);
	   path = path.stripWhiteSpace();
	   tmp = tmp.mid(pos + 1);
	   removeDoubleSlashes(path);
	   //locate dir
	   pCurCvsDir = searchDirOfPath(path);
	   if(pCurCvsDir) { //subdir
	     pCurCvsDir->setAndAppendEntry(tmp, ES_up_to_date);
	   }
	 }
	 else {
	   setAndAppendEntry(tmp, ES_up_to_date); // local dir
	 }
	 curCvsUpdateFileList.remove(file);
       }
       break;
     }
     case UPDATE_DIR_CMD: { //we didn't set states to up-to-date for the finally scanned dir yet
       if (pCurCvsLastUpdateDir) {
	 QDir dir(pCurCvsLastUpdateDir->fullName());
	 if (dir.exists()) {
	   pCurCvsLastUpdateDir->setFilesToUpToDate(curCvsUpdateFileList);
	 } else {
	   ((Directory*)pCurCvsLastUpdateDir->parent())->removeChild(pCurCvsLastUpdateDir->fullName());
	 }
       }

       while (!addedDirList.isEmpty()) {//add new dirs, all in up to date state since they are just updated

	 QString dirName = addedDirList.first();
	 Directory * tmpDir = searchLastValidDirOfPath(dirName);
	 QString newDir = dirName.left(dirName.find("/",tmpDir->fullName().length()+1));

	 if (tmpDir->fullName().compare(newDir)) {//only add once
	   //check if exists, don't know wether cvs runns in 'prune empty dirs' mode
	   QFileInfo info(newDir);
	   if (info.exists()) {
	     
	     if (tmpDir->m_isAnalyzedDir) {//don't continue if not analyzed in on-the-fly-mode
	       Directory * unanalyzedDir = tmpDir->addDir(newDir);//add in case it was just updated
	       if (unanalyzedDir && tmpDir->isOpen()) {//will not be analyzed on open() because it is allready open
		 unanalyzedDir->analyzeDirs(false);
	       }
	     }
	   }
	 }
	 addedDirList.remove(dirName);
       }
       break;
     }
     default: {
       break;
     }
  }
  return retVal;
}

void Directory::afterCall( int, CvsBuffer*, bool) {

}

bool Directory::hasDisabled() {
  if (m_disabled) return TRUE;
  Directory * C = (Directory *)firstChild();
  while( C ) {
    if (C->hasDisabled()) {
      return TRUE;
    }
    C = (Directory *)C->nextSibling();
  }
  return FALSE;
}

/**
 * Executes a self contained command on the directory structure beginning
 * with the current item.  The target item (and NOT the command) is
 * responsible for making sure that recursive commands are applied to all
 * child items.
 *
 * @param	CvsCommand
 *	command to execute of the directory structure 
 * @return
 *	true on success.  Otherwise false.
 */

bool Directory::execute(CCvsCommand &cmd)
{
  if (m_disabled) return FALSE;
  if (!cmd.isRecursive()) {
    return cmd.execute(*this);
  } else {
    if (ONTHEFLYSCANNING) analyzeAll();
    return recExecute(cmd);
  }
}


/**
 * Undo the changes made by previously executing the given command. 
 *
 * @param	CvsCommand
 *	command to undo
 * @return
 *	true on success.  Otherwise false.
 */

bool Directory::unexecute(CCvsCommand &cmd)
{
  if (m_disabled) return FALSE;
  if (!cmd.isRecursive()) {
    return cmd.unexecute(*this);
  } else {
    if (ONTHEFLYSCANNING) analyzeAll();
    return recUnexecute(cmd);
  }
}

bool Directory::recExecute(CCvsCommand &cmd) {
  bool retVal = TRUE;
  Directory * C = (Directory *)firstChild();
  while( C ) {
    if (!C->m_disabled) {
      if (!C->recExecute(cmd)) retVal = FALSE;
    } else {
      retVal = FALSE;
    }
    C = (Directory *)C->nextSibling();
  }
  if (!cmd.execute(*this)) retVal = FALSE;
  return retVal;
}

bool Directory::recUnexecute(CCvsCommand &cmd) {
  bool retVal = TRUE;
  Directory * C = (Directory *)firstChild();
  while( C ) {
    if (!C->m_disabled) {
      if (!C->recUnexecute(cmd)) retVal = FALSE;
    } else {
      retVal = FALSE;
    }
    C = (Directory *)C->nextSibling();
  }
  if (!cmd.unexecute(*this)) retVal = FALSE;
  return retVal;
}

void Directory::setStatusText( const QString & txt, int ms) {
  ((CvsDirListView*)m_topView)->setStatusBarText( txt, ms);
}

//--------------------------------------------------------------------------------------------

//return QString* of files to ignore, taken from hardcoded presets plus:
//$HOME/.cvsignore plus $CVSIGNORE
//lastChecked: when this dir's ignores were last checked for changes
//changed: set to true if there was a change detected (so the dir's file cache needs a refresh)
const QString Directory::getHomeFilesToIgnore(QDateTime& lastChecked, bool& changed) {

  static QString ignoreString;
  static QDateTime modTimeOfIgnores;
  static bool noIgnores = FALSE;

  changed = TRUE;
  QFile cvsignore;
  cvsignore.setName(QDir::homeDirPath() + "/.cvsignore");
  if (cvsignore.exists()) {
    QDateTime lastModTimeOfIgnores = QFileInfo(cvsignore).lastModified();
    if ( (!modTimeOfIgnores.isValid()) || (modTimeOfIgnores != lastModTimeOfIgnores)) {
      ignoreString = "";
      QString ign_line;
      if(cvsignore.open(IO_ReadOnly)) {
	QTextStream ignoreStream(&cvsignore); 
	while(!ignoreStream.eof()) {
	  ign_line = ignoreStream.readLine();
	  ignoreString += " " + ign_line.stripWhiteSpace();
	}
	cvsignore.close();
      }
      ignoreString = ignoreString.stripWhiteSpace();
      modTimeOfIgnores = lastModTimeOfIgnores;
      lastChecked = lastModTimeOfIgnores;
    } else if ( lastChecked.isValid() && (lastChecked >= lastModTimeOfIgnores) ) {
      changed = FALSE;
      return ignoreString;
    } else {
      lastChecked = lastModTimeOfIgnores;
      return ignoreString;
    }
    noIgnores = FALSE;
  } else if (!noIgnores) {
    noIgnores = TRUE;
  } else {
    changed = FALSE;
    return ignoreString;
  }

  QString retval = CVSIGNOREFILES + " " + ignoreString;
  retval = retval.stripWhiteSpace();
  retval += " " + QString(getenv( "CVSIGNORE")).simplifyWhiteSpace();
  int pos;
  if ( (pos = retval.findRev("!")) > -1) {
    ignoreString = retval.mid(pos + 1);
    ignoreString = ignoreString.stripWhiteSpace();
    return ignoreString;
  }

  //use reverse order to encrease parsing speed
  ignoreString += " " + QString(getenv( "CVSIGNORE")).stripWhiteSpace() + " " + CVSIGNOREFILES;
  ignoreString = ignoreString.stripWhiteSpace();
  return ignoreString;
}

//--------------------------------------------------------------------------------------------

//return QString* of files to ignore, taken from per dir .cvsignore
//homeIgnores: QString* holding the files to ignore from getHomeFilesToIgnore(QDateTime& lastChecked, bool& changed)
//changed: set to true if there was a change detected (so the dir's file cache needs a refresh)
const QString Directory::getDirFilesToIgnore(const QString homeIgnores, bool& changed) {

  changed = FALSE;
  QFile cvsignore;
  cvsignore.setName(m_fullName + "/.cvsignore");
  if (cvsignore.exists()) {
    QDateTime lastModTimeOfIgnores = QFileInfo(cvsignore).lastModified();
    m_checkedForIgnoresFile = FALSE;
    if ( (!m_lastTimeDirIgnoresChanged.isValid()) || (m_lastTimeDirIgnoresChanged != lastModTimeOfIgnores) ) {
      m_pDirIgnores = "";
      if(cvsignore.open(IO_ReadOnly)) {//read per dir .cvsignore
	QString ign_line;
	QTextStream ignoreStream(&cvsignore); 
	while(!ignoreStream.eof()) {
	  ign_line = ignoreStream.readLine();
	  m_pDirIgnores = m_pDirIgnores + " " + ign_line.stripWhiteSpace();
	}
	cvsignore.close();
      }
      m_pDirIgnores = m_pDirIgnores.stripWhiteSpace();
      m_lastTimeDirIgnoresChanged = lastModTimeOfIgnores;
      changed = TRUE;
    } else {
      return m_pDirIgnores;
    }
  } else if (!m_checkedForIgnoresFile) {
    m_checkedForIgnoresFile = TRUE;
    changed = TRUE;
  } else {
    return m_pDirIgnores;
  }
  
  int pos;
  if ( (pos = m_pDirIgnores.findRev("!")) > -1) {
    m_pDirIgnores = m_pDirIgnores.mid(pos + 1);
  } else {
    //reverse order to increase parsing speed
    m_pDirIgnores = m_pDirIgnores + " " + homeIgnores;
  }
  m_pDirIgnores = m_pDirIgnores.stripWhiteSpace();
  return m_pDirIgnores;
}


// ------------------------ functions telling about cvs file state -------------------------------

bool Directory::isBinary(QString fileName) {
  QDict<CVSEntries> *cvsEntries = getCVSEntries();
  CVSEntries *cvsEntry = cvsEntries->find(fileName);
  if (cvsEntry) {
    return cvsEntry->option.b;
  } else return FALSE;
}

//--------------------------------------------------------------------------------------------

bool Directory::isCvsRegistered(QString fileName) {
  QDict<CVSEntries> *cvsEntries = getCVSEntries();
  CVSEntries *cvsEntry = cvsEntries->find(fileName);
  if (cvsEntry) {
    return TRUE;
  } else return FALSE;
}
