#include "lpapp.h"

#include "absplayer.h"
//platform dependant
#ifdef Q_WS_X11
    #include "mpg123.h"
    #include "xmms.h"
#endif
#ifdef Q_OS_MACX
    #include "itunes.h"
#endif

#ifdef _WS_WIN_
    #include "winamp.h"
#else
#endif

#include "lpfactory.h"
#include "lpsettings.h"
#include "lplistener.h"
#include "lplayer.h"
#include "lpdatabase.h"
#include "lpdatabaseiterator.h"
#include "compositeplaylist.h"
#include "directoryplaylist.h"
#include "m3uplaylist.h"
#include "playlistreader.h"
#include "lpremotegeneric.h"
#include "queue.h"
#include "autoqueue.h"
#include "threadedcommand.h"
#include "queuebuilder.h"

IntSetting LPApp::SettingAutoQueue("main", "auto-queue",1);
IntSetting LPApp::SettingQueueOnStart("main", "queueonstart",0);

/* LPApp is the "reference" LongPlayer application using lplayer-lib
 * it supports:
 * - a number of media players
 * - auto-random-queue
 * - user browsing and rating (todo)
 */
LPApp::LPApp(LPSettings* settings) :  _quitting(false), _settings(settings), _lplayer(0), _queuer(0)  {
	_commands = vector<ThreadedCommand*>();

#ifdef _WS_WIN_
	absPlayer::setPlayer(new Winamp());
#endif
	
#ifdef Q_OS_MACX
	absPlayer::setPlayer(new ITunes());
#endif

#ifdef Q_WS_X11
	if (getSettings()->getISetting("main","nogui"))
		absPlayer::setPlayer(new MPG123());
	else
		absPlayer::setPlayer(new XMMS());
#endif
	absPlayer::getPlayer()->setSettings(getSettings());
	LPFactory* fac = new LPFactory(getSettings()); 
	LPFactory::setFactory(fac); 
	_lplayer = fac->getLPlayer();
	getLPlayer()->init();

	_remote = new LPRemoteGeneric(this);
	logDebug("LPApp::creating queuer.");
	_queuer = new ThreadedCommand(new AutoQueue(getLPlayer(), absPlayer::getPlayer()));

	//------------------------------
	
	logDebug("LPApp::subscribing for player events.");
	//connect mplayer to lplayer
    getLPlayer()->subscribe(this);
	absPlayer::getPlayer()->subscribe(getLPlayer());
    absPlayer::getPlayer()->start();
	logDebug("LPI::subscribing for player events. done.");
			
	checkFirstTime();
    if (getSettings()->getSetting(SettingQueueOnStart))
	   getSettings()->setISetting("main","play",1,false);
	
    //commandline option overrides
    if (getSettings()->testSetting("main","play") && (getSettings()->getISetting("main","play") == 1)) {   
        getSettings()->setISetting("main","play",0,false);
		((AutoQueue*)getQueuer()->getCommand())->setForce(true);
		getQueuer()->start();
	}
}

LPApp::~LPApp(){}
ThreadedCommand* LPApp::getQueuer() { return _queuer; }

void LPApp::checkFirstTime() {
    logDebug("LPlayer::Checking first-time settings...");
    if (!getSettings()->testSetting("main","init")) {
        logInfo("First Time! Welcome to LPlayer!");
        logMessage(
		"Welcome to LongPlayer!\n\n"
		"The first thing you need to do is create some playlists.\n "
		"A playlist is just a collection of directories and m3u files that LongPlayer uses to queue music from.\n "
		"You can make a playlist by choosing Playlists - Manage Playlists in the menu.\n");
    }
    getSettings()->setSetting("main", "init","ok");
}

LPRemoteGeneric* LPApp::getRemote() { return _remote; }
LPlayer* LPApp::getLPlayer() { return _lplayer; }
absPlayer* LPApp::getMPlayer() const { return absPlayer::getPlayer(); }

LPSettings* LPApp::getSettings() { return _settings; }

bool LPApp::getAutoQueue() { return getSettings()->getSetting(SettingAutoQueue); }
void LPApp::setAutoQueue(bool b) {
	getSettings()->setSetting(SettingAutoQueue,b);
	if (b && getQueuer()) { getQueuer()->start(); }
}

string LPApp::currentPlaylist() { return getSettings()->getSetting("lplayer","playlist"); }

void LPApp::changePlaylist(string name) {
	if (currentPlaylist() == name) return;
	logDebug("LPApp::changing current playlist.");
		
	//restart autoqueueing
	if (getQueuer()) getQueuer()->getCommand()->stop();
	getSettings()->setSetting("lplayer", "playlist",name);
	if (getSettings()->getSetting(SettingAutoQueue)) {
		getLPlayer()->getPlayer()->cleanQueued();
		if (getQueuer()) getQueuer()->start();
	}
	logDebug("LPApp::changing current playlist. done.");
}


void LPApp::queue(bool now) {
	getQueuer()->getCommand()->stop();
	((AutoQueue*)getQueuer()->getCommand())->setForce(now);
	getQueuer()->start();
}

void LPApp::play(vector<string> filenames,bool force) {
	vector<string> files;
	for (uint i=0;i<filenames.size();i++) {
		if (DirectoryPlaylist::isDirectory(filenames[i])) {
			DirectoryPlaylist p; p.addDir(filenames[i],true); p.startBuilding();
			vector<string>* filenames2 = p.filenames();
			for (uint j=0;filenames2 && (j<filenames2->size());j++) {
				files.push_back((*filenames2)[j]);
			}
		}
		else
		if (M3UPlaylist::isM3U(filenames[i])) {
			//XMMS won't queue m3u's, but we can
			M3UPlaylist p; p.setM3U(filenames[i]); p.startBuilding();
			vector<string>* filenames2 = p.filenames();
			for (uint j=0;filenames2 && (j<filenames2->size());j++) {
				files.push_back((*filenames2)[j]);
			}
		}
		else files.push_back(filenames[i]);
	}

	if (force)
		getLPlayer()->getPlayer()->play(files);
	else
		getLPlayer()->getPlayer()->queue(files);
}

void LPApp::search(string pattern,bool playnow) {
	CompositePlaylist* p = getLPlayer()->getPlaylists()->getPlaylist(getSettings()->getSetting("lplayer","playlist"));
	vector<string> contains = p->getFilenamesContain();
	contains.push_back(pattern);
	p->setFilenamesContain(contains);
	p->startBuilding();
	vector<string> fs = *(p->filenames());
	if (fs.size()) {
		if (playnow) getLPlayer()->getPlayer()->play(fs) ;
		else getLPlayer()->getPlayer()->queue(fs);
	}
	delete p;
}

void LPApp::savePlaylist(CompositePlaylist* p) { getLPlayer()->getPlaylists()->setPlaylist(p); }
void LPApp::removePlaylist(string name) { getLPlayer()->getPlaylists()->removePlaylist(name); }
void LPApp::DatabaseImportXML(string filename) { getLPlayer()->getDatabase()->importXML(filename); }
void LPApp::DatabaseExportXML(string filename) { getLPlayer()->getDatabase()->exportXMLTo(filename); }
Song LPApp::SongRetrieveFromFilename(string filename) { return getLPlayer()->getDatabase()->retrieveAndCorrect(filename, ""); }
Song LPApp::SongRetrieve(string key) { return getLPlayer()->getDatabase()->retrieve(key); }

void LPApp::SongStore(Song s) {	
	logDebug(string("LPApp::storing song ") + s.getKey());
	getLPlayer()->getDatabase()->store(s,false);//hard merge
	//perhaps we changed the current song
	getLPlayer()->notify();
}

Song LPApp::SongCurrent() {	return getLPlayer()->currentSong(); }
vector<string> LPApp::getPlaylists() { return getLPlayer()->getPlaylists()->getPlaylists(); }
CompositePlaylist* LPApp::getPlaylist(string name) { return getLPlayer()->getPlaylists()->getPlaylist(name); }

void LPApp::savePreferences() {
	getSettings()->save();
	getLPlayer()->getPlaylists()->save();
}

void LPApp::restorePreferences() {
	getSettings()->revert();
	getLPlayer()->getPlaylists()->revert();
}

void LPApp::notify(int /*i*/) {
	notifyListeners();
	if (!getMPlayer()->isPlaying()) return;
	if (!getSettings()->getSetting(SettingAutoQueue)) return;
	((AutoQueue*)getQueuer()->getCommand())->setForce(false);
	getQueuer()->start();
	getLPlayer()->getPlayer()->checkPlayed();
}

bool LPApp::quitting() { return _quitting; }

void LPApp::quit() {
	if (_quitting) return;
	_quitting = true;
    if (getRemote()) getRemote()->flagQuit = true;
    if (getMPlayer()) getMPlayer()->brun = false;

	if (getQueuer()) getQueuer()->getCommand()->stop();
	for (uint i=0;i<_commands.size();i++) {
		_commands[i]->getCommand()->stop();
	}
	notifyListeners();

    logDebug("LPApp::waiting for queuer thread to finish..");
    while (getQueuer() && !getQueuer()->getCommand()->done()) { getQueuer()->wait(100); }   
    logDebug("LPApp::waiting for remoteControl thread to finish..");
	// running() does not work under windows
    while (getRemote()->brunning) {   getRemote()->wait(100); }   
    logDebug("LPApp::remoteControl thread finished.");
	logDebug("LPApp::waiting for mediaplayer thread to finish..");    
	//running() does not work under windows    
    while (getMPlayer()->brunning) {   getMPlayer()->wait(100); }   
    logDebug("LPApp::mediaplayer thread finished.");

	if (getLPlayer()) getLPlayer()->quit();
    LPFactory::getFactory()->destroyLPlayer();

	getSettings()->save();
}

void LPApp::subscribe(LPListener* gui) { _listeners.push_back(gui); }

void LPApp::notifyListeners() {
	for (uint i=0;i < _listeners.size();i++) {
		_listeners[i]->notify();
	}
}

void LPApp::applyRatingToDir(string dir, int rating) {
	//let's use a playlist for this
	DirectoryPlaylist* p = new DirectoryPlaylist("temp");
	p->addDir(dir, true);
	p->startBuilding();
	vector<string>* files = p->filenames();
	doInitProgress(string("Processing ") + absUtil::its(files->size()) + " songs...",files->size());
	for (uint i=0;i<files->size();i++) {
	    doSetProgress(i);
        Song s = SongRetrieveFromFilename((*files)[i]);
		s.setLike(rating);
    	getLPlayer()->getDatabase()->store(s,false);//hard merge
	}
    doEndProgress();
	delete p;
    getLPlayer()->notify();
}

void LPApp::applyGenreToDir(string dir, int x, int y, int z) {
	//let's use a playlist for this
	DirectoryPlaylist* p = new DirectoryPlaylist("temp");
	p->addDir(dir, true);
	p->startBuilding();
	vector<string>* files = p->filenames();
	doInitProgress(string("Processing ") + absUtil::its(files->size()) + " songs...",files->size());
	for (uint i=0;i<files->size();i++) {
	    doSetProgress(i);
        Song s = SongRetrieveFromFilename((*files)[i]);
		s.setColor(x,y,z);
    	getLPlayer()->getDatabase()->store(s,false);//hard merge
	}
    doEndProgress();
	delete p;
    getLPlayer()->notify();
}

