#include "../config.h"
#include "lpdatabase.h"
#include "absutil.h"
#include "lpdatabasereader.h"
#include "lpdatabasepersistentreader.h"
#include "lpdatabasemagicreader.h"
#include "lpdatabasecolorreader.h"
//can you smell a factory somewhere?
#include "lpdatabasexmlreader.h"
#include "lpdatabasexmlwriter.h"
#include "lpdatabaseiterator.h"
#include "playlist.h"
#include "synchronizeddbaccess.h"
#include "songaccess.h"
#ifdef HAVE_LIBDB_CXX
#include "lpberkeleydbreader.h"
#endif

#include <string>
#include <iostream>

#include <qstring.h>
#include <qdir.h>
#include <qdom.h>
#include <qxml.h>
#include <qtextstream.h>
#include <qprogressdialog.h>
#include <qmap.h>

StringSetting LPDatabase::SettingLocation("database","location", LPSettings::getLPHomeDir()+"/songs.lp2");
IntSetting LPDatabase::SettingLastSize("database","size");
IntSetting LPDatabase::SettingSaveOnExit("database","saveonexit",1);

LPDatabase::LPDatabase(LPSettings* settings) {
	_settings = settings;
	_filename = getSettings()->getSetting(SettingLocation);
	_reader = 0;

//convert if necessary and load the appropriate reader
	load(_filename);
	//test();

	//test database
	logDebug("LPDatabase::checking database.");
    if (getSettings()->testSetting(SettingLastSize) && (getSize() < getSettings()->getSetting(SettingLastSize))) {
		logWarning("LPDatabase::size has reduced! " + _filename);
		logError("The database is missing songs! You may want to check/restore the file " + _filename + ".");
	}
}

void LPDatabase::test() {
	logDebug("testing..");
	absUtil::getUtil()->copyFile(_filename,"test1.mp3");
	absUtil::getUtil()->copyFile(_filename,"test2.mp3");
	string key = absUtil::getUtil()->getFileID("test1.mp3");
	Song song1; song1.setKey(absUtil::getUtil()->getFileID("test1.mp3"));
	Song song2; song2.setKey(absUtil::getUtil()->getFileID("test2.mp3"));
	song1.setFilename("test1.mp3");
	song2.setFilename("test2.mp3");
	song1.setName("song 1");
	song2.setName("song 2");
	song1.setTimes(5);
	song1.setLike(3);
	//some song equivalence functions tests
	/*
	logDebug("here we go!");
	if (!song1.equal(song2)) logWarning("not equal");
	//logDebug("1!");
	song2.setName("song 2");
	if (song1.equal(song2)) logWarning("equal");
	logDebug("2!");
	song2.setFilename("test2.mp3");
	song2.setName("song 1");
	if (!song1.equivalent(song2)) logWarning("not equivalent");
	logDebug("3!");
	*/
	//DB tests
	absUtil::getUtil()->removeFile("bla.lp2");
	SongAccess sa("bla.lp2");
	sa.store(song1);

	Song tstsng = sa.retrieve(key);
	if (!song1.equal(tstsng)) {
	 	logWarning("simple store retrieve failed");
		logDebug(song1.toString());
		logDebug(tstsng.toString());
	}

	//key clashes
	sa.store(song2);
	tstsng = sa.retrieve(key);
	if (tstsng.getName() != "song 1") {
		logWarning("original entry overwritten: store/retrieve failed");
		logDebug(song1.toString());
		logDebug(tstsng.toString());
	}
	logDebug("testing complete!");
}

LPDatabase::~LPDatabase() {
	if (getSettings()->getSetting(SettingSaveOnExit)) compact();
	if (getReader()) delete getReader();
}

void LPDatabase::load(string filename) {
	logInfo("Loading database...",0);

#ifdef HAVE_LIBDB_CXX
    _reader = new LPBerkeleyDBReader(filename);
#else
	if (getReader()) delete getReader();
	_filename = filename;
	logDebug("LPDatabase::probing database " + filename);

	QFile f(QString::fromLocal8Bit(filename.c_str()));
	if (!f.exists()) {
		_reader = new SongAccess(filename);
		_reader = new SynchronizedDBAccess(getReader());
		return;
	}
	if (!f.open(IO_ReadOnly)) {
		logWarning(string("LPDatabase::probing of database ") + filename + " failed.");
		logError(string("Could not open database file ") + filename + ". Please check file permissions and quit any processes using it.");
		return;
	}
	QTextStream t( &f );
	QString st;
	//preparatory stuff
	if (t.atEnd()) {
		//create new database, nothing to see here...
		f.close();
		_reader = new SongAccess(_filename);
		_reader = new SynchronizedDBAccess(getReader());
	}
	else {
		st = t.readLine(); //read header line and see what we have..
		f.close();
		//the default DB type
		if (st == "{{color}}") {
			_reader = new SongAccess(_filename);
			_reader = new SynchronizedDBAccess(getReader());
		}
		else
		{
			//old
			if (st == "{{persistent}}") {
				absUtil::getUtil()->copyFile(_filename, _filename + ".bak");
				absUtil::getUtil()->copyFile(_filename + ".idx", _filename + ".bak.idx");
				absUtil::getUtil()->removeFile(_filename);
				absUtil::getUtil()->removeFile(_filename + ".idx");
				LPDatabaseReader* oldreader = new LPDatabasePersistentReader(_filename + ".bak");
				_reader = new SongAccess(_filename);
				_reader = new SynchronizedDBAccess(getReader());
				LPDatabaseIterator* it = oldreader->iterator();
				it->reset();
				logDebug("converting..");
				while(!it->atEnd() ) { getReader()->store(it->next()); }
                delete it;
				delete oldreader;
				return;
			}
			//less old
			if (st == "{{magic}}") {
				absUtil::getUtil()->copyFile(_filename, _filename + ".bak");
				absUtil::getUtil()->copyFile(_filename + ".idx", _filename + ".bak.idx");
				absUtil::getUtil()->removeFile(_filename);
				absUtil::getUtil()->removeFile(_filename + ".idx");
				LPDatabaseReader* oldreader = new LPDatabaseMagicReader(_filename + ".bak");
				_reader = new SongAccess(_filename);
				_reader = new SynchronizedDBAccess(getReader());
				LPDatabaseIterator* it = oldreader->iterator();
				it->reset();
				logDebug("converting..");
				while(!it->atEnd() ) { _reader->store(it->next()); }
                delete it;
				delete oldreader;
				return;
			}
			logWarning(string("LPDatabase::don't know how to handle ") + string(st.local8Bit()));
			logError("Could not load a database: unsupported database file.");
		}
	}
#endif
    
	logDebug("LPDatabase::reading database. done.");
}

void LPDatabase::compact() {
	if (!getReader()) return;
	getReader()->compact();
	getSettings()->setSetting(SettingLastSize, getSize());
}

void LPDatabase::flush() {
	if (!getReader()) return;
	getReader()->flush();
	getSettings()->setSetting(SettingLastSize, getSize());
}

void LPDatabase::importXML(string filename) {
	logDebug("LPDatabase::importing database from XML...");
	logInfo("Importing database from XML...");
	QFile file(QString::fromLocal8Bit(filename.c_str()));
	QXmlInputSource source = QXmlInputSource(file);
	QXmlSimpleReader reader;
	LPDatabaseXMLReader* handler = new LPDatabaseXMLReader(this);
	LPDatabaseXMLReaderErrorHandler* errorHandler = new LPDatabaseXMLReaderErrorHandler();
	reader.setContentHandler(handler);
	reader.setErrorHandler(errorHandler);
	reader.parse(source);
	delete(handler);
	delete(errorHandler);
	logInfo("Database imported.");
	logDebug("LPDatabase::importing database from XML. Done.");
}

void LPDatabase::exportXMLTo(string filename) {
	logInfo("Exporting database to XML...");
	logDebug("LPDatabase::exporting database to XML...");
	LPDatabaseXMLWriter* writer = new LPDatabaseXMLWriter(filename);
	if(!writer->isOpen()){
		logError("Exporting database to XML... Error opening file");
		delete writer;
		return; //Abort! Abort! Abort!
	}
	int i=0;
	LPDatabaseIterator* it = getReader()->iterator();
	it->reset();
	doInitProgress("Exporting songs",getReader()->getSize());
	while(!it->atEnd() ) {
		doSetProgress(i++);
		Song s = it->next();
		writer->writeSong(&s);
	}
    delete it;
	writer->close();
	delete writer;
	logDebug("LPDatabase::exporting database to XML. done.");
	logInfo("Database exported.");
	doEndProgress();
}

void LPDatabase::store(Song song, bool softmerge ) {
	if (!getReader()) return;
	if (song.empty()) {
		logWarning("LPDatabase::empty song, not storing.");
		return;
	}
	Song old = retrieve(song.getKey());
	Song toStore = song;
	
	if (!old.empty()) {
		old.repair(); toStore.repair();
		if (softmerge && old.equivalent(toStore)) {
			logDebug(string("LPDatabase::updating ")  + old.getName());
			old.softmerge(toStore);
			toStore = old;
		}
	}
	getReader()->store(toStore);
}

LPSettings* LPDatabase::getSettings() const { return _settings; }
void LPDatabase::remove(Song song) { getReader()->remove(song); }

int LPDatabase::getSize() const {
	if (!getReader()) return 0;
	return getReader()->getSize();
}

Song LPDatabase::retrieveAndCorrect(string filename, string name, int length) {
	if (!getReader()) return Song();
	Song song;
	song.setFilename(filename); song.setName(name); song.setLength(length);
	song.setKey(absUtil::getUtil()->getFileID(filename));
	Song sng = getReader()->retrieve(song);
	if (sng.empty()) //it may be empty, but it contains the correct key
	{ logDebug("LPDatabase::song entry is new."); }
	//we are 99% certain that this is the song we want; so update the song with (our) newest info
	if (sng.getKey().empty()) sng.setKey(song.getKey());
	if (sng.getFilename() != filename)  sng.setFilename(filename); 
	if (length && (sng.length() != length))  sng.setLength(length);
	if ((sng.getName() != name) && !name.empty()) sng.setName(name);
	return sng;
}

bool LPDatabase::inDatabase(string filename) { return getReader()->inDatabase(filename); }

//returns Song() if not in database
Song LPDatabase::retrieve(string skey) { return getReader()->retrieve(skey); }
LPDatabaseReader* LPDatabase::getReader() const { return _reader; }

void LPDatabase::cleanDatabase(Playlist* plist) {
	logInfo(string ("Backing up database first to ") + _filename + ".bak");
	absUtil::getUtil()->copyFile(_filename,_filename + ".bak");
	logInfo("Finding invalid song entries...This may take a while...");
	//create a map with flags for all Database entries
	QMap<string,bool> dbtable;
	LPDatabaseIterator* iter = getReader()->iterator();
	iter->reset();
	Song itersong;
	while (!iter->atEnd()) {
		itersong = iter->next();
		(dbtable)[itersong.getKey()] = 0;
	}
    delete iter;
	
	//iterate over all files in the playlist...
	plist->startBuilding();
	vector<string> *list = plist->filenames();
	for (uint i=0;i< list->size();i++) {
		itersong = retrieveAndCorrect((*list)[i],"");
		(dbtable)[itersong.getKey()] = 1;
	}
	
	//now delete all entries that are not marked
	QMap<string,bool>::Iterator it;
	for (it = dbtable.begin();it != dbtable.end(); it++) {
		if (!it.data()) {
			logDebug(string("LPDatabase::removing ") +  it.key());
			remove(Song(it.key()));
		}
	}
}
