#include "lpdatabasecolorreader.h"
#include "lpglobal.h"
#include "absutil.h"
#include <qstring.h>
#include <qdir.h>
#include <qdom.h>
#include <qtextstream.h>
#include <qprogressdialog.h>
#include <string>
#include <iostream>


/** opens the associated database file and goes to the first record, leaving it in read-only modus
  * if it does not exists, it is created
  * returns false if something went wrong
  */
bool LPDatabaseColorReader::initFile(string filename)
{
	if (_origFile) delete _origFile;
    _origFile = new QFile(QString::fromLocal8Bit(filename.c_str()));
	//initialize an empty DB if necessary
	if (!_origFile->exists())
	{
    	if (!_origFile->open(IO_WriteOnly | IO_Truncate))
    	{
            logDebug("LPDatabaseColorReader::cannot save.");
            logError(string("Could not save database! Please check file permissions or free space."));
			_origFile = 0;
            return false;
   		 }
		else
		{
			QTextStream t(_origFile);
			t.setEncoding(QTextStream::UnicodeUTF8);
			t << "{{color}}\n";
			//logDebug(absUtil::its(_origFile->at()));
			setWriteMagic(_origFile);
			//logDebug(absUtil::its(_origFile->size()));
		}
	}

	resetPosition();
	return true;
}

void LPDatabaseColorReader::resetPosition()
{
	if (!_origFile) return;
	_origFile->close();
    if (!_origFile->open(IO_ReadOnly))
    {
        logWarning(string("LPDatabaseColorReader::opening ") +
				string(_origFile->name().local8Bit()) + " failed.");
		logError(string("Could not reset database file ") + string(_origFile->name().local8Bit()) + ". Please check file permissions and quit any processes using it.");
        return ;
    }
	QTextStream t( _origFile );
		t.setEncoding(QTextStream::UnicodeUTF8);
	QString str;
	if (!t.atEnd())
	{
		str = t.readLine();
		if (str != "{{color}}")
		{
			_origFile->close();
			logError("LPDatabaseColorReader::reset failed: this is not a magical color file.");
			logDebug(string(str.local8Bit()));
			return ;
		}
	}
	//don't close!

	//the right file just at the right position... (being the first magic marker..)
	setFile(_origFile);

}

//private!
void LPDatabaseColorReader::setFile(QFile* file)
{
	if (file) _file = file;
	else _file = _origFile; //reset file
}

QFile* LPDatabaseColorReader::getFile() const
{
	return _file;
}

LPDatabaseColorReader::LPDatabaseColorReader(string filename)
{
	//class initialisation
	MAGIC_READ_MARKER[0]= 27;
	MAGIC_READ_MARKER[1]= 52;
	MAGIC_WRITE_MARKER[0]= 27;
	MAGIC_WRITE_MARKER[1]= 54;
	_file = 0;
	_filePointers = 0;
	_origFile = 0;

	open(filename);
}

void LPDatabaseColorReader::open(string filename)
{
	_filename = filename;
	initFile(filename);
	if (!quickLoad()) fullLoad();
}

void LPDatabaseColorReader::close()
{
	if (_origFile) delete _origFile;
	_origFile = 0;
	if (_filePointers) delete _filePointers;
}

//skip a song at current position
string LPDatabaseColorReader::skipRecord()
{
    QTextStream t(getFile());
		t.setEncoding(QTextStream::UnicodeUTF8);
    QString gr;
    string st;
	if (t.atEnd()) return st;

	if (!getReadMagic(getFile()))
		return st;
	if (t.atEnd()) return st;

	//now read the damn song!
	gr = t.readLine();
	if (gr.length()) //conversion of empty QString to string crashes
		st = string(gr.local8Bit());
	else return st;
	if (!t.atEnd()) t.readLine();//name
	if (!t.atEnd()) t.readLine();//filename
	if (!t.atEnd()) t.readLine();//date
	if (!t.atEnd()) t.readLine();//olddate
	if (!t.atEnd()) t.readLine();//rating
	if (!t.atEnd()) t.readLine();//times
	if (!t.atEnd()) t.readLine();//length
	if (!t.atEnd()) t.readLine();//color
	return st;
}
//read a song at current position
Song LPDatabaseColorReader::readRecord()
{
    QTextStream t(getFile());
		t.setEncoding(QTextStream::UnicodeUTF8);
    QString gr;
    string st;
	int i;
	Song s;

    if (t.atEnd() )
	{
		logWarning("found eof while reading magic marker");
		return s;
	}

	//read a magic number at the current position
	if (!getReadMagic(getFile()))
		return s;
	if (t.atEnd()) 
	{
		logWarning("found eof while reading record");
		return s;
	}
	//now read!
	gr = t.readLine();
	if (gr.length()) //conversion of empty QString to string crashes
		s.setKey(string(gr.local8Bit()));
	else
	{
		logWarning("empty record");
		return s;
	}

	//logDebug(string("reading record ") + s.getKey());
	if (!t.atEnd())
	{
		gr = t.readLine();
		if (gr.length()) //conversion of empty QString to string crashes
			s.setName(string(gr.local8Bit()));
		else s.setName("");
	}
	if (!t.atEnd()) s.setFilename(string(t.readLine().local8Bit()));
	if (!t.atEnd()) s.setDate(t.readLine().toInt());
	if (!t.atEnd()) s.setOldDate(t.readLine().toInt());
	if (!t.atEnd()) s.setTimes(t.readLine().toInt());
	if (!t.atEnd()) s.setLike(t.readLine().toInt());
	if (!t.atEnd()) s.setLength(t.readLine().toInt());
	if (!t.atEnd()) 
	{
		i = t.readLine().toInt();
		s.setColor(i / (256*256), (i%(256*256))/ 256, i%256 );
	}

	return s;
}

void LPDatabaseColorReader::writeRecord(Song s)
{
    QTextStream t(getFile());
		t.setEncoding(QTextStream::UnicodeUTF8);
	int pos = getFile()->at();
	if (!getWriteMagic(getFile()))
	{
		logOops("An invalid database write was stopped. Please report.");
		logWarning("LPDatabaseColorReader::no write marker found while trying to write a record");
		return;
	}
	getFile()->at(pos);
	setReadMagic(getFile());//new magic!
    t << s.getKey().c_str() << "\n";
    t << QString::fromLocal8Bit(s.getName().c_str()) << "\n";
    t << QString::fromLocal8Bit(s.getFilename().c_str()) << "\n";
    t << s.getDate()  << "\n";
    t << s.getOldDate()  << "\n";
    t << s.getTimes() << "\n";
    t << s.getLike() << "\n";
    t << s.length() << "\n";
	int hue,sat,val;
	s.getColor(&hue,&sat,&val);	
	t << hue*256*256 + sat * 256 + val << "\n";
	setWriteMagic(getFile());//new magic!
}

LPDatabaseColorReader::~LPDatabaseColorReader()
{
	close();
	QFile::remove(QString::fromLocal8Bit(string(_filename + ".tmp").c_str()));
}

/** returns true if success */
bool LPDatabaseColorReader::quickLoad()
{
	string filename = _filename + ".idx";
    logInfo("Loading database...",0);
    logDebug("LPDatabaseColorReader::index database " + filename);
    QFile f(QString::fromLocal8Bit(filename.c_str()));
    if (!f.open(IO_ReadOnly))
    {
        logWarning(string("LPDatabaseColorReader::reading indexes ") + filename + " failed.");
        return false;
    }

    QTextStream t( &f );
		t.setEncoding(QTextStream::UnicodeUTF8);
	string key; long pos; QString tmpStr;
    if (t.atEnd() || t.readLine() != "{{index}}")
    {
			f.close(); return false;
	}

	_filePointers = new QMap<string,long>;
    while (!t.atEnd())
    {
        if (!t.atEnd()) tmpStr = t.readLine();
		if (tmpStr.length()) key = tmpStr.local8Bit(); else key = "";
        if (!t.atEnd()) pos = t.readLine().toLong();
		if (!key.empty())
		{
			if ((*_filePointers).contains(key)) (*_filePointers).replace(key, pos);
            else (*_filePointers)[key]=pos;
			
		}
		else logWarning("encountered empty record while reading db");
	}
	f.close();
    logInfo("Loading database... Done.");
    logDebug("LPDatabaseColorReader::index database loaded." );
	return true;
}
void LPDatabaseColorReader::fullLoad()
{
	string filename = _filename;
	logInfo("Loading database...",0);
	logDebug("LPDatabaseColorReader::building index for " + filename);

	QFile* f = getFile(); QTextStream t( f );
		t.setEncoding(QTextStream::UnicodeUTF8);
	string songkey;

	_filePointers = new QMap<string,long>;

	long pos = f->at();
	while (!t.atEnd() && getReadMagic(f))
	{
		f->at(pos);
		songkey = skipRecord();
		if (!songkey.empty())
		{
			if ((*_filePointers).contains(songkey)) (*_filePointers).replace(songkey, pos);
			else (*_filePointers)[songkey]=pos;
		}
		else
		{
			logWarning("encountered empty record while reading db");
		}
		pos = f->at();
	}

	initFile(_filename);//re-init, no need to close
	logDebug(string("LPDatabaseColorReader::entries found: ") + absUtil::its(_filePointers->count()));
	logDebug("LPDatabaseColorReader::reading database. done.");
}

void LPDatabaseColorReader::flush()
{

    string toname = _filename + ".idx";
    logDebug("LPDatabaseColorReader::saving index file...");
    QMapIterator<string,long> it;
    Song s;
   	QFile to(QString::fromLocal8Bit(toname.c_str()));
    if (!to.open(IO_WriteOnly | IO_Truncate))
    {
            logDebug("LPDatabaseColorReader::cannot save index file.");
            logError(string("Could not write index! Please check file permissions or free space to create ") + toname);
            return;
	}
	QTextStream tto(&to);
		tto.setEncoding(QTextStream::UnicodeUTF8);
    tto << QString("{{index}}\n");
    for( it = (*_filePointers).begin(); it != (*_filePointers).end(); ++it )
    {
      tto << it.key().c_str() << "\n";
      tto << it.data() << "\n";
    }
    to.close();
    logDebug("LPDatabaseColorReader::saving index file... done.");
}

//AND rebuilds filepointers!!! AND makes filename this database
void LPDatabaseColorReader::compact()
{
	string compactfile = _filename + ".tmp";

	QMapIterator<string,long> it;
	QMap<string,long>* compactPointers = new QMap<string,long>;

	logDebug("LPDatabaseColorReader::compacting database...");
    QFile from(QString::fromLocal8Bit(_filename.c_str()));
    QFile to(QString::fromLocal8Bit(compactfile.c_str()));
    if (!from.open(IO_ReadOnly))
    {
			logDebug("LPDatabaseColorReader::cannot read.");
			logError(string("Could not read database! Please check file permissions to read ") + _filename);
			return;
    }
	if (!to.open(IO_ReadWrite | IO_Truncate))
    {
			logDebug("LPDatabaseColorReader::cannot save.");
			logError(string("Could not save database! Please check file permissions or free space to create ") + compactfile);
			return;
    }
	QTextStream tto(&to);
		tto.setEncoding(QTextStream::UnicodeUTF8);
	tto << "{{color}}\n";

	setWriteMagic(&to);
	
	string key; long pos;
	Song s; Song sv;
    for( it = (*_filePointers).begin(); it != (*_filePointers).end(); ++it )
	{
		//logDebug(absUtil::getUtil()->its(i++));
		from.at(it.data());
		setFile(&from);
		s = readRecord();
		//logDebug(s.getKey());
		key = s.getKey();
		if (!s.getKey().empty())
		{
			pos = to.at();
			if ((*compactPointers).contains(key)) (*compactPointers).replace(key, pos);
			else (*compactPointers)[key]=pos;
			setFile(&to);
			writeRecord(s);
		}
	}
	to.close();
	from.close();
	absUtil::getUtil()->copyFile(compactfile,_filename);
	_filePointers = compactPointers;
	initFile(_filename);//clean up the mess...
	flush();
	logDebug("LPDatabaseColorReader::compacting database. done.");
}

void LPDatabaseColorReader::store(Song song )
{
//	logDebug("3");
	if (song.empty())
	{
			logWarning("LPDatabaseColorReader::empty song, not storing.");
			return;
	}
	Song old = retrieve(song.getKey());
	Song toStore = song;

	if (!old.empty())
	{
		if (!old.equivalent(song))
		{
			logWarning(string("LPDatabaseColorReader::key clash! fixing " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName() + " vs" + old.getName());
			song.setKey(song.getKey() + "A");
			store(song);
			return;
		}
		else logDebug(string("LPDatabaseColorReader::storing, overwriting DB entry " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName());
	}
//	logDebug("4");

	QFile f(QString::fromLocal8Bit(_filename.c_str()));
	if (!f.open(IO_ReadWrite))
    {
		logWarning("LPDatabaseColorReader::could not open database to store song");
		return;
	}
	//back up to the magic marker
	f.at(f.size()-2); //note: this does not solve the problem when another lplayer adds records: it won't hurt the database, but data could be lost in the end
	if ((*_filePointers).contains(toStore.getKey())) (*_filePointers).replace(toStore.getKey(), f.at());
	else (*_filePointers)[toStore.getKey()]=(f.at());
	setFile(&f);
	writeRecord(toStore);
		//string vkey = toStore.getKey();
		//logDebug("verifying..");
		//f.at((*_filePointers)[toStore.getKey()]);
		//setFile(&f);
		//if (readRecord().getKey() == vkey)  logDebug("pass."); else logDebug("fail.");
				
	f.close();
}
void LPDatabaseColorReader::remove(Song song)
{
	(*_filePointers).remove(song.getKey());
}

int LPDatabaseColorReader::getSize() 
{
	return (*_filePointers).count();
}

bool LPDatabaseColorReader::inDatabase(string key)
{
	QMapIterator<string,long> i = (*_filePointers).find(key);
	if (i == (*_filePointers).end())
	{
		return false;
	}
	return true;
}

//returns Song() if not in database
Song LPDatabaseColorReader::retrieve(string skey)
{
	QMapIterator<string,long> p = (*_filePointers).find(skey);
	Song song;
	if (p == (*_filePointers).end())
	{
	}
	else
	{
		QFile f(QString::fromLocal8Bit(_filename.c_str()));
		if (!p.data()) return song;
		if (!f.open(IO_ReadOnly)) return song;
		f.at(p.data());
		setFile(&f);
		song = readRecord();
		if (song.getKey() != skey)
		{
			logWarning(string("asked to retrieve ") + skey + " at position " + absUtil::getUtil()->its(p.data()) + " in file " + _filename + " and got " + song.getKey() + ".");
			logError("There is a problem with your database! Try removing the idx file in your homedir and restart LongPlayer. If that doesn't, please dig up a backup...");
		}
		setFile(0);
		f.close();
	}
	return song;
}


void LPDatabaseColorReader::setReadMagic(QFile* f)
{
    QTextStream t(f);
	//int pos = f->at();
	t.writeRawBytes(MAGIC_READ_MARKER,2);
}
bool LPDatabaseColorReader::getReadMagic(QFile* f)
{
    QTextStream t(f);
	char test[3];
	t.readRawBytes(test,2);
	t.setEncoding(QTextStream::UnicodeUTF8);
	if (strncmp(test,MAGIC_READ_MARKER,2))
	{
		//f->at(f->at()-2);
		logWarning("LPDatabaseColorReader::no magic read marker found; got:");
		test[2]='\0';
		QString temp(test);
		logDebug(string(temp.local8Bit()));
		temp = t.readLine();
		logDebug("what follows is:");
		if (!temp.isEmpty()) logDebug(string(temp.local8Bit()));
		return false;
	}
	return true;
}
//the only function that does not change the position in the file
void LPDatabaseColorReader::setWriteMagic(QFile* f)
{
    QTextStream t(f);
	int pos = f->at();
	t.writeRawBytes(MAGIC_WRITE_MARKER,2);
	f->at(pos);
}
bool LPDatabaseColorReader::getWriteMagic(QFile* f)
{
    QTextStream t(f);
	char test[3];
	t.readRawBytes(test,2);
	if (strncmp(test,MAGIC_WRITE_MARKER,2))
	{
		logWarning("LPDatabaseColorReader::no magic write marker found.");
		test[2]='\0';
		QString temp(test);
		logDebug(string(temp.local8Bit()));
		return false;
	}
	return true;
}
void LPDatabaseColorReader::test()
{
	int pos = getFile()->at();
	if (!getWriteMagic(getFile())) logWarning("no write marker found");
	getFile()->at(pos);
	if (!getReadMagic(getFile())) logWarning("no read marker found");
	getFile()->at(pos);
}

LPDatabaseColorReader::LPColorDBIterator::
  LPColorDBIterator(LPDatabaseColorReader* reader) {
    _reader = reader;
}

void LPDatabaseColorReader::LPColorDBIterator::reset() {
    _reader->resetPosition();
    _file = _reader->getFile();
}
	
Song LPDatabaseColorReader::LPColorDBIterator::next() {
    _reader->setFile(_file);
    return _reader->readRecord();
}

bool LPDatabaseColorReader::LPColorDBIterator::atEnd() {
    return (_file->atEnd());
}
	

LPDatabaseIterator* LPDatabaseColorReader::iterator()
{
    return new LPDatabaseColorReader::LPColorDBIterator(this);
}
