#include "lpdatabasemagicreader.h"
#include "lpglobal.h"
#include "absutil.h"

#include <qstring.h>
#include <qdir.h>
#include <qdom.h>
#include <qtextstream.h>
#include <qprogressdialog.h>

#include <string>
#include <iostream>

/** opens the associated database file and goes to the first record, leaving it in read-only modus
  * if it does not exists, it is created
  * returns false if something went wrong
  */
bool LPDatabaseMagicReader::initFile(string filename)
{
	if (_origFile) delete _origFile;
    _origFile = new QFile(QString::fromLocal8Bit(filename.c_str()));
	//initialize an empty DB if necessary
	if (!_origFile->exists())
	{
    	if (!_origFile->open(IO_WriteOnly | IO_Truncate))
    	{
            logDebug("LPDatabaseMagicReader::cannot save.");
            logError(string("Could not save database! Please check file permissions or free space."));
			_origFile = 0;
            return false;
   		 }
		else
		{
			QTextStream t(_origFile);
			t << "{{magic}}\n";
			logDebug(absUtil::its(_origFile->at()));
			setWriteMagic(_origFile);
			logDebug(absUtil::its(_origFile->size()));
		}
	}

	resetPosition();
	return true;
}

void LPDatabaseMagicReader::resetPosition()
{
	if (!_origFile) return;
	_origFile->close();
    if (!_origFile->open(IO_ReadOnly))
    {
        logWarning(string("LPDatabaseMagicReader::opening ") +
string(_origFile->name().local8Bit()) + " failed.");
		logError(string("Could not reset database file ") + string(_origFile->name().local8Bit()) + ". Please check file permissions and quit any processes using it.");
        return ;
    }
	QTextStream t( _origFile );
	QString str;
	if (!t.atEnd())
	{
		str = t.readLine();
		if (str != "{{magic}}")
		{
			_origFile->close();
			logError("LPDatabaseMagicReader::reset failed: this is not a magical file.");
			logDebug(string(str.local8Bit()));
			return ;
		}
	}
	//don't close!

	//the right file just at the right position... (being the first magic marker..)
	setFile(_origFile);

}

//private!
void LPDatabaseMagicReader::setFile(QFile* file)
{
	if (file) _file = file;
	else _file = _origFile; //reset file
}

QFile* LPDatabaseMagicReader::getFile() const
{
	return _file;
}

LPDatabaseMagicReader::LPDatabaseMagicReader(string filename)
{
	//class initialisation
	MAGIC_READ_MARKER[0]= 27;
	MAGIC_READ_MARKER[1]= 52;
	MAGIC_WRITE_MARKER[0]= 27;
	MAGIC_WRITE_MARKER[1]= 54;
	_file = 0;
	_filePointers = 0;
	_origFile = 0;

	open(filename);
}

void LPDatabaseMagicReader::open(string filename)
{
	_filename = filename;
	initFile(filename);
	if (!quickLoad()) fullLoad();
}

void LPDatabaseMagicReader::close()
{
	if (_origFile) delete _origFile;
	_origFile = 0;
	dbAccess.lock();
	if (_filePointers) delete _filePointers;
	dbAccess.unlock();
}

//skip a song at current position
string LPDatabaseMagicReader::skipRecord()
{
    QTextStream t(getFile());
    QString gr;
    string st;
	if (t.atEnd()) return st;

	if (!getReadMagic(getFile()))
		return st;
	if (t.atEnd()) return st;

	//now read the damn song!
	gr = t.readLine();
	if (gr.length()) //conversion of empty QString to string crashes
		st = string(gr.local8Bit());
	else return st;
	if (!t.atEnd()) t.readLine();//name
	if (!t.atEnd()) t.readLine();//filename
	if (!t.atEnd()) t.readLine();//date
	if (!t.atEnd()) t.readLine();//olddate
	if (!t.atEnd()) t.readLine();//rating
	if (!t.atEnd()) t.readLine();//times
	if (!t.atEnd()) t.readLine();//length
	return st;
}
//read a song at current position
Song LPDatabaseMagicReader::readRecord()
{
    QTextStream t(getFile());
    QString gr;
    string st;
	Song s;

    if (t.atEnd() )
	return s;

	//read a magic number at the current position
	if (!getReadMagic(getFile()))
		return s;
	if (t.atEnd()) return s;

	//now read!
	gr = t.readLine();
	if (gr.length()) //conversion of empty QString to string crashes
		s.setKey(string(gr.local8Bit()));
	else return s;

	//logDebug(string("reading record ") + s.getKey());
	if (!t.atEnd())
	{
		gr = t.readLine();
		if (gr.length()) //conversion of empty QString to string crashes
			s.setName(string(gr.local8Bit()));
		else s.setName("");
	}
	if (!t.atEnd()) s.setFilename(string(t.readLine().local8Bit()));
	if (!t.atEnd()) s.setDate(t.readLine().toInt());
	if (!t.atEnd()) s.setOldDate(t.readLine().toInt());
	if (!t.atEnd()) s.setTimes(t.readLine().toInt());
	if (!t.atEnd()) s.setLike(t.readLine().toInt());
	if (!t.atEnd()) s.setLength(t.readLine().toInt());
	return s;
}

void LPDatabaseMagicReader::writeRecord(Song s)
{
    QTextStream t(getFile());
	int pos = getFile()->at();
	if (!getWriteMagic(getFile()))
	{
		logOops("An invalid database write was stopped. Please report.");
		logWarning("LPDatabaseMagicReader::no write marker found while trying to write a record");
		return;
	}
	getFile()->at(pos);
	setReadMagic(getFile());//new magic!
    t << s.getKey().c_str() << "\n";
    t << QString::fromLocal8Bit(s.getName().c_str()) << "\n";
    t << QString::fromLocal8Bit(s.getFilename().c_str()) << "\n";
    t << s.getDate()  << "\n";
    t << s.getOldDate()  << "\n";
    t << s.getTimes() << "\n";
    t << s.getLike() << "\n";
    t << s.length() << "\n";
	setWriteMagic(getFile());//new magic!
}

LPDatabaseMagicReader::~LPDatabaseMagicReader()
{
	close();
	QFile::remove(QString::fromLocal8Bit(string(_filename + ".tmp").c_str()));
}

/** returns true if success */
bool LPDatabaseMagicReader::quickLoad()
{
	string filename = _filename + ".idx";
    logInfo("Loading database...",0);
    logDebug("LPDatabaseMagicReader::index database " + filename);
    QFile f(QString::fromLocal8Bit(filename.c_str()));
    if (!f.open(IO_ReadOnly))
    {
        logWarning(string("LPDatabaseMagicReader::reading indexes ") + filename + " failed.");
        return false;
    }

    QTextStream t( &f );
	string key; long pos;
    if (t.atEnd() || t.readLine() != "{{index}}")
    {
			f.close(); return false;
	}

	dbAccess.lock();
	_filePointers = new QMap<string,long>;
    while (!t.atEnd())
    {
        if (!t.atEnd()) key = string(t.readLine().local8Bit());
        if (!t.atEnd()) pos = t.readLine().toLong();
        if ((*_filePointers).contains(key)) (*_filePointers).replace(key, pos);
            else (*_filePointers)[key]=pos;
	}
	f.close();
	dbAccess.unlock();
    logInfo("Loading database... Done.");
    logDebug("LPDatabaseMagicReader::index database loaded." );
	return true;
}
void LPDatabaseMagicReader::fullLoad()
{
	string filename = _filename;
	logInfo("Loading database...",0);
	logDebug("LPDatabaseMagicReader::building index for " + filename);

	QFile* f = getFile(); QTextStream t( f );
	string songkey;

	dbAccess.lock();
	_filePointers = new QMap<string,long>;

	long pos = f->at();
	while (!t.atEnd() && getReadMagic(f))
	{
		f->at(pos);
		songkey = skipRecord();
		if ((*_filePointers).contains(songkey)) (*_filePointers).replace(songkey, pos);
		else (*_filePointers)[songkey]=pos;
		pos = f->at();
	}
	dbAccess.unlock();

	initFile(_filename);//re-init, no need to close
	logDebug(string("LPDatabaseMagicReader::entries found: ") + absUtil::its(_filePointers->count()));
	logDebug("LPDatabaseMagicReader::reading database. done.");
}

void LPDatabaseMagicReader::flush()
{

	dbAccess.lock();
    string toname = _filename + ".idx";
    logDebug("LPDatabaseMagicReader::saving index file...");
    QMapIterator<string,long> it;
    Song s;
   	QFile to(QString::fromLocal8Bit(toname.c_str()));
    if (!to.open(IO_WriteOnly | IO_Truncate))
    {
            logDebug("LPDatabaseMagicReader::cannot save index file.");
            logError(string("Could not write index! Please check file permissions or free space to create ") + toname);
            return;
	}
	QTextStream tto(&to);
    tto << QString("{{index}}\n");
    for( it = (*_filePointers).begin(); it != (*_filePointers).end(); ++it )
    {
      tto << it.key().c_str() << "\n";
      tto << it.data() << "\n";
    }
    to.close();
	dbAccess.unlock();
    logDebug("LPDatabaseMagicReader::saving index file... done.");
}

//AND rebuilds filepointers!!! AND makes filename this database
void LPDatabaseMagicReader::compact()
{
	dbAccess.lock();
	//copy database file
	string tmpfile = _filename + ".tmp";
	absUtil::getUtil()->copyFile(_filename,tmpfile);

	QMapIterator<string,long> it;
	QMap<string,long>* copy = new QMap<string,long>(*_filePointers);
	logDebug("LPDatabaseMagicReader::writing database...");
	Song s;
    QFile from(QString::fromLocal8Bit(tmpfile.c_str()));
    QFile to(QString::fromLocal8Bit(_filename.c_str()));
    if (!from.open(IO_ReadOnly))
    {
			logDebug("LPDatabaseMagicReader::cannot read.");
			logError(string("Could not read database! Please check file permissions to read ") + tmpfile);
			return;
    }
	if (!to.open(IO_ReadWrite | IO_Truncate))
    {
			logDebug("LPDatabaseMagicReader::cannot save.");
			logError(string("Could not save database! Please check file permissions or free space to create ") + _filename);
			return;
    }
	QTextStream tfrom(&from);
	QTextStream tto(&to);
	tto << "{{magic}}\n";
	setWriteMagic(&to);
	string key; long pos;
    for( it = (*copy).begin(); it != (*copy).end(); ++it )
	{
		from.at(it.data());
		setFile(&from);
		s = readRecord();
		key = s.getKey();
		pos = to.at();
		if ((*_filePointers).contains(key)) (*_filePointers).replace(key, pos);
		else (*_filePointers)[key]=pos;
		setFile(&to);
		writeRecord(s);
	}
	to.close();
	from.close();
	initFile(_filename);//clean up the mess...
	dbAccess.unlock();
	flush();
	logDebug("LPDatabaseMagicReader::saving database. done.");
}

void LPDatabaseMagicReader::store(Song song )
{
//	logDebug("3");
	if (song.empty())
	{
			logWarning("LPDatabaseMagicReader::empty song, not storing.");
			return;
	}
	Song old = retrieve(song.getKey());
	Song toStore = song;

	if (!old.empty())
	{
		if (!old.equivalent(song))
		{
			logWarning(string("LPDatabaseMagicReader::key clash! fixing " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName() + " vs" + old.getName());
			song.setKey(song.getKey() + "A");
			store(song);
			return;
		}
		else logDebug(string("LPDatabaseMagicReader::overwriting old DB entry since " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName() + " is equivalent to " + old.getName());
	}
//	logDebug("4");

	dbAccess.lock();
	QFile f(QString::fromLocal8Bit(_filename.c_str()));
	if (!f.open(IO_ReadWrite))
    {
		logWarning("LPDatabaseMagicReader::could open database to store song");
		return;
	}
	//back up to the magic marker
	f.at(f.size()-2); //note: this does not solve the problem when another lplayer adds records: it won't hurt the database, but data could be lost in the end
	if ((*_filePointers).contains(toStore.getKey())) (*_filePointers).replace(toStore.getKey(), f.at());
	else (*_filePointers)[toStore.getKey()]=(f.at());
	setFile(&f);
		writeRecord(toStore);
	f.close();
	dbAccess.unlock();
}
void LPDatabaseMagicReader::remove(Song song)
{
	dbAccess.lock();
	(*_filePointers).remove(song.getKey());
	dbAccess.unlock();
}

int LPDatabaseMagicReader::getSize() 
{
	return (*_filePointers).count();
}

bool LPDatabaseMagicReader::inDatabase(string key)
{
	dbAccess.lock();
	QMapIterator<string,long> i = (*_filePointers).find(key);
	if (i == (*_filePointers).end())
	{
		dbAccess.unlock();
		return false;
	}
	dbAccess.unlock();
	return true;
}

//returns Song() if not in database
Song LPDatabaseMagicReader::retrieve(string skey)
{
	dbAccess.lock();
	QMapIterator<string,long> p = (*_filePointers).find(skey);
	Song song;
	if (p == (*_filePointers).end())
	{
		dbAccess.unlock();
	}
	else
	{
		dbAccess.unlock();
		QFile f(QString::fromLocal8Bit(_filename.c_str()));
		if (!p.data()) return song;
		if (!f.open(IO_ReadOnly)) return song;
		f.at(p.data());
		setFile(&f);
		song = readRecord();
		if (song.getKey() != skey)
		{
			logError("There is a problem with your database! Try removing the idx file in your homedir and restart LongPlayer. If that doesn't, please dig up a backup...");
		}
		setFile(0);
		f.close();
	}
	return song;
}

void LPDatabaseMagicReader::setReadMagic(QFile* f)
{
    QTextStream t(f);
	//int pos = f->at();
	t.writeRawBytes(MAGIC_READ_MARKER,2);
}
bool LPDatabaseMagicReader::getReadMagic(QFile* f)
{
    QTextStream t(f);
	char test[3];
	t.readRawBytes(test,2);
	if (strncmp(test,MAGIC_READ_MARKER,2))
	{
		logWarning("LPDatabaseMagicReader::no magic read marker found.");
		test[2]='\0';
		QString temp(test);
		logDebug(string(temp.local8Bit()));
		return false;
	}
	return true;
}
//the only function that does not change the position in the file
void LPDatabaseMagicReader::setWriteMagic(QFile* f)
{
    QTextStream t(f);
	int pos = f->at();
	t.writeRawBytes(MAGIC_WRITE_MARKER,2);
	f->at(pos);
}
bool LPDatabaseMagicReader::getWriteMagic(QFile* f)
{
    QTextStream t(f);
	char test[3];
	t.readRawBytes(test,2);
	if (strncmp(test,MAGIC_WRITE_MARKER,2))
	{
		logWarning("LPDatabaseMagicReader::no magic write marker found.");
		test[2]='\0';
		QString temp(test);
		logDebug(string(temp.local8Bit()));
		return false;
	}
	return true;
}
void LPDatabaseMagicReader::test()
{
	int pos = getFile()->at();
	if (!getWriteMagic(getFile())) logWarning("no write marker found");
	getFile()->at(pos);
	if (!getReadMagic(getFile())) logWarning("no read marker found");
	getFile()->at(pos);
}

LPDatabaseMagicReader::LPMagicDBIterator::
  LPMagicDBIterator(LPDatabaseMagicReader* reader) {
    _reader = reader;
}

void LPDatabaseMagicReader::LPMagicDBIterator::reset() {
    _reader->resetPosition();
    _file = _reader->getFile();
}
	
Song LPDatabaseMagicReader::LPMagicDBIterator::next() {
    _reader->setFile(_file);
    return _reader->readRecord();
}

bool LPDatabaseMagicReader::LPMagicDBIterator::atEnd() {
    return (_file->atEnd());
}
	

LPDatabaseIterator* LPDatabaseMagicReader::iterator()
{
    return new LPDatabaseMagicReader::LPMagicDBIterator(this);
}

