#include "lpdatabasepersistentreader.h"
#include "lpdatabaseiterator.h"
#include "absutil.h"
#include <qstring.h>
#include <qdir.h>
#include <qdom.h>
#include <qtextstream.h>
#include <qprogressdialog.h>
#include <string>
#include <iostream>
#include "lpglobal.h"

bool LPDatabasePersistentReader::initFile(string filename)
{
	_origFile = new QFile( QString::fromLocal8Bit(filename.c_str()) );
	if (!_origFile->exists()) 
	{ 
    	if (!_origFile->open(IO_WriteOnly))
    	{
            logDebug("LPDatabasePersistentReader::cannot save.");
            logError(string("Could not save database! Please check file permissions or free space."));
			_origFile = 0;
            return false;
   		 } 
		else
		{
			QTextStream t(_origFile);
			t << "{{persistent}}\n";
		}
	}
	resetPosition();
	return true;
}

void LPDatabasePersistentReader::setFile(QFile* file) 
{
	if (file) _file = file;
	else _file = _origFile;
}

void LPDatabasePersistentReader::resetPosition()
{
    if (!_origFile->open(IO_ReadOnly))                   
    {                                           
        logWarning(string("LPDatabasePersistentReader::opening ") +
string(_origFile->name().local8Bit()) + " failed.");
		logError(string("Could not reset database file ") + string(_origFile->name().local8Bit()) + ". Please check file permissions and quit any processes using it.");
        return;
    } 
	QTextStream t( _origFile );
	QString str;
	if (!t.atEnd())
	{
		str = t.readLine();
		if (str != "{{persistent}}")
		{
			_origFile->close();
			logError("LPDatabasePersistentReader::reset failed: unsupported database file.");
			logDebug(string(str.local8Bit()));
			return;
		}
	}
	//don't close!

	//the right file just at the right position...
	_file = _origFile;
}

QFile* LPDatabasePersistentReader::getFile() const
{
	return _file;
}

LPDatabasePersistentReader::LPDatabasePersistentReader(string filename)
{
	_file = 0;
	_filename = filename;
	initFile(filename);

	//preparations before loading a database
	dbAccess.lock();
	_base = new QMap<string,Song>;
	_filePointers = new QMap<string,long>;
	dbAccess.unlock();

	//avoid error when new database needs to be created
   	if (!QFile::exists(QString::fromLocal8Bit(filename.c_str()))) return;
	if (!quickLoad()) fullLoad();
}

//read a song at current position
Song LPDatabasePersistentReader::readRecord()
{
    QTextStream t(getFile());
    QString gr;
    string st;
	Song s;
    if (!t.atEnd())
    {
		gr = t.readLine();
		if (gr.length()) //conversion of empty QString to string crashes
			s.setKey(string(gr.local8Bit()));
		else return s;

		//logDebug(string("reading record ") + s.getKey());
		if (!t.atEnd())
		{
			gr = t.readLine();
			if (gr.length()) //conversion of empty QString to string crashes
				s.setName(string(gr.local8Bit()));
			else s.setName("");
		}
		if (!t.atEnd()) s.setFilename(string(t.readLine().local8Bit()));
		if (!t.atEnd()) s.setDate(t.readLine().toInt());
		if (!t.atEnd()) s.setOldDate(t.readLine().toInt()); 
		if (!t.atEnd()) s.setTimes(t.readLine().toInt());
		if (!t.atEnd()) s.setLike(t.readLine().toInt());
		if (!t.atEnd()) s.setLength(t.readLine().toInt());
    }           
	return s; 
}
void LPDatabasePersistentReader::writeRecord(Song s)
{
        QTextStream t(getFile());
		char test[2];
		t.readRawBytes(test,2);
        t << s.getKey().c_str() << "\n";
        t << QString::fromLocal8Bit(s.getName().c_str()) << "\n";
        t << QString::fromLocal8Bit(s.getFilename().c_str()) << "\n";
        t << s.getDate()  << "\n";
        t << s.getOldDate()  << "\n";
        t << s.getTimes() << "\n";
        t << s.getLike() << "\n";
        t << s.length() << "\n";
}

LPDatabasePersistentReader::~LPDatabasePersistentReader()
{
	QFile::remove(QString::fromLocal8Bit(string(_filename + ".tmp").c_str()));
}

/** returns true if success */
bool LPDatabasePersistentReader::quickLoad()
{
	string filename = _filename + ".idx";
    logInfo("Loading database...",0);
    logDebug("LPDatabasePersistentReader::index database " + filename);
    QFile f(QString::fromLocal8Bit(filename.c_str()));
    if (!f.open(IO_ReadOnly))
    {
        logWarning(string("LPDatabasePersistentReader::reading indexes ") + filename + " failed.");
        return false;
    }
    QTextStream t( &f );
    QString st; 
	string key; long pos;
    if (!t.atEnd())
    {
        st = t.readLine(); //header line (must be index)
        if (st != "{{index}}")
        {                    
			f.close();
			return false;
        }                                                                               
	}
    while (!t.atEnd())
    {
        if (!t.atEnd()) key = t.readLine().local8Bit();
        if (!t.atEnd()) pos = t.readLine().toLong();
        if ((*_filePointers).contains(key)) (*_filePointers).replace(key, pos);
            else (*_filePointers)[key]=pos;  
	}
	f.close();
    logInfo("Loading database... Done.");
    logDebug("LPDatabasePersistentReader::index database loaded." );
	return true;
}
void LPDatabasePersistentReader::fullLoad()
{
	string filename = _filename;
	logInfo("Loading database...",0);
	logDebug("LPDatabasePersistentReader::reading database " + filename);

	QFile* f = getFile();
	QTextStream t( f );
	string st; QString gr;
	Song s; Song old;
	long pos;
	while (!t.atEnd())
	{
		pos = f->at();
		gr = t.readLine();
		s.setKey(string(gr.local8Bit()));
		if ((*_filePointers).contains(s.getKey())) (*_filePointers).replace(s.getKey(), pos);
		else (*_filePointers)[s.getKey()]=pos;
		if (!t.atEnd()) t.readLine();//name 
		if (!t.atEnd()) t.readLine();//filename
		if (!t.atEnd()) t.readLine();//date
		if (!t.atEnd()) t.readLine();//olddate
		if (!t.atEnd()) t.readLine();//rating
		if (!t.atEnd()) t.readLine();//times
		if (!t.atEnd()) t.readLine();//length
	}
	f->close();

	logDebug(string("LPDatabasePersistentReader::entries found: ") + absUtil::its(_filePointers->count()));
	logDebug("LPDatabasePersistentReader::reading database. done.");
}

void LPDatabasePersistentReader::flush()
{
	
	dbAccess.lock();
    string toname = _filename + ".idx";
    logDebug("LPDatabasePersistentReader::saving index file...");
    QMapIterator<string,long> it;
    Song s;
   	QFile to(QString::fromLocal8Bit(toname.c_str()));
    if (!to.open(IO_WriteOnly))
    {
            logDebug("LPDatabasePersistentReader::cannot save index file.");
            logError(string("Could not write index! Please check file permissions or free space to create ") + toname);
            return;
	}
	QTextStream tto(&to);
    tto << QString("{{index}}\n");
    for( it = (*_filePointers).begin(); it != (*_filePointers).end(); ++it )                        
    {
      tto << it.key().c_str() << "\n";
      tto << it.data() << "\n";
    }
    to.close();
	dbAccess.unlock();
    logDebug("LPDatabasePersistentReader::saving index file... done.");
}

//AND rebuilds filepointers!!! AND makes filename this database
void LPDatabasePersistentReader::compact()
{
	dbAccess.lock();
	//copy database file
	string tmpfile = _filename + ".tmp";
	absUtil::getUtil()->copyFile(_filename,tmpfile);

	QMapIterator<string,long> it;
	QMap<string,long>* copy = new QMap<string,long>(*_filePointers); 
	logDebug("LPDatabasePersistentReader::writing database...");
	Song s;
    QFile from(QString::fromLocal8Bit(tmpfile.c_str()));
    QFile to(QString::fromLocal8Bit(_filename.c_str()));
    if (!from.open(IO_ReadOnly))
    {
			logDebug("LPDatabasePersistentReader::cannot read.");
			logError(string("Could not read database! Please check file permissions to read ") + tmpfile);
			return;
    }
	if (!to.open(IO_WriteOnly))
    {
			logDebug("LPDatabasePersistentReader::cannot save.");
			logError(string("Could not save database! Please check file permissions or free space to create ") + _filename);
			return;
    }
	QTextStream tfrom(&from);
	QTextStream tto(&to);
	tto << "{{persistent}}\n";
	string key; long pos;
    for( it = (*copy).begin(); it != (*copy).end(); ++it )
	{
		from.at(it.data());
		setFile(&from);
		s = readRecord();
		key = s.getKey();                                          
		pos = to.at();
		if ((*_filePointers).contains(key)) (*_filePointers).replace(key, pos); 
		else (*_filePointers)[key]=pos;  
		setFile(&to);
		writeRecord(s);
	}
	to.close();
	from.close();
	initFile(_filename);//clean up the mess...
	dbAccess.unlock();
	flush();
	logDebug("LPDatabasePersistentReader::saving database. done.");
}

void LPDatabasePersistentReader::store(Song song )
{
//	logDebug("3");
	if (song.empty())
	{
			logWarning("LPDatabasePersistentReader::empty song, not storing.");
			return;
	}
	Song old = retrieve(song.getKey());
	Song toStore = song;
	
	if (!old.empty())
	{
		if (!old.equivalent(song))
		{
			logWarning(string("LPDatabasePersistentReader::key clash! fixing " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName() + " vs" + old.getName());
			song.setKey(song.getKey() + "A");
			store(song); 
			return;
		}
		else logDebug(string("LPDatabasePersistentReader::overwriting old DB entry since " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName() + " is equivalent to " + old.getName());
	}
//	logDebug("4");

	dbAccess.lock();
	QFile f(QString::fromLocal8Bit(_filename.c_str()));
	if (!f.open(IO_ReadWrite | IO_Append))
    {
		logWarning("could not store song");
		return;
	}
	if ((*_filePointers).contains(toStore.getKey())) (*_filePointers).replace(toStore.getKey(), f.at());
	else (*_filePointers)[toStore.getKey()]=f.at();
	setFile(&f);
	writeRecord(toStore);
	f.close();
	dbAccess.unlock();
}
void LPDatabasePersistentReader::remove(Song song)
{
	dbAccess.lock();
	(*_filePointers).remove(song.getKey());
	dbAccess.unlock();
}

int LPDatabasePersistentReader::getSize() 
{
	return (*_filePointers).count();
}

bool LPDatabasePersistentReader::inDatabase(string key)
{
	dbAccess.lock();
	QMapIterator<string,long> i = (*_filePointers).find(key);
	if (i == (*_filePointers).end())
	{
		dbAccess.unlock();
		return false;
	}
	dbAccess.unlock();
	return true;
}

//returns Song() if not in database
Song LPDatabasePersistentReader::retrieve(string skey)
{
	dbAccess.lock();
	QMapIterator<string,long> p = (*_filePointers).find(skey);
	Song song;
	if (p == (*_filePointers).end())
	{
		dbAccess.unlock();
	}
	else
	{
		dbAccess.unlock();
		QFile f(QString::fromLocal8Bit(_filename.c_str()));
		if (!p.data()) return song;
		if (!f.open(IO_ReadOnly)) return song;
		f.at(p.data());
		setFile(&f);
		song = readRecord();
		if (song.getKey() != skey) 
		{
			logError("There is a problem with your database! Try removing the idx file in your homedir and restart LongPlayer. If that doesn't, please dig up a backup...");
		}
		setFile(0);
		f.close();
	}
	return song;
}

LPDatabasePersistentReader::LPPersistentDBIterator::
  LPPersistentDBIterator(LPDatabasePersistentReader* reader) {
    _reader = reader;
}

void LPDatabasePersistentReader::LPPersistentDBIterator::reset() {
    _reader->resetPosition();
    _file = _reader->getFile();
}
	
Song LPDatabasePersistentReader::LPPersistentDBIterator::next() {
    _reader->setFile(_file);
    return _reader->readRecord();
}

bool LPDatabasePersistentReader::LPPersistentDBIterator::atEnd() {
    return (_file->atEnd());
}
	

LPDatabaseIterator* LPDatabasePersistentReader::iterator()
{
    return new LPDatabasePersistentReader::LPPersistentDBIterator(this);
}

