#include "lpsettings.h"
#include "playlist.h"
#include "lpsongfilter.h"
#include "lpdatabase.h"
#include "lplayer.h"

#include "lpplaylistiterator.h"
#include "m3uplaylist.h"

#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>

/* adapt means: don't spend too much time, adjust so that there is a fair amount spent, and, if a good song is not found, take the best song */
LPPlaylistIterator::LPPlaylistIterator(Playlist* playlist, LPSongFilter* filter, LPDatabase* database, bool random, bool adapt)
{
	_playlist = playlist; _filter = filter; _random = random; _database = database; _adapt = adapt;
	_settings = _database->getSettings();
	_currpos = 0;

}
//may return empty songs
Song LPPlaylistIterator::next()
{
	bool stop(false);
	Song song;
	Song bestSong;
	float bestValue(-10000);
	float currValue(-10000);
	int iterations(0);
	int minRandomSearch = _settings->getSetting(LPlayer::SettingEffort);
	int maxSearch = _settings->getSetting(LPlayer::SettingEffort) * 2;//may cause empty songs to be returned, but puts an upper boundary on the search effort

	string filename;

	while (!stop  && hasNext()) {
		logDebug(string("LPIterator::iteration ") + absUtil::its(iterations) + " / position " + absUtil::its(_currpos));
		//step1: acquire possibile candidate
		vector<string> *filelist = _playlist->filenames();
		if (!filelist) logError("PlaylistIterator::playlist is not built");

		if (_random) {		
			//thanks to spike for this wonderful idea :)
			//1. get random number between curpos and filelistsize
			uint rand = (uint) (_currpos + absUtil::getUtil()->random()*(filelist->size()-_currpos-1));
			//2. switch found song with current position
			filename = (*filelist)[rand];
			(*filelist)[rand] = (*filelist)[_currpos];
			//(*filelist)[_currpos] = filename; //we leave a perfectly randomized list behind (between 0 and _currpos)
			
		}
		else { filename = (*filelist)[_currpos]; }
		_currpos++;
		song = _database->retrieveAndCorrect(filename,"");
    	logDebug(string("LPIterator::about to judge ") + song.getFilename() + "/" + song.getKey());
	
		currValue = songGoodahness(song);
		logDebug(string("LPIterator::current/best goodahness: ") + absUtil::fts(currValue) + "/" + absUtil::fts(bestValue) );
		//we want the song with the highest value
		bestSong = (currValue > bestValue) ? song : bestSong;
		//change value after changing songs
		bestValue = (currValue > bestValue) ? currValue : bestValue;
	
		stop = stop || ((!_random || (iterations > minRandomSearch)) && (bestValue > (LPSongFilter::DISREC + LPSongFilter::DELTA))); //song is good enough provided we've searched enough times
		stop = stop || (maxSearch && (iterations > maxSearch));	//still nothing? shame...
		iterations++;
	}
	
	if (bestValue <= (LPSongFilter::REJECT + LPSongFilter::DELTA)) { //abnormal error?  
		logWarning("LPIterator::no songs found. sorry.");
		bestSong = Song();
	}
	else if (!_adapt) {//=strict
		if (bestValue <= (LPSongFilter::DISREC + LPSongFilter::DELTA)) {
			logDebug("LPIterator::no songs found. sorry.");
			bestSong = Song();
		}
	}
	logDebug("LPIterator::going with " + bestSong.getFilename() + "/" + bestSong.getKey() + " with a value of " + absUtil::fts(bestValue));
	return bestSong;
}

//note: it could be that hasNext() == true  and next does not return a valid song
bool LPPlaylistIterator::hasNext() {
	if (!_playlist->filenames()) { logError("no playlist"); return false; }
	if (_currpos < _playlist->filenames()->size()) return true;
	return false;
}

uint LPPlaylistIterator::getSize() { if (_playlist->filenames()) return _playlist->filenames()->size(); else return 0; }

uint LPPlaylistIterator::getPosition() { return _currpos; }

float LPPlaylistIterator::songGoodahness(Song song) {
    if (song.getFilename().empty()) {
        logWarning(string("LPIterator::error while analyzing: song does not exist: ") + song.getFilename());
        return -1000;
    }
	if (isPlaylist(song.getFilename())) return m3uGoodahness(song.getFilename());
	else return _filter->goodahness(song);
}

bool LPPlaylistIterator::isPlaylist(string filename) {
	return M3UPlaylist::isM3U(filename);
}

float LPPlaylistIterator::m3uGoodahness(string filename) {
    logDebug("LPIterator::judging m3u " + filename);
    QFile f(QString::fromLocal8Bit(filename.c_str()));
    if (f.size() > 5000) {
        logDebug("LPIterator::m3u too big. reject.");
        return LPSongFilter::REJECT;
    }

	M3UPlaylist p;
	p.setM3U(filename);
	p.startBuilding();

	vector<string>* filenames = p.filenames();
    if (!filenames || filenames->empty()) return LPSongFilter::REJECT;

    int nrvalues = 0;
    float sumvalues =0;

    float currvalue = 0;
	int random = 0;

    for (int i=0;i<4;i++) {
		//find random song
        random = (int)(((filenames->size()-1) * absUtil::getUtil()->random()));//minimum 1
        Song s = _database->retrieveAndCorrect((*filenames)[random],"");
        currvalue = songGoodahness(s);
		if (currvalue < 0) {
		   logDebug("LPIterator::m3u: getting less attractive...");
			//bad is bad bad
			sumvalues += 2 * currvalue;
			nrvalues += 2;
		}
		else {
			logDebug("LPIterator::m3u:song ok!");
			//good is.. good
			sumvalues += currvalue;
			nrvalues += 1;
		}
    }
    logDebug(string("LPIterator::m3u: total value: ") + absUtil::dts(sumvalues/nrvalues));
    return (sumvalues/nrvalues);
}
