#include <qfile.h>
#include <qfileinfo.h>

#include "lpsong.h"
#include "lpglobal.h"

//pre: valid strings
Song::Song(string key) {
	setLike(0);
	setDate(0); //has never been listened to until one calls listen()
	setOldDate(0);
	setTimes(0);
	setName("");
	setFilename("");
	setKey(key);
	setColor(-1,-1,-1);
	setSize(0);
	setLength(0);
}

//pre: valid strings
Song::Song(string nname, string filename) {
	setLike(0);
	setDate(0); //has never been listened to until one calls listen()
	setOldDate(0);
	setTimes(0);
	setName(nname);
	setFilename(filename);
	setColor(-1,-1,-1);
	setSize(0);
	setLength(0);
}

//empty song
//fastest constructor
Song::Song() {
	_key= string("");
	_filename = string("");
	_name = string("");
	_like = 0;
 	_date = 0; _olddate=0;
	_times = 0;
	_colors = -1; _colorh = -1; _colorv = -1;
	_size = 0;
	_length = 0;
}

void Song::getColor(int* h, int* s, int* v) const {
	*h = _colorh; *s = _colors; *v = _colorv;
}

void Song::setColor(int h, int s, int v) { _colorh = h; _colors = s; _colorv = v; }
int Song::getTimes() const { return _times; }
void Song::setTimes(int times) { _times = times; }
string Song::getName() const { return _name; }
void Song::setName(string name) { _name = name; }

string Song::getKey() const { return _key; }

void Song::setKey(string key) {
	if (key.empty()) _key = "";
	else _key = absUtil::uppercase(key);
}

bool Song::empty() const { return ( getFilename().empty() || getKey().empty() ); }

bool Song::valid() const {
	if (empty()) return false;
	if (QFile::exists(QString::fromLocal8Bit(getFilename().c_str()))) return true;
	return false;
}

/** tries to detect inconsistensies (date vs olddate) */
void Song::repair() {
   //correct dates if necessary (strange...)
   if (getOldDate() > getDate()) {
       setDate(getOldDate());
       logWarning("LPSong::had to correct date!");
   }
}

/** certain that those songs are the same (not their dates, ratings, genres..)
 *  same songs could return false
 */
bool Song::equal(Song song) const {
	if (getName() != song.getName()) return false;
	if (getFilename() != song.getFilename()) return false;
	if (getKey() != song.getKey()) return false;
	return true;
}

/** fairly certain that those songs are the same (not their dates, ratings, genres..)
 *  different songs could return true
 */
bool Song::equivalent(Song song) const {
	//logDebug(getKey() + " vs " + song.getKey());
	if (getKey() != song.getKey()) return false;
	//logDebug(getName() + " vs " + song.getName());
	if (!getName().empty() && (getName() == song.getName())) return true;
	//logDebug(getFilename() + " vs " + song.getFilename());
	if (equivalentFilename(song.getFilename())) return true;
	return false;
}

bool Song::equivalentFilename(string filename) const {
	//TODOreplaceslashes, QT for linux does not recognise windows paths properly
	//filename1.replace( QRegExp("a.*a"), "" );
    string filename1 = absUtil::uppercase(getFilename());
    string filename2 = absUtil::uppercase(filename);
    if (filename1.empty() || filename2.empty()) return false;
	if (filename1 == filename2) return true;
     //potential conflict: full filename paths differ
	 QString file1, file2, dir1, dir2;
	 //assume it's still the same file if the filenames OR the paths are the same
	 file1 = QFileInfo(QString::fromLocal8Bit(filename1.c_str())).fileName();
	 file2 = QFileInfo(QString::fromLocal8Bit(filename2.c_str())).fileName();
	 dir1 = QFileInfo(QString::fromLocal8Bit(filename1.c_str())).dirPath();
	 dir2 = QFileInfo(QString::fromLocal8Bit(filename1.c_str())).dirPath();
	if (file1 == file2)
	{ logDebug(string("LPSong::song " + getFilename() + " has moved from ") + filename); return true; }
	if (dir1 == dir2)
	{ logDebug(string("LPSong::song " + getFilename() + " has been renamed from ") + filename); return true; }

	logDebug(string("LPSong::song " + getFilename() + " does not seem to relate to ") + filename);
	return false;
}
    
/** updates song stats if equal and argument is NEWER */
void Song::softmerge(Song song)
{
	if (!equivalent(song)) return;
//reasonably safe to assume songs are the same, so update stats
	if (getDate() <= song.getDate()) 
	{
//                  logDebug(string("LPDatabase::updating (a bit) ")  + old.name);
			setDate(song.getDate());
			setLike(song.getLike());
			setTimes(song.getTimes());
			//don't touch filename as they may differ on different machines
	}
	else
	{
		logDebug(string("LPSong::not storing ")  + getName() + " new " + absUtil::its(song.getDate()) + " difference in date " + absUtil::longTimeSpan(getDate() - song.getDate()) + ", new: " + absUtil::its(song.getDate()) + ", old: " + absUtil::its(getDate()));
	}
}

int Song::getDate() const { return _date; }
int Song::getOldDate() const { return _olddate; }
void Song::setDate(int newdate) { _date = newdate; }
void Song::setOldDate(int newdate) { _olddate = newdate; }
int Song::getLike() const { return _like; }

void Song::setLike(int newlike) {
	if (newlike > 5) newlike = 5;
	if (newlike < -5) newlike = -5;
	_like = newlike;
}

void Song::setSong(string filename) {
	setFilename(filename);
	if (!filename.empty()) {
		setKey(absUtil::getUtil()->getFileID(filename));
	}
	else {
		logWarning("LPSong::constructing empty song!");
	}
}

void Song::setLength(int length) { _length = length; }
int Song::length() const { return _length; }

void Song::setSize(int size) {
	_size = (size ? size : absUtil::getUtil()->getSize(getFilename()));
}

int Song::size() const {
	return (_size ? _size : absUtil::getUtil()->getSize(getFilename()));
}

string Song::getFilename() const { return _filename; }
void Song::setFilename(string filename) { _filename = filename; }

bool Song::firstTime() const {
	//return (!heard() || ( hearing() && (getTimes() < 2) ) );
	return (!heard() || ( (getTimes() < 2) ) );
}

bool Song::hearing() const {
	return (heard() && (absUtil::getUtil()->date() < length() + getDate()));
}

bool Song::heard() const { return (getTimes() > 0); }

void Song::resetListen() {
	if (_times) _times--;
	_date = _olddate;
	if (!_date) _times = 0;
	_olddate = 0;
}

void Song::listen() {
	_times++;
	_olddate = _date;
	if (_olddate == 0) _olddate = absUtil::getUtil()->date();
	_date = absUtil::getUtil()->date();
}

string Song::toString() const {
	return string(string("songkey: ") + getKey() + "/" + getName() + "/" +  getFilename() + "/ like: " + absUtil::its(getLike()) + "/ ago: " + absUtil::longTimeSpan(absUtil::getUtil()->date() - getDate()) + "/ color: " + absUtil::its(_colorh) + "." + absUtil::its(_colors) + "." + absUtil::its(_colorv)) ;
}
