/** LongPlayer - cross-platform multimedia queuer
 *  Copyright (C) 2001-2004 Andrew Wils
 *
 *  This program is free software; See the
 *  GNU General Public License for more details.
 */
#ifndef H_ABSPLAYER
#define H_ABSPLAYER

#include <iostream>
#include <vector>

#include <qthread.h>

#include "lpglobal.h"
#include "lplistener.h"
#include "lpsettings.h"

using namespace std;

/**
 * absPlayer.h: provides an abstract interface to a media player like Winamp or XMMS
 *
 */
class absPlayer : public QThread {
public:
	static IntSetting SettingPlaylistTrim;
	static IntSetting SettingPlaylistTrimVal;

	static absPlayer* getPlayer();
	static void setPlayer(absPlayer*);

	/** starts playing the file with the given filename
	 *  should do nothing when argument is invalid or not a media file
	 */
	virtual void play(vector<string> filenames)=0;
	/** like play, but only puts the file with the given filename in the playlist queue, does _not_ start playing it
	 */
	virtual void queue(vector<string> filename)=0;

	/** volume must be between 0 and 100 */
	virtual int getVolume()=0;
	virtual void setVolume(int)=0;

	/** play/pause the player */
	virtual void play()=0;
	/** typical controls... */
	virtual void next()=0;
	virtual void previous()=0;
	virtual void stop()=0;
	
			
	//TODOlow
	/** like queue, but this is instantly reflected in the inPlaylist and 
	virtual void queueVector(string filename)=0;*/

	/** returns the name of the song that is currently playing (if any)
	 *  returns an empty string if no song is playing
	 */
	virtual string getSongName() const;

	/** returns the length of the song that is currently playing (in seconds)
	 *  returns 0 if no song is playing
	 */
	virtual int getSongLength() const;
	virtual int getSongPosition() const=0;

	/** returns the full path to the file of the song that is currently playing
	 *  returns an empty string if no song is playing
	 */
	virtual string getFileName() const;

	/** returns true if the song with the given filename is in the playlist queue
	 */
	virtual bool inPlaylist(string filename) const=0;

	/** returns the position of the currently playing song in the playlist queue
	 */
	virtual int getPlaylistPosition() const=0;

	/** returns the number of entries in the playlist queue
	 */
	virtual int getPlaylistLength() const=0;

	/** tidies up the playlist (e.g. chops off the songs that have already been played) until smaller than argument
	 *  not mandatory to implement
	 */
	virtual void cleanPlayed(int newsize)=0;
	/** invokes cleanPlayed when necessary */
	void checkPlayed();
	
	/** removes all songs to be played so that newly queued songs will be heard immediately
	*/
	virtual void cleanQueued()=0;
	  
	/** subscribes the give LPListener to the following events:
	 *    - the current song changes
	 *    - nearing end of playlist
	 */
	virtual void subscribe(LPListener* lplayer);

	/** returns true if there is a song playing */
	virtual bool isPlaying() const=0;
	/** returns true if the player is running */
	virtual bool isRunning() const=0;
	/** < 20 seconds before end of playlist */
	virtual bool endOfPlaylist() const;

	/** starts the thread to monitor player events
	 */
	virtual void run();

	virtual ~absPlayer();
	void setSettings(LPSettings*);
	LPSettings* getSettings();

//TODOhigh move to private
	bool brun;
	bool brunning;
private:
	//notifies all subscribed LPListeners
	void notifyListeners();
	//see getSongName, but uncached
	virtual string getRSongName() const=0;
	//see getSongname, but uncached
	virtual int getRSongLength() const=0;
	//see getSongname, but uncached
	virtual string getRFileName() const=0;

	vector<LPListener*> _listeners;
	//cached values
	string currenttitle;
	int currentlength;
	string currentfilename;
	static absPlayer* _player;
	LPSettings* _settings;
};


#endif
