#include "lplayer.h"

#include <qdir.h>
#include <qfile.h>
#include <qdatastream.h>

#include "absplayer.h"
#include "filewriter.h"
#include "lpdatabase.h"
#include "lpplaylistiterator.h"
#include "lpsongfilter.h"
#include "queuebuilder.h"
#include "playlistreader.h"
#include "lpsettings.h"

IntSetting LPlayer::SettingEffort("lplayer","effort",5);

LPlayer::LPlayer(LPSettings* settings) : _settings(settings), _player(0), _database(0), _playlists(0), _queuebuilder(0), _builder(0), _queueList(0), _writer(0), _isRunning(false) {
}

LPlayer::~LPlayer() { }
void LPlayer::test() { }
bool LPlayer::ready() { return _isRunning; }

void LPlayer::init() {
	logInfo("Starting LPlayer component...");
	initing.lock();

	logDebug("LPlayer::I'm alive! instantiating modules..");
	_playlists = new PlaylistReader(getSettings());
	_database = new LPDatabase(getSettings());
	_player = absPlayer::getPlayer();
	_queuebuilder = new QueueBuilder(getSettings(), getDatabase(),getPlayer());
	logDebug("LPlayer::subscribing for player events..");
	
	_writer = new fileWriter(this);
	initing.unlock();
	_isRunning = true;
	logDebug("LPlayer startup done.");
}

void LPlayer::quit() {
	logDebug("LPlayer::shutting down.");
	logDebug("LPlayer::waiting for LPlayer to finish initialization..");
	initing.lock();
	logDebug("LPlayer::waiting for LPlayer to finish initialization. done.");
	if (_queueList) _queueList->stopBuilding();
	if (_database) delete _database; _database = 0;
	if (_writer) delete _writer; _writer = 0;
	initing.unlock();
	logDebug("LPlayer::disconnected.");
}

PlaylistReader* LPlayer::getPlaylists() const { return _playlists; }
LPDatabase* LPlayer::getDatabase() const { return _database; }
LPSettings* LPlayer::getSettings() const { return _settings; }
absPlayer* LPlayer::getPlayer() const { return _player; }
QueueBuilder* LPlayer::getQueueBuilder() { return _queuebuilder; }

bool LPlayer::shouldBuild() {
    CompositePlaylist* playlist = getPlaylists()->getPlaylist(getSettings()->getSetting("lplayer","playlist") );
    if (!_queueList || !_queueList->equals(playlist)) { 
		logDebug("LPlayer::cached playlist has become invalid."); 
		if (_queueList) delete _queueList;
		_queueList = playlist; 
	}
	else { delete playlist; }
    if (!_queueList->filenames() || !_queueList->filenames()->size()) return true;
    int effort = getSettings()->getSetting(SettingEffort);
    if (effort < 3) return false;
	//4 and 5 -> daily
    if (effort < 6) return (absUtil::getUtil()->date() - _queueListTimeStamp > (3600*24));
	//6 and 7 -> hourly
    if (effort < 8) return (absUtil::getUtil()->date() - _queueListTimeStamp > (3600));
	return true;    
}

//todo move to app
Playlist* LPlayer::getBuiltPlaylist() {
	if (shouldBuild()) {
		logBusy("Building list of files..",0);
		_queueList->startBuilding();
		_queueListTimeStamp = absUtil::getUtil()->date();
		logReady("Building list of files..");
		logInfo(string("Found ") + absUtil::its(_queueList->filenames()->size()) + " files.");
	}
	return _queueList;
}

Song LPlayer::currentSong() {
	change.lock();
		Song curr = _currsong;
	change.unlock();
		return curr;
}

//you know... to make it all official
void LPlayer::setCurrentSong(Song song) {
	change.lock();
		_currsong = song;
	change.unlock();
		//alert the GUI
		notifyListeners();
}

//what happens when LPlayer is notified of a new song
void LPlayer::notify(int /*i*/) {
	logDebug("LPlayer::notified about a new song!");
	if (!ready()) {
		logWarning("LPlayer::...but I'm not ready yet");
		return;
	}
	Song song;//empty song

	string filename = getPlayer()->getFileName();
	if (!filename.empty()) song = getDatabase()->retrieveAndCorrect(filename,getPlayer()->getSongName(),getPlayer()->getSongLength());
	
	if (!song.equivalent( currentSong() ) ) {
		logDebug(string("LPlayer::looks like you are playing yet another song! ") + song.getKey() + " (the last one was " + currentSong().getKey() + ")");
		getSettings()->add("stats", "nrplayed");
		if (!song.empty()) {
			song.listen();
			getDatabase()->store(song);
			setCurrentSong(song);//in case the song details changed or a different song is playing (this will also notify the gui)
		}
	}
	else {
		//song could be same but title/filename may have changed
		if (!song.equal(currentSong())) {
			getDatabase()->store(song);
		}
		setCurrentSong(song);//display song changes
	}
	if (!getSettings()->getISetting("database","nosaveafterchange")) getDatabase()->flush();//do something else
	logDebug("LPlayer::notified by mediaplayer. all done.");
}

void LPlayer::subscribe(LPListener* songlistener) {
	_listeners.push_back(songlistener);
}

void LPlayer::notifyListeners() {
	for (uint i=0;i<_listeners.size();i++)
	_listeners[i]->notify();
}
