#include "lpsettings.h"
#include "absutil.h"

#ifdef Q_OS_MACX
#include <Carbon.h>
#endif

#include <qtextstream.h>
#include <qfile.h>
#include <qdir.h>
#include <qregexp.h>
#include <qstringlist.h>
#include <qdatetime.h>
#include <qmap.h>

LPSettings* LPSettings::_singleton=0;

/* important: lpsettings cannot rely on util at construction time */
LPSettings::LPSettings() {
   	logDebug("LPSettings::creating settings...");
	_singleton = this;
	_resourceDir = "";
#ifdef Q_OS_MACX
    CFURLRef pluginRef = CFBundleCopyBundleURL(CFBundleGetMainBundle());
    CFStringRef macPath = CFURLCopyFileSystemPath(pluginRef, kCFURLPOSIXPathStyle);
    _resourceDir = string(CFStringGetCStringPtr(macPath, CFStringGetSystemEncoding())) + "/Contents/Resources/";
#else
    if (QDir("../share/lplayer").exists()) 
        _resourceDir = "../share/lplayer/";
    else
        _resourceDir = RESOURCE_DIR;
#endif
	string filename = getLPHomeDir() + "/settings.xml";
 	_rtdoc = QDomDocument( "mydocumentrt" );
    QDomElement sel = _rtdoc.createElement("settings");
    _rtdoc.appendChild (sel);
	loadSettings(filename);
	setSetting("settings","location",filename);
    init();
}

void LPSettings::revert() {
	string filename;

	if (testSetting("settings","location")) { filename = getSetting("settings","location"); }
	else { filename = getLPHomeDir() + "/settings.xml"; }

	loadSettings(filename);
	setSetting("settings","location",filename);
    init();
}

LPSettings::LPSettings(string ini) {
	loadSettings(ini);
	setSetting("settings","location",ini);
	init();
}

LPSettings::~LPSettings() { }

void LPSettings::checkInit() {
	//legacy..
	logDebug("LPSettings::checking settings.");
	if (!absUtil::getUtil()->existsFile(getLPHomeDir() + "/settings.xml")  && absUtil::getUtil()->existsFile(getLPHomeDir() + "/../.lpsettings.xml")) {
		absUtil::getUtil()->copyFile(getLPHomeDir() + "/../.lpsettings.xml", getLPHomeDir() + "/settings.xml");
		revert();//reload settings from new file and continue..
	}
	if (!absUtil::getUtil()->existsFile(getLPHomeDir() + "/songs.lp2")  && absUtil::getUtil()->existsFile(getLPHomeDir() + "/../.lptable.lp2")) {
		absUtil::getUtil()->copyFile(getLPHomeDir() + "/../.lptable.lp2", getLPHomeDir() + "/songs.lp2");
		setSetting("database","location",getLPHomeDir() + "/songs.lp2");
	}

	if (!absUtil::getUtil()->existsFile(getLPHomeDir() + "/playlists.xml")  && absUtil::getUtil()->existsFile(getLPHomeDir() + "/../.lpplaylists.xml")) { 
		absUtil::getUtil()->copyFile(getLPHomeDir() + "/../.lpplaylists.xml", getLPHomeDir() + "/playlists.xml");
		setSetting("playlists","location",getLPHomeDir() + "/playlists.xml");
	}
}

void LPSettings::init() {
	checkSetting("playlistactions","lastdir", string(QDir::homeDirPath().local8Bit()));
	checkSetting("fileactions","lastdir", string(QDir::homeDirPath().local8Bit()));
	checkSetting("searchactions","lastdir", string(QDir::homeDirPath().local8Bit()));
	checkSetting("database","location",getLPHomeDir()+"/songs.lp2");
	//legacy.. :(  old lplayer versions put their files in the root directory of drive C
	if (QString(getSetting("database","location").c_str()).lower() == "c://.lptable.lp2")
		setSetting("database","location",getLPHomeDir()+"/songs.lp2");
    checkSetting("playlists","location",getLPHomeDir() + "/playlists.xml");
	if (QString(getSetting("playlists","location").c_str()).lower() == "c://.lpplaylists.xml")
		setSetting("playlists","location",getLPHomeDir()+"/playlists.xml");
	checkSetting("remote","location",getLPHomeDir() + "/remote.txt");
    checkISetting("main", "nogui",0);
}

void LPSettings::checkFSetting(string category,string name,float val) { if (!testSetting(category,name)) setFSetting(category,name,val); }
void LPSettings::checkISetting(string category,string name,int val) { if (!testSetting(category,name)) setISetting(category,name,val); }
void LPSettings::checkSetting(string category,string name,string val) { if (!testSetting(category,name)) setSetting(category,name,val); } 
string LPSettings::getResourceDir() { return _singleton->_resourceDir; }

string LPSettings::getLPHomeDir() {
	//return the working directory if we cannot obtain the homedir
	if (QDir::homeDirPath().upper() == QDir::rootDirPath().upper() ||
	       QDir::homeDirPath().upper() == "C:/")
	return ".";

	QDir dir(QDir::homeDirPath());
	if (!dir.cd(".lplayer")) {
		dir.mkdir(".lplayer") ; 
		dir.cd(".lplayer"); 
		}
	return string(dir.path().local8Bit());
}

void LPSettings::loadSettings(string ini) {
    logDebug("LPSettings::loading settings.");
    _doc = QDomDocument( "mydocument" );
    QFile f( ini.c_str() );
    if ( !f.open( IO_ReadOnly ) ) {
        logWarning(string("LPSettings::settings file ") + ini + " cannot be opened. creating default.");
        QDomElement sel = _doc.createElement("settings");
        _doc.appendChild (sel);
    }
    else if ( !_doc.setContent( &f ) ) {
        QDomElement sel = _doc.createElement("settings");
        _doc.appendChild (sel);
        logWarning(string("LPSettings::settings file ") + ini + " containts invalid XML. erasing.");
        logError(string("Your settings (") + ini + ") contained invalid syntax and have been reinitialised.");
    }
    if (f.exists()) f.close();
    logDebug("LPSettings::loading settings. done.");
}

void LPSettings::save() {
	string ini = getSetting("settings","location");
    logDebug("LPSettings::saving settings.");
    QFile f(ini.c_str());
    if (!f.open(IO_WriteOnly)) {
        logWarning(string("LPSettings::writing settings ") + ini + " failed.");
        logError(string("Could not save settings into file ") + ini + ". Please check file permissions and free space.");
        return;
    }
    QTextStream t( &f );
    t.setEncoding(QTextStream::UnicodeUTF8);
    t << _doc.toString();
    f.close();
}

void LPSettings::setSetting(StringSetting type, string value, bool persist) { setSetting(type.getCategory(), type.getName(), value, persist); }
void LPSettings::setSetting(IntSetting type, int value, bool persist) { setISetting(type.getCategory(), type.getName(), value, persist); }
void LPSettings::setSetting(FloatSetting type, float value, bool persist) { setFSetting(type.getCategory(), type.getName(), value, persist); }

void LPSettings::setSetting(string category, string name, string value, bool persist) {
    QDomDocument* docp;
    if (persist) docp = &_doc;
    else docp = &_rtdoc;

    QDomDocument & doc = *docp;
	QDomNodeList stuff;
  	stuff = doc.elementsByTagName (string(category + name).c_str()) ;
  	QDomElement node;
  	if (!stuff.length()) {
    	logDebug(string("LPSettings::creating new setting ") + category + name);
    	node = doc.createElement ( string(category + name).c_str());
    	doc.documentElement ().appendChild (node);
  	}
	else node = stuff.item(0).toElement();;

	node.setAttribute("value",QString::fromLocal8Bit(value.c_str()));
}

void LPSettings::setISetting(string category, string name, int value, bool persist) {
	QDomDocument* docp;
	if (persist) docp = &_doc;
    else docp = &_rtdoc;

	QDomDocument & doc = *docp;
    QDomNodeList stuff;
    stuff = doc.elementsByTagName (string(category + name).c_str()) ;
   	QDomElement node;
  	if (!stuff.length()) {
		logDebug(string("LPSettings::creating new setting ") + category + name);
		node = doc.createElement ( string(category + name).c_str());
		doc.documentElement ().appendChild (node);
  	}
  	else node = stuff.item(0).toElement();;

  	node.setAttribute("value",value);
}

void LPSettings::setFSetting(string category, string name, float value, bool persist) {
    QDomDocument* docp;
    if (persist) docp = &_doc;
    else docp = &_rtdoc;

    QDomDocument & doc = *docp;
    QDomNodeList stuff;
    stuff = doc.elementsByTagName (string(category + name).c_str()) ;
	QDomElement node;
	if (!stuff.length()) {
		logDebug(string("LPSettings::creating new setting ") + category + name);
		node = doc.createElement ( string(category + name).c_str());
		doc.documentElement ().appendChild (node);
	}
	else node = stuff.item(0).toElement();;

	node.setAttribute("value",value);
}

bool LPSettings::testSetting(StringSetting type) { return testSetting(type.getCategory(), type.getName()); }
bool LPSettings::testSetting(IntSetting type) { return testSetting(type.getCategory(), type.getName()); }
bool LPSettings::testSetting(FloatSetting type) { return testSetting(type.getCategory(), type.getName()); }

bool LPSettings::testSetting(string category, string name) {
	QDomNodeList setting = _rtdoc.elementsByTagName (string(category + name).c_str()) ;
	if (!setting.length()) setting = _doc.elementsByTagName (string(category + name).c_str()) ;
	if (!setting.length()) return false;
	return true;
}

string LPSettings::getSetting(StringSetting type) { 
	if (!testSetting(type)) return type.getDefault(); 
	return getSetting(type.getCategory(), type.getName()); 
}

int LPSettings::getSetting(IntSetting type) { 
	if (!testSetting(type)) return type.getDefault(); 
	return getISetting(type.getCategory(), type.getName()); 
}

float LPSettings::getSetting(FloatSetting type) { 
	if (!testSetting(type)) return type.getDefault(); 
	return getFSetting(type.getCategory(), type.getName());
}

string LPSettings::getSetting(string category, string name) {
	QDomNodeList setting = _rtdoc.elementsByTagName (string(category + name).c_str()) ;
	if (!setting.length()) setting = _doc.elementsByTagName (string(category + name).c_str()) ;
	if (!setting.length())  return ""; 
	QCString value = setting.item(0).toElement().attribute("value","0").local8Bit();
	if (value.isEmpty()) return "";
	return string(value);
}

int LPSettings::getISetting(string category, string name) {
  QDomNodeList setting = _rtdoc.elementsByTagName (string(category + name).c_str()) ;
  if (!setting.length()) setting = _doc.elementsByTagName (string(category + name).c_str()) ;
  if (!setting.length()) return 0; 
  return setting.item(0).toElement().attribute("value","0").toInt();
}

float LPSettings::getFSetting(string category, string name) {
  QDomNodeList setting = _rtdoc.elementsByTagName (string(category + name).c_str()) ;
  if (!setting.length()) setting = _doc.elementsByTagName (string(category + name).c_str()) ;
  if (!setting.length()) return 0;
  return setting.item(0).toElement().attribute("value","0").toFloat();
}

void LPSettings::add(string cat, string name, int value) {
    setISetting(cat,name,getISetting(cat, name) + value);
}
