#include "lpsongfilter.h"
#include "lpglobal.h"
#include <qstringlist.h>
#include <qfileinfo.h>

const int LPSongFilter::REJECT = -500;
const int LPSongFilter::DISREC = -400;
const int LPSongFilter::NORMAL = 0;
const int LPSongFilter::DELTA = 50;

LPDummyFilter::LPDummyFilter() {}
LPDistanceFilter::LPDistanceFilter(vector<string>* history) { _history = history; }
LPAddFilter::LPAddFilter(LPSongFilter* f1, float w1, LPSongFilter* f2, float w2) : _f1(f1), _f2(f2),_w1(w1),_w2(w2) { }
LPAndFilter::LPAndFilter(LPSongFilter* f1, LPSongFilter* f2) : _f1(f1), _f2(f2) { }
LPOrFilter::LPOrFilter(LPSongFilter* f1, LPSongFilter* f2) : _f1(f1), _f2(f2) { }
LPRatingFilter::LPRatingFilter(int type, int val, bool strict) : _type(type), _val(val), _strict(strict) { }
LPLengthFilter::LPLengthFilter(int type, int val) : _type(type), _val(val) { }
LPSecondsAgoFilter::LPSecondsAgoFilter(int type, int val) : _type(type), _val(val) { }
LPTimesFilter::LPTimesFilter(int type, int val) : _type(type), _val(val) { }
LPColorFilter1::LPColorFilter1(int h, int s, int v, int hdelta, int sdelta, int vdelta, bool drift) : _h(h), _s(s),_v(v),_hdelta(hdelta),_sdelta(sdelta),_vdelta(vdelta),_drift(drift) { }
LPColorFilter2::LPColorFilter2(int x1, int y1, int x2, int y2) : _x1(x1), _y1(y1), _x2(x2), _y2(y2) { 
	int temp;
	if (x1 > x2) { temp = x1; x1 = x2; x2 = temp; }
	if (y1 > y2) { temp = y1; y1 = y2; y2 = temp; }
}

LPDummyFilter::~LPDummyFilter() {}
LPDistanceFilter::~LPDistanceFilter() {}
LPColorFilter1::~LPColorFilter1() {}
LPColorFilter2::~LPColorFilter2() {}
LPAddFilter::~LPAddFilter() { delete _f1; delete _f2; }
LPAndFilter::~LPAndFilter() { delete _f1; delete _f2; }
LPOrFilter::~LPOrFilter() { delete _f1; delete _f2; }

void LPDummyFilter::confirm(const Song& /*song*/) {}
void LPDistanceFilter::confirm(const Song& /*s*/) { }
void LPAddFilter::confirm(const Song& s ) { _f1->confirm(s); _f2->confirm(s); }
void LPAndFilter::confirm(const Song& s ) { _f1->confirm(s); _f2->confirm(s); }
void LPOrFilter::confirm(const Song& s ) { _f1->confirm(s); _f2->confirm(s); }
void LPRatingFilter::confirm(const Song& /*song*/) {}
void LPLengthFilter::confirm(const Song& /*song*/) {}
void LPSecondsAgoFilter::confirm(const Song& /*song*/) {}
void LPTimesFilter::confirm(const Song& /*song*/) {}
void LPColorFilter1::confirm(const Song& s) { if (_drift) s.getColor(&_h,&_s,&_v); }
void LPColorFilter2::confirm(const Song& /* s */) { }

float LPDummyFilter::goodahness(const Song& /*song*/) { return 1; }

float LPDistanceFilter::goodahness(const Song& song) {
	logDebug(string("    DISTANCE-goodahnessing with history of ") + absUtil::its(_history->size()));
	if (_history->empty()) return 0 + NORMAL;

	QString filename1 = QString::fromLocal8Bit(song.getFilename().c_str());
	char delimiter1; 
	if (filename1.contains("/")) delimiter1 = '/';
	else delimiter1 = '\\';
	QStringList dirs1 = QStringList::split( delimiter1,filename1,FALSE);

	//calc directory distance of each song
	for(uint i=0;i<_history->size();i++) {
		QString filename2 = QString::fromLocal8Bit((*_history)[i].c_str());
		if (filename1 == filename2) return REJECT; //apparently we queued the song before
		if (QFileInfo(filename1).dirPath() == QFileInfo(filename2).dirPath()) return NORMAL - DELTA;
	}
	return NORMAL;
}

float LPColorFilter1::goodahness(const Song& song) {
		int h,s,v;
		song.getColor(&h, &s, &v);
		
		float diff, refdiff, level;
		float val=0;
		float maxdiff = 256;
		level = NORMAL;

		//general idea:
		// if we are within 2*refdiff the interval must be reversed and between - refdiff and refdiff + map

		diff = h - _h;//always positive difference
		refdiff = _hdelta;//always positive
		if (diff < 0) diff *= -1; if (refdiff < 0) refdiff *= -1;

		if (diff < (2 * refdiff)) { val += (refdiff + (-1*diff)) * DELTA/refdiff ; }
		else { level = DISREC; val += (maxdiff/2 + (-1 * diff)) * DELTA/(maxdiff/2); }

		diff = s - _s;//always positive difference
		refdiff = _sdelta;//always positive
		if (diff < 0) diff *= -1; if (refdiff < 0) refdiff *= -1;

		if (diff < (2 * refdiff)) { val += (refdiff + (-1*diff)) * DELTA/refdiff ; }
		else { level = DISREC; val += (maxdiff/2 + (-1 * diff)) * DELTA/(maxdiff/2); }

		diff = v - _v;//always positive difference
		refdiff = _vdelta;//always positive
		if (diff < 0) diff *= -1; if (refdiff < 0) refdiff *= -1;

		if (diff < (2 * refdiff)) { val += (refdiff + (-1*diff)) * DELTA/refdiff ; }
		else { level = DISREC; val += (maxdiff/2 + (-1 * diff)) * DELTA/(maxdiff/2); }

		val = level + val/3;
		logDebug(string("    COLOR-goodahnessing"));
		return val;
}

float LPColorFilter2::goodahness(const Song& song) {
		//pathological case = no constraint
		if ((_x1 == _x2) && (_y1 == _y2)) return NORMAL;

		int x,y,z;
		song.getColor(&x, &y, &z);
		logDebug(string("    COLOR-goodahnessing ") + absUtil::its(_x1) + "<" + absUtil::its(x) + "<" + absUtil::its(_x2));

		if (x < _x1) return DISREC;
		if (x > _x2) return DISREC;
		if (y < _y1) return DISREC;
		if (y > _y2) return DISREC;
		return NORMAL;
}

//returns REJECT if one of values is REJECT
//return value is again normalized
float LPAddFilter::goodahness(const Song& song) {
		//TODO also weigh values if not NORMAL
		float g1 = _f1->goodahness(song);
		if (g1 <= REJECT + DELTA ) return g1;
		if (g1 <= DISREC + DELTA) return g1;
		float g2 = _f2->goodahness(song);
		if (g2 <= REJECT + DELTA) return g2;
		if (g2 <= DISREC + DELTA) return g2;
		float val =  ((g1*_w1) + (g2*_w2))/(_w2+_w1);
		logDebug(string("    ADD-goodahnessing result (normalized val*weight):"));
		logDebug(string("    ") + absUtil::fts(g1) + string(" * ") + absUtil::fts(_w1) + " + " + absUtil::fts(g2) + " * " + absUtil::fts(_w2) + " = " + absUtil::fts(val));
		return val;
}

float LPAndFilter::goodahness(const Song& song)
{
		float val;
		float g1 = _f1->goodahness(song);
		float g2 = _f2->goodahness(song);
		g1 < g2 ? val = g1 : val = g2;
		return val;
}

float LPOrFilter::goodahness(const Song& song) {
		float val;
		float g1 = _f1->goodahness(song);
		float g2 = _f2->goodahness(song);
		g1 > g2 ? val = g1 : val = g2;
		return val;
}

float LPRatingFilter::goodahness(const Song& song) {
		if (song.getLike() == -5) {
			logDebug(string("    RATING-goodahnessing rating ") + absUtil::its(song.getLike()) + " against: " + absUtil::its(_val) + "; result: Reject!");
			return REJECT;// 0/10 is the Ultimate Bad Song
		}

		float result;
		float level = NORMAL;
		int diff =  song.getLike() - _val;
		
		int max=10;
		
		switch (_type) {
		case EQUAL:
			if (_strict && diff) level = DISREC;
			if (diff < 0) diff *= -1;
			result = (-1 * diff) * (DELTA/max); //make diff negative
			break;
		case MORE://diff should be positive
			if (_strict && (diff<=0)) level = DISREC;
			result = (diff) * (DELTA/max); 
			break;
		case LESS:
			if (_strict && (diff>=0)) level = DISREC;
			result = (-1 * diff) * (DELTA/max); 
			break;
		}
		result += level;
		logDebug(string("    RATING-goodahnessing rating ") + absUtil::its(song.getLike()) + " against: " + absUtil::its(_val) + "; result: " + absUtil::fts(result));
		return result;
}

float LPTimesFilter::goodahness(const Song& song)
{
	int _strict = true;
		float result;
		float level = NORMAL;
		int diff =  song.getTimes() - _val;
		
		switch (_type)
		{
		case EQUAL:
			if (_strict && diff) level = DISREC;
			result = (-1 * diff * diff) * (DELTA/100); //make diff negative
			break;
		case MORE://diff should be positive
			if (_strict && (diff<=0)) level = DISREC;
			result = (diff) * (DELTA/10); 
			break;
		case LESS:
			if (_strict && (diff>=0)) level = DISREC;
			result = (-1 * diff) * (DELTA/10); 
			break;
		}
		result += level;
		logDebug(string("    TIMES-goodahnessing times ") + absUtil::its(song.getTimes()) + " against: " + absUtil::its(_val) + "; result: " + absUtil::fts(result));
		return result;
}

//when never heard and not EQUAL: 0
float LPLengthFilter::goodahness(const Song& song)
{
	int _strict = false;//for now not used in a strict way
//TODO add DISREC when difference is large
		float result;
		float level = NORMAL;
		if (!song.length()) return 0;
		int diff =  song.length() - _val;
		
		int max=3600;
		if (diff > max) diff = max;
		if (diff < (-1 * max)) diff = -1 * max;

		switch (_type)
		{
		case EQUAL:
			if (_strict && diff) level = DISREC;
			if (diff < 0) diff *= -1;
			result = (-1 * diff) * (DELTA/max); //make diff negative
			break;
		case MORE://diff should be positive
			if (_strict && (diff<0)) level = DISREC;
			result = (diff) * (DELTA/max); 
			break;
		case LESS:
			if (_strict && (diff>0)) level = DISREC;
			result = (-1 * diff) * (DELTA/max); 
			break;
		}
		result += level;
		logDebug(string("    LENGTH-goodahnessing length ") + absUtil::shortTimeSpan(song.length()) + " against: " + absUtil::shortTimeSpan(_val) + "; result: " + absUtil::fts(result));
		return result;
}

//when never heard and MORE: +DELTA
float LPSecondsAgoFilter::goodahness(const Song& song)
{
	int _strict = true;
	float result;
	float level = NORMAL;
	int currdate = absUtil::getUtil()->date();
    int songdate = (song.getDate()) ? song.getDate() : song.getOldDate();
	
	int interval = currdate - songdate; //currdate > songdate
	float diff = interval - _val;
    if ((!song.heard()) && (_type == LPSongFilter::MORE)) 
    {
		logDebug(string("    SECONDSAGO-goodahnessing ") + absUtil::longTimeSpan(interval) + " against: " + absUtil::longTimeSpan(_val) + "; result: song not heard before, returning standard result.");
   		return DELTA;
	}
	if ((!song.heard()) && (_type == LPSongFilter::LESS)) 
	{
		logDebug(string("    SECONDSAGO-goodahnessing ") + absUtil::longTimeSpan(interval) + " against: " + absUtil::longTimeSpan(_val) + "; result: song not heard before, returning standard result.");
	    return DISREC;
	}
	//max = 100 weeks
	float max = 3600 * 24 * 700;
	if (diff > max) diff = max; //constrain diff
	if (diff < (-1*max)) diff = -1 * max;

	switch (_type)
	{
	case EQUAL: //not very useful
		if (diff > 0) diff *= -1; //pos diff is also bad
		if (_strict && diff) level = DISREC;
		result = (diff) * (DELTA/max); 
		break;
	case MORE://diff should be positive
		if (_strict && (diff<=0)) level = DISREC;
		result = (diff) * (DELTA/max); 
		break;
	case LESS:
		if (_strict && (diff>=0)) level = DISREC;
		result = (-1 * diff) * (DELTA/max); 
		break;
	}
	result += level;
	logDebug(string("    SECONDSAGO-goodahnessing ") + absUtil::longTimeSpan(interval) + " against: " + absUtil::longTimeSpan(_val) + "; result: " + absUtil::fts(result));
	return result;
}
