/** LongPlayer - cross-platform multimedia queuer
 *  Copyright (C) 2001-2004 Andrew Wils
 *
 *  This program is free software; See the
 *  GNU General Public License for more details.
 */
#ifndef H_LPSONGFILTER
#define H_LPSONGFILTER

#include "lpsong.h"
#include <vector>

class LPSettings;

class LPSongFilter {
	public:
		/* Song Filter principles
		 * a song filter returns an integer value, indicating the level it has passed 
		 * the specified filter
		 *
		 * a filter can return one of three levels, each with a fine-tuning subscore 
		 * reject: according to the criteria the song is completely unacceptable
		 * disrec: the song did not pass the criteria
		 * normal: up to a certain level, the song passed the criteria
		 *
		 * the subscore should remain within the range: level - DELTA < subscore < level + delta
		 * 
		 * possible uses:
		 * combining scores can be normalized so that a score still remains in the same (lowest) level
		 * queuebuilders can choose to accept disrecommended songs or not, or decide to allocate more searching effort
		 *
		 */	
		static const int REJECT;//unacceptable
		static const int DISREC;//should keep looking
		static const int NORMAL;//adequate
		static const int DELTA;

		/** how good the song is according to the filter */
		virtual float goodahness(const Song& s)=0;
		/** let this filter know that the song has been chosen 
		 *  possible uses: adjust average, update statistics
		 */
		virtual void confirm(const Song& s)=0;

		enum filtervals {
			        MORE = 1, 
			        LESS = 2,
			        EQUAL = 3
			    };
			   
};

/** 
 * Filter that measures and penalizes file system distance
 * e.g. /foo/bar/song1.mp3 vs /fred/flintstone/song2.mp3
 * it compares this with the history 
 */
class LPDistanceFilter : public LPSongFilter {
	public:
		LPDistanceFilter(vector<string> *history);
		virtual ~LPDistanceFilter();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		vector<string>* _history;
};

class LPDummyFilter : public LPSongFilter {
	public:
		LPDummyFilter();
		virtual ~LPDummyFilter();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
};

/* filters songs not in the given rectangle */
class LPColorFilter2 : public LPSongFilter {
	public:
		LPColorFilter2(int x1, int y1, int x2, int y2);
		virtual ~LPColorFilter2();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		 int _x1,_y1,_x2,_y2;
};

/* filters songs not close enough to a point */
class LPColorFilter1 : public LPSongFilter {
	public:
		LPColorFilter1(int h, int s, int v, int hdelta, int sdelta, int vdelta, bool drift=false);
		virtual ~LPColorFilter1();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		 int _h,_s,_v;
		 int _hdelta,_sdelta,_vdelta;
		 bool _drift;
};
class LPAddFilter : public LPSongFilter
{
	public:
		LPAddFilter(LPSongFilter* f1, float weight1,LPSongFilter* f2, float weight2);
		virtual ~LPAddFilter();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		LPSongFilter *_f1,*_f2;
		float _w1,_w2;
};
class LPOrFilter : public LPSongFilter
{
	public:
		LPOrFilter(LPSongFilter* f1, LPSongFilter* f2);
		virtual ~LPOrFilter();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		LPSongFilter *_f1,*_f2;
};
class LPAndFilter : public LPSongFilter
{
	public:
		LPAndFilter(LPSongFilter* f1, LPSongFilter* f2);
		virtual ~LPAndFilter();
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		LPSongFilter *_f1,*_f2;
};

class LPRatingFilter : public LPSongFilter
{
	public:
		LPRatingFilter(int type,int val, bool strict=false);
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		int _type, _val, _strict;
};
class LPSecondsAgoFilter : public LPSongFilter
{
	public:
		LPSecondsAgoFilter(int type,int val);
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		int _type; int _val;
};
class LPLengthFilter : public LPSongFilter
{
	public:
		LPLengthFilter(int type,int val);
		float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		int _type; int _val;
};

class LPTimesFilter : public LPSongFilter
{
	public:
		LPTimesFilter(int type,int val);
		 float goodahness(const Song& s);
		virtual void confirm(const Song& s);
	private:
		int _type; int _val;
};

#endif
