#include "playlistreader.h"

#include "absutil.h"
#include "lpsettings.h"
#include "directoryplaylist.h"
#include "m3uplaylist.h"

#include <qtextstream.h>
#include <qfile.h>
#include <qdir.h>
#include <qregexp.h>
#include <qstringlist.h>

PlaylistReader::PlaylistReader(LPSettings* settings) : _settings(settings) {
	loadPlaylists();
}

PlaylistReader::~PlaylistReader() {
}

LPSettings* PlaylistReader::getSettings() { return _settings; }
void PlaylistReader::revert() { loadPlaylists(); }

void PlaylistReader::loadPlaylists() {	
    _playlistdoc = QDomDocument( "myplaylists" );
   	logDebug("PlaylistReader::readerizing playlists-thingies...");
    string playlistfilename = getSettings()->getSetting("playlists","location");
    QFile file(QString::fromLocal8Bit(playlistfilename.c_str()) );
    if ( !file.exists() || !file.open( IO_ReadOnly ) ) {
        logWarning(string("PlaylistReader::playlists file ") + playlistfilename + " could not be loaded. creating default.");
        QDomElement pel = _playlistdoc.createElement("playlists");
        _playlistdoc.appendChild (pel);
    }
    else if ( !_playlistdoc.setContent( &file ) ) {
        logError(string("Your playlists (") + playlistfilename + ") contained invalid syntax and have been reinitialised.");
        logWarning(string("PlaylistReader::playlists file ") + playlistfilename + " contains invalid XML. erasing.");
        _playlistdoc = QDomDocument( "myplaylists" );
        QDomElement pel = _playlistdoc.createElement("playlists");
        _playlistdoc.appendChild (pel);
    }
    if (file.exists()) file.close();
    logDebug("PlaylistReader::loading playlists. done.");
}

void PlaylistReader::save() {
	string ini = getSettings()->getSetting("settings","location");
    logDebug("PlaylistReader::saving playlists.");
    string playlistfilename = getSettings()->getSetting("playlists","location");

    QFile file(QString::fromLocal8Bit(playlistfilename.c_str()));
    if (!file.open(IO_WriteOnly)) {
        logError(string("Could not save playlists into file ") + playlistfilename + ". Please check file permissions and free space.");
        logWarning(string("PlaylistReader::writing playlists ") + playlistfilename + " failed.");
        return;
    }
    QTextStream t( &file );
    t.setEncoding(QTextStream::UnicodeUTF8);
    t << _playlistdoc.toString();
    file.close();
    logDebug("PlaylistReader::saving playlists. done.");
}

void PlaylistReader::removePlaylist(string playlist) {
    QDomNodeList stuff = _playlistdoc.elementsByTagName ("playlist") ;

    QDomElement node; QDomAttr att;
	bool found = false;
    for (uint i=0;!found && (i<stuff.length()) ;i++) {
        node = stuff.item(i).toElement();
        att = node.attributes().namedItem(QString::fromLocal8Bit("name")).toAttr();
        logDebug(string(att.value().local8Bit()));
        if (att.value() == QString::fromLocal8Bit(playlist.c_str())) {
			found = true;
        	_playlistdoc.documentElement().removeChild(stuff.item(i));
		}
    }
}

void PlaylistReader::setPlaylist(CompositePlaylist* cp) {
    QDomNodeList stuff = _playlistdoc.elementsByTagName ("playlist") ;

	DirectoryPlaylist playlist = *((DirectoryPlaylist*) cp->getPlaylists()[0]); 

	//remove existing entry
	QDomElement node; QDomAttr att;
	bool found = false;
	for (uint i=0;!found && (i<stuff.length())  ;i++) {
		node = stuff.item(i).toElement();
		att = node.attributes().namedItem(QString::fromLocal8Bit("name")).toAttr();
		if (att.value() == QString::fromLocal8Bit(cp->getName().c_str())) found = true;
	}
    if (found) {
		_playlistdoc.documentElement().removeChild(node);
    }

	//create new entry
    node = _playlistdoc.createElement ( "playlist");
	node.setAttribute("name",QString::fromLocal8Bit(cp->getName().c_str()));
    QDomElement dnode;

	//with dirs
    vector<string> dirs = playlist.getDirs();
    vector<bool> incldirs = playlist.getIncludeSubDirs();
    for (uint j=0;j<dirs.size();j++) {
        dnode = _playlistdoc.createElement ("dir");
        dnode.setAttribute("path",QString::fromLocal8Bit(dirs[j].c_str()));
        dnode.setAttribute("includesubdir",incldirs[j]);
        node.appendChild(dnode);
    }
	//with m3us
	for (uint m=1;m<cp->getPlaylists().size();m++) {
        dnode = _playlistdoc.createElement ("m3u");
        dnode.setAttribute("path", QString::fromLocal8Bit(((M3UPlaylist*)(cp->getPlaylists()[m]))->getM3U().c_str()));
        node.appendChild(dnode);
	}

	//with masks
	vector<string> filenamesEnd = cp->getFilenamesEndWith();
	string mask = ""; 
    dnode = _playlistdoc.createElement ("mask");
	for (uint k=0;k<filenamesEnd.size();k++) {
		mask = mask +  string("*.") + filenamesEnd[k];   
		if (k != filenamesEnd.size()-1) mask = mask + ";";
	}
    dnode.setAttribute("extension",QString::fromLocal8Bit(mask.c_str()));
    node.appendChild(dnode);

   string emask = "";
   vector<string> filenamesDoNotEnd = cp->getFilenamesDoNotEndWith();
	for (uint l=0;l<filenamesDoNotEnd.size();l++) {
		emask = emask +  string("*.") + filenamesDoNotEnd[l];
		if (l != filenamesDoNotEnd.size()-1) emask = emask + ";";
	}
    dnode = _playlistdoc.createElement ("excludemasks");
    dnode.setAttribute("value",QString::fromLocal8Bit(emask.c_str()));
    node.appendChild(dnode);

	//TODO save contains

    _playlistdoc.documentElement ().appendChild (node);
}

CompositePlaylist* PlaylistReader::getPlaylist(string name) {
    QDomNodeList stuff = _playlistdoc.elementsByTagName ("playlist") ;

    QDomElement node; QDomAttr att;
	bool found = false;
	for (uint i=0;!found && (i<stuff.length())  ;i++) {
        node = stuff.item(i).toElement();
        att = node.attributes().namedItem(QString::fromLocal8Bit("name")).toAttr();
        logDebug(string(att.value().local8Bit()));
        if (att.value() == QString::fromLocal8Bit(name.c_str())) found = true;
    }
    if (!found) {
      logWarning(string("PlaylistReader::playlist ") + name + " does not exist.");
      return new CompositePlaylist("dummy");
    }
	CompositePlaylist* cp = new CompositePlaylist(name);
    DirectoryPlaylist* pr = new DirectoryPlaylist(name);
   //dirs
    QDomNodeList dirs = node.elementsByTagName ("dir") ;
    for (uint j=0;j<dirs.count();j++) {
          pr->addDir(
              string(dirs.item(j).toElement().attribute("path").local8Bit()),
              (dirs.item(j).toElement().attribute("includesubdir") == "1")
              );
    }
	cp->addPlaylist(pr);

	//m3us
    QDomNodeList m3us = node.elementsByTagName ("m3u") ;
    for (uint m=0;m<m3us.count();m++) {
		string path = string(m3us.item(m).toElement().attribute("path").local8Bit());
		M3UPlaylist *p = new M3UPlaylist(path); p->setM3U(path);
		cp->addPlaylist(p);
    }
	//masks
    QDomNodeList masks = node.elementsByTagName ("mask") ;
    QString qmask = masks.item(0).toElement().attribute("extension");
	vector<string> filenamesEnd = vector<string>();
	string mask = ""; 
	if (qmask) { 
		QStringList splitted = QStringList::split ( ";", qmask, FALSE );
		for (uint k=0;k<splitted.count();k++) {
			QRegExp num("[*. ]*");        
			qmask = splitted[k].replace(num,"");//first bring into a QString to avoid a Windows crash
			if (!qmask.isEmpty()) mask = string(qmask.local8Bit()); else mask = "";
			if (mask != "") filenamesEnd.push_back(mask);
		}
		cp->setFilenamesEndWith(filenamesEnd);
	}

    QDomNodeList exclude = node.elementsByTagName ("excludemasks") ;
    QString emask = exclude.item(0).toElement().attribute("value");
	vector<string> filenamesDoNotEnd = vector<string>();    
	if (emask) {
		QStringList splitted = QStringList::split ( ";", emask, FALSE );
		mask = ""; 
		for (uint l=0;l<splitted.count();l++) {
			QRegExp num("[*. ]*");
			qmask = splitted[l].replace(num,"");//first bring into a QString to avoid a Windows crash        
			if (!qmask.isEmpty()) mask = string(qmask.local8Bit()); else mask = "";
	        if (mask != "") filenamesDoNotEnd.push_back(mask);    
		}
    	cp->setFilenamesDoNotEndWith(filenamesDoNotEnd);
	}
    return cp;
}

vector<string> PlaylistReader::getPlaylists() {
    vector<string> vpres;

    QDomNodeList stuff = _playlistdoc.elementsByTagName ("playlist") ;

    QDomElement node; QDomAttr att;
    for (uint i=0;i<stuff.length() ;i++) {
        node = stuff.item(i).toElement();
        att = node.attributes().namedItem(QString::fromLocal8Bit("name")).toAttr();
        vpres.push_back(string(att.value().local8Bit()));
    }
    return vpres;
}
