#include "queuebuilder.h"
#include "absplayer.h"
#include "absutil.h"
#include "lpdatabase.h"
#include "playlist.h"
#include "lpsettings.h"
#include "lpplaylistiterator.h"
#include "lpsongfilter.h"
#include "directoryplaylist.h"
#include "m3uplaylist.h"
#include "lplayer.h"

IntSetting QueueBuilder::ColorConstraintX1("heuristic","colorx1",0);
IntSetting QueueBuilder::ColorConstraintY1("heuristic","colorx2",0);
IntSetting QueueBuilder::ColorConstraintX2("heuristic","colory1",0);
IntSetting QueueBuilder::ColorConstraintY2("heuristic","colory2",0);
IntSetting QueueBuilder::SettingMinimumRating("heuristic","minimumrating",0);
FloatSetting QueueBuilder::SettingAverageInterval("heuristic","avginterval",10000);
FloatSetting QueueBuilder::SettingAverageRating("heuristic","avgrating",0);
FloatSetting QueueBuilder::SettingIntervalWeight("heuristic","intervalweight",0.2f);
FloatSetting QueueBuilder::SettingRatingWeight("heuristic","ratingweight",3);
FloatSetting QueueBuilder::SettingLengthWeight("heuristic","lengthweight",0);
IntSetting QueueBuilder::SettingDJMode("queue","djmode",0);
IntSetting QueueBuilder::SettingDJEvery("queue","djevery",0);
StringSetting QueueBuilder::SettingDJSong("queue","djfile");

QueueBuilder::~QueueBuilder() { }

/* QueueBuilder bases its queueing behavior on the settings */
QueueBuilder::QueueBuilder(LPSettings* settings, LPDatabase* database, absPlayer* player) : _stop(false), _settings(settings),_database(database),_player(player), _djval(0) {
	_filter = 0; _historyPointer = 0;
}
LPDatabase* QueueBuilder::getDatabase() const { return _database; }
LPSettings* QueueBuilder::getSettings() const { return _settings; }
absPlayer* QueueBuilder::getPlayer() const { return _player; }

void QueueBuilder::notifyNewSettings() { buildFilter(); }

LPSongFilter* QueueBuilder::getFilter() {
	if (!_filter) buildFilter();
	return _filter;
}

void QueueBuilder::abort() { _stop = true; }

vector<string> QueueBuilder::getQueue(Playlist* playlist,int amount) {
	_stop = false;
	logDebug("QueueBuilder::finding songs to queue...");
	buildFilter();
	mutexFilter.lock();
	LPSongFilter* filter = getFilter();
	LPPlaylistIterator* it = new LPPlaylistIterator(playlist,filter ,getDatabase(),true, true);	

	bool reject = false;
	bool enough = false;
	Song song;

	vector<string> list;
			
    float avgInterval = getSettings()->getSetting(SettingAverageInterval);
    float avgRating = getSettings()->getSetting(SettingAverageRating);
	float intervalWeight = _settings->getSetting(SettingIntervalWeight);
	
	for (int j=0; (!_stop && !enough);j++) {
		reject = false;
		if (it->hasNext()) song = it->next(); 
		else { reject = true; enough = true; }
		if (song.getFilename().empty()) { reject = true; enough = true;} //couldn't find a song
		//todo move valid check to iterator/filter
		if (!song.valid()) { 
			reject = true; 
			//bad playlist, bad bad playlist!
			logDebug("QueueBuilder::playlist invalid, rebuilding");
			playlist->startBuilding();
		} 
		if (getPlayer()->inPlaylist(song.getFilename())) reject = true;
		if (!reject) {
			if (DirectoryPlaylist::isDirectory(song.getFilename())) {
				DirectoryPlaylist p; p.addDir(song.getFilename(),true); p.startBuilding();
				vector<string>* filenames2 = p.filenames();
				for (uint j=0;filenames2 && (j<filenames2->size());j++) {
					addToList(&list, (*filenames2)[j]);
				}
			}
			else
			if (M3UPlaylist::isM3U(song.getFilename())) {
				//XMMS won't queue m3u's, but we can
				M3UPlaylist p; p.setM3U(song.getFilename()); p.startBuilding();
				vector<string>* filenames2 = p.filenames();
				for (uint j=0;filenames2 && (j<filenames2->size());j++) {
					addToList(&list,(*filenames2)[j]);
				}
			}
			else {
				addToList(&list,song.getFilename());
				filter->confirm(song);
			}
			//we found a/the (best)song! now tweak
			float currInterval = absUtil::getUtil()->date() - song.getDate();
			if ((currInterval > 3600) && (song.getDate() > 0)) {
				getSettings()->setSetting(SettingAverageInterval, (avgInterval * (1 - intervalWeight)) + (intervalWeight * currInterval));
				getSettings()->setSetting(SettingAverageRating, (avgRating * (1 - intervalWeight)) + (intervalWeight * song.getLike()));
			}
		}
		if (!reject) { logInfo(string("Found ") + song.getFilename(), 500); }
		if ((int)(list.size()) >= amount) enough = true;
	}
	delete it;
	mutexFilter.unlock();
	return list;
}

void QueueBuilder::test() {
	LPSongFilter* filter = getFilter();
	Song song;

	logDebug("testing standard filter with avginterval of 10 days, avg rating of 5.5 and rating weight of 5/10");
	logDebug("=======================================================================================");

//TODO not valid anymore
	getSettings()->setISetting("queue","onlyrating", false);
	getSettings()->setISetting("queue","onlycolor", false);
	getSettings()->setISetting("queue","ratingmin",0);
	getSettings()->setISetting("queue","colorh",50);
	getSettings()->setISetting("queue","colors",50);
	getSettings()->setISetting("queue","colorv",50);

	buildFilter();
	filter = getFilter();
	song.setDate(absUtil::getUtil()->date()-(24*3600*12));
	song.setLike(1);
	song.setTimes(1);
	float f = filter->goodahness(song);
	logDebug(song.toString()  + " gives " + absUtil::fts(f));
	song.setLike(-1);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date());
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setLike(1);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date()-(24*3600*1));
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date()-(24*3600*31));
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));

	logDebug("testing filter with minimum of 5/10, avginterval of 10 days, avg rating of 5.5 and rating weight of 5/10");
	logDebug("=======================================================================================");

	getSettings()->setISetting("queue","onlyrating", true);
	getSettings()->setISetting("queue","ratingmin",5);

	buildFilter();
	filter = getFilter();
	song.setDate(absUtil::getUtil()->date()-(24*3600*12));
	song.setLike(1);
	song.setTimes(1);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setLike(-1);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date());
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setLike(1);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date()-(24*3600*1));
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date()-(24*3600*31));
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));

	logDebug("testing filter with color 50,50,50, avginterval of 10 days, avg rating of 5.5 and rating weight of 5/10");
	logDebug("=======================================================================================");

	getSettings()->setISetting("queue","onlyrating", false);
	getSettings()->setISetting("queue","onlycolor", true);
	getSettings()->setISetting("queue","colorh",50);
	getSettings()->setISetting("queue","colors",50);
	getSettings()->setISetting("queue","colorv",50);

	buildFilter();
	filter = getFilter();
	song.setDate(absUtil::getUtil()->date()-(24*3600*12));
	song.setLike(1);
	song.setTimes(1);
	song.setColor(50,50,50);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setLike(-1);
	song.setColor(10,10,10);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date());
	song.setColor(47,47,47);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setLike(1);
	song.setColor(50,50,70);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date()-(24*3600*1));
	song.setColor(70,50,50);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));
	song.setDate(absUtil::getUtil()->date()-(24*3600*31));
	song.setColor(45,45,45);
	logDebug(song.toString()  + " gives " + absUtil::fts(filter->goodahness(song)));

}

/* side-effect: invalidates all references obtained by getFilter */
void QueueBuilder::buildFilter() {
	mutexFilter.lock();
	if (_filter) delete _filter; 
	_filter = 0;
	logDebug("QueueBuilder::building songfilter...");
    float avgInterval = getSettings()->getSetting(SettingAverageInterval);
    float avgRating = getSettings()->getSetting(SettingAverageRating);
    float ratingWeight = getSettings()->getSetting(SettingRatingWeight);// from 0 to 10
    float lengthWeight = getSettings()->getSetting(SettingLengthWeight);
	if ((avgInterval < 0)) avgInterval *= -1;
	if ((avgInterval == 0)) avgInterval = 1;//prevent divide by zero
	LPSongFilter* filter1 = new LPSecondsAgoFilter(LPSongFilter::MORE, (int) (avgInterval /2));
	LPSongFilter* filter2 = new LPRatingFilter(LPSongFilter::MORE, (int) avgRating);
	LPSongFilter* filter = new LPAddFilter(filter1, 5,filter2,ratingWeight);
	filter1 = filter;
	filter2 = new LPLengthFilter(LPSongFilter::LESS, 300);
	filter = new LPAddFilter(filter1, 5,filter2,lengthWeight);
	filter1 = filter;
	filter2 = new LPDistanceFilter(&_history);
	filter = new LPAddFilter(filter1, 5,filter2,2);

	if (getSettings()->getSetting(SettingMinimumRating) > 0) {
		filter1 = filter;
		filter2 = new LPRatingFilter(LPSongFilter::MORE, getSettings()->getSetting(SettingMinimumRating) -1 -5, true);
		filter = new LPAddFilter(filter1, 100,filter2,1);//the weight is not important as we only need the level
	}
	
	filter1 = filter;
	filter2 = new LPColorFilter2(getSettings()->getSetting(ColorConstraintX1), getSettings()->getSetting(ColorConstraintY1),getSettings()->getSetting(ColorConstraintX2),getSettings()->getSetting(ColorConstraintY2));
	filter = new LPAddFilter(filter1, 3,filter2,1);

	_filter = filter;	
	mutexFilter.unlock();
}

//replace this by a good fifo
void QueueBuilder::addToList(vector<string>* list, string filename) {
	if (getSettings()->getSetting(SettingDJMode)) {
		_djval--;
		if (_djval < 0) {
			_djval = getSettings()->getSetting(SettingDJEvery) -1;
			list->push_back(getSettings()->getSetting(SettingDJSong));
		}
	}
	list->push_back(filename);
	uint historyLength = getSettings()->getSetting(LPlayer::SettingEffort)*2;
	if (historyLength <= _history.size()) {//recycle
		_historyPointer = (_historyPointer + 1) % historyLength; 
		_history[_historyPointer] = filename;
	}
	else _history.push_back(filename);
}
