#include "songaccess.h"
#include "lpglobal.h"
#include "absutil.h"

#include <iostream>
#include <string>

#include <qstring.h>
#include <qdir.h>
#include <qdom.h>
#include <qtextstream.h>
#include <qprogressdialog.h>

QFile* SongAccess::getFile() { return _file; }

SongAccess::SongAccess(string filename) {
	//class initialisation
	MAGIC_READ_MARKER[0]= 27;
	MAGIC_READ_MARKER[1]= 52;
	MAGIC_WRITE_MARKER[0]= 27;
	MAGIC_WRITE_MARKER[1]= 54;
	_file = 0;
	_index = new SongAccessIndex();	
	open(filename);
}

void SongAccess::open(string filename) {
	_filename = filename;
	_file = openFile(filename);
	if (!_file) {
		logError("could not open database");
		return;
	}
	string indexFilename = filename + ".idx";
	if (testLock(filename) || !getIndex()->load(indexFilename))
		buildIndex(_file);
	if (absUtil::getUtil()->existsFile(indexFilename)) absUtil::getUtil()->removeFile(indexFilename);//index only saved at the end
	setLock(filename, true);
}

QFile* SongAccess::openFile(string filename) {
   	QFile* file = new QFile(QString::fromLocal8Bit(filename.c_str()));

	if (!absUtil::getUtil()->existsFile(filename)) {
    	if (!file->open(IO_WriteOnly | IO_Truncate)) {
            logDebug("SongAccess::cannot save.");
            logError(string("Could not save database! Please check file permissions or free space."));
			delete file; 
            return 0;
   		 }
		else {
			QTextStream t(file); t.setEncoding(QTextStream::UnicodeUTF8);
			t << "{{color}}\n";
			file->close();
		}
	}

   	if (!file->open(IO_ReadWrite)) {
		logError("could not open");
		delete file; return 0;
	}
   	if (!consumePrelude(file)) {
		logError("could not read prelude: not a valid file");
		file->close(); delete file; return 0;
	}
	return file;	
}

bool SongAccess::consumePrelude(QFile* f) {
 	QTextStream t(f); t.setEncoding(QTextStream::UnicodeUTF8); f->at(0);
	if (t.atEnd()) return false;
	QString str = t.readLine();
	if (str != "{{color}}") {	
		logWarning(string("SongAccess::{{color}} != ") + string(str.local8Bit()));
		return false;
	}
	return true;
}


SongAccessIndex* SongAccess::getIndex() const { return _index; }

void SongAccess::close() {
	setLock(_filename, false);
	if (getFile()) delete getFile(); _file = 0;
	if (getIndex()) delete getIndex(); _index = 0;
}

int SongAccess::consumeMarker(QFile* f) {
    QTextStream t(f);
	char test[3]; test[2]='\0';
	int i = f->at();
	t.readRawBytes(test,2);
	int skipped = 0;

	while (!f->atEnd() && strncmp(test,MAGIC_READ_MARKER,2)) {
		f->at(i); //go back in case we read over a line
		//advance and try again
		t.readLine(); i = f->at(); skipped++;
		if (!f->atEnd()) t.readRawBytes(test,2); 
	}
	//logDebug(absUtil::its(skipped) + " lines skipped");
	if (f->atEnd()) return 0; //if we're at the end, there's no marker..
	return i;
}

Song SongAccess::consumeSong(QFile* f) {
	if (!consumeMarker(f)) return Song(); //at the end
	
    QTextStream t(f); t.setEncoding(QTextStream::UnicodeUTF8);
	Song s; QString gr;
	
	if (t.atEnd()) return s; gr = t.readLine(); if (gr.length()) s.setKey(string(gr.local8Bit()));
	//logDebug(string("SongAccess::reading record ") + s.getKey());

	if (t.atEnd()) return s; gr = t.readLine(); if (gr.length()) s.setName(string(gr.local8Bit()));
	if (t.atEnd()) return s; s.setFilename(string(t.readLine().local8Bit()));
	if (t.atEnd()) return s; s.setDate(t.readLine().toInt());
	if (t.atEnd()) return s; s.setOldDate(t.readLine().toInt());
	if (t.atEnd()) return s; s.setTimes(t.readLine().toInt());
	if (t.atEnd()) return s; s.setLike(t.readLine().toInt());
	if (t.atEnd()) return s; s.setLength(t.readLine().toInt());
	if (t.atEnd()) return s; int i = t.readLine().toInt(); s.setColor(i / (256*256), (i%(256*256))/ 256, i%256 );
	return s;
}

//writes at current position
void SongAccess::writeSong(QFile* f, Song s) {
    QTextStream t(f); t.setEncoding(QTextStream::UnicodeUTF8);
	
	writeMarker(f);
    t << s.getKey().c_str() << "\n";
    t << QString::fromLocal8Bit(s.getName().c_str()) << "\n";
    t << QString::fromLocal8Bit(s.getFilename().c_str()) << "\n";
    t << s.getDate()  << "\n";
    t << s.getOldDate()  << "\n";
    t << s.getTimes() << "\n";
    t << s.getLike() << "\n";
    t << s.length() << "\n";
	int hue,sat,val; s.getColor(&hue,&sat,&val);	
	t << hue*256*256 + sat * 256 + val << "\n";
}

SongAccess::~SongAccess() { close(); }

void SongAccess::setLock(string filename, bool b) {
	string lockFile = filename + ".lock";
	if (b) {	
		QFile file(QString::fromLocal8Bit(lockFile.c_str()));
    	if (!file.open(IO_WriteOnly | IO_Truncate)) {
            logDebug("SongAccess::cannot save lock file.");
            logError(string("Could not write lock! Please check for other lplayer instances or free space to create ") + lockFile);
            return;
		}
		QTextStream tto(&file); tto.setEncoding(QTextStream::UnicodeUTF8);
    	tto << QString("{{lock}}\n");
		file.close();
	}
	else {
		if (absUtil::getUtil()->existsFile(lockFile)) absUtil::getUtil()->removeFile(lockFile);
	}
}

bool SongAccess::testLock(string filename) {
	string lockFile = filename + ".lock";
	return (absUtil::getUtil()->existsFile(lockFile));
}

void SongAccess::buildIndex(QFile* file) {
	logBusy("Building index...",false);
	
	if (!getIndex()) delete getIndex();
	_index = new SongAccessIndex();
	
	QTextStream t( file ); t.setEncoding(QTextStream::UnicodeUTF8);
	QString key; int pos;
	if (!consumePrelude(file)) return;
	while (!t.atEnd()) {
		pos = consumeMarker(file);
		if (!pos) continue; //nothing to consume
		//read the key
    	key = t.readLine();
		if (!key.length()) continue;
		getIndex()->store(string(key.local8Bit()), pos);
	}
	logDebug(string("SongAccess::entries found: ") + absUtil::its(getIndex()->getSize()));
	logDebug("SongAccess::reading database. done.");
	logReady("Building index...");
}

void SongAccess::flush() { getFile()->flush(); }

//rebuilds indices and database
void SongAccess::compact() {
	logDebug("SongAccess::compacting database...");
	string newFilename = _filename + ".tmp";
	if (absUtil::getUtil()->existsFile(newFilename)) absUtil::getUtil()->removeFile(newFilename);

	SongAccessIndex* newIndex = new SongAccessIndex();
	SongAccessIndex* oldIndex = getIndex();
	QFile* oldFile = getFile();
	QFile* newFile = openFile(newFilename);//automatically creates and consumes prelude
	newFile->at(newFile->size());//necessary in windows

	LPDatabaseIterator* oldIterator = iterator(); oldIterator->reset();

	Song song;
	while (!oldIterator->atEnd()) {
		song = oldIterator->next();
		if (!song.empty()) {
			newIndex->store(song.getKey(),newFile->at());
			writeSong(newFile, song);
		}
	}
	oldFile->close();
	newFile->close();

	absUtil::getUtil()->copyFile(newFilename,_filename);

	delete (oldFile); _file = 0;
	delete (newFile); 
	delete (oldIndex);
	delete (oldIterator);
	_index = newIndex;//use the new index
	_file = openFile(_filename);//use the new file
	getIndex()->save(_filename + ".idx");//save the new index
	absUtil::getUtil()->removeFile(newFilename);//remove the tmp file - last action
	logDebug("SongAccess::compacting database. done.");
}

void SongAccess::store(Song song ) {
	if (song.empty()) return;

	Song old = retrieve(song.getKey());
	if (!old.empty()) {
		if (!old.equivalent(song)) {
			logWarning(string("SongAccess::key clash! fixing " ) + song.getKey() + "/" + song.getFilename() + ". " + song.getName() + " vs" + old.getName());
			song.setKey(song.getKey() + "A");
			store(song);
			return;
		}
	}
	//song may be equivalent, but not the same, so store it
	QFile* f = getFile();
	f->flush();
	f->at(f->size()); 
	getIndex()->store(song.getKey(),f->at());
	writeSong(f, song);
}

void SongAccess::remove(Song song) { getIndex()->remove(song.getKey()); }
int SongAccess::getSize() { return (getIndex()->getSize()); }
bool SongAccess::inDatabase(string key) { return (getIndex()->retrieve(key) != -1); }

//returns Song() if not in database
Song SongAccess::retrieve(string skey) {
	Song song;
	int pos = getIndex()->retrieve(skey);
	if (pos < 0) return song;
	getFile()->at(pos);
	song = consumeSong(getFile());
	if (song.getKey() != skey) {
		logWarning(string("SongAccess::asked to retrieve ") + skey + " and got " + song.getKey() + ".");
		//invalid index
		buildIndex(getFile());
		return retrieve(skey); //try again
	}
	return song;
}

void SongAccess::writeMarker(QFile* f) {
    QTextStream t(f);
	t.writeRawBytes(MAGIC_READ_MARKER,2);
}

LPDatabaseIterator* SongAccess::iterator() {
    return new SongAccess::SongAccessIterator(this);
}

void SongAccess::test() {
}

SongAccess::SongAccessIterator:: SongAccessIterator(SongAccess* reader) { _reader = reader; }
SongAccess* SongAccess::SongAccessIterator::getReader() { return _reader; }

void SongAccess::SongAccessIterator::reset() {
	_pos = 0;
	getReader()->consumePrelude(getReader()->getFile());
	_pos = getReader()->getFile()->at();
}
	
Song SongAccess::SongAccessIterator::next() {
    QFile* f = getReader()->getFile();
	f->at(_pos);

	int lastpos = _pos;
	Song s = getReader()->consumeSong(f);
	_pos = f->at();
	if (s.empty()) {
		if (atEnd()) return s;
		logDebug(string("SongAccess::found empty song ") + s.getKey() + ", skipping..");
		return next();
	}

	//verify that this is indeed the last version
	int properpos = getReader()->getIndex()->retrieve(s.getKey());

	if (properpos == -1) {
		logDebug(string("SongAccess::found deleted song ") + s.getKey() + ", skipping..");
		return next();//deleted
	}
	else if (properpos != lastpos)  {
		logDebug(string("SongAccess::found old version of song ") + s.getKey() + ", skipping..");
		return next();//we'll encounter it later on
	}
	return s;
}

SongAccessIndex::SongAccessIndex() { _index = new QMap<string,long>; }
SongAccessIndex::~SongAccessIndex() { delete _index; }

bool SongAccess::SongAccessIterator::atEnd() { return (getReader()->getFile()->size() <= (_pos)); }
	
/** returns true if success */
bool SongAccessIndex::load(string filename) {
    logInfo("Loading index...",0);

	_index->clear();
    logDebug("SongAccess::index database " + filename);

    QFile f(QString::fromLocal8Bit(filename.c_str()));
    if (!f.open(IO_ReadOnly)) {
        logWarning(string("SongAccess::reading indexes ") + filename + " failed.");
        return false;
    }

    QTextStream t( &f );
		t.setEncoding(QTextStream::UnicodeUTF8);
	string key; int pos; QString tmpStr;
    if (t.atEnd() || t.readLine() != "{{index}}") {
			f.close(); return false;
	}

    while (!t.atEnd()) {
        if (!t.atEnd()) tmpStr = t.readLine();
		if (tmpStr.length()) key = tmpStr.local8Bit(); else key = "";
        if (!t.atEnd()) pos = t.readLine().toLong();
		if (!key.empty()) {
			if ((*_index).contains(key)) (*_index).replace(key, pos);
            else (*_index)[key]=pos;
			
		}
		else logWarning("SongAccess::encountered empty record while reading index");
	}
	f.close();
    logInfo("Loading index... Done.");
    logDebug("SongAccess::index database loaded." );
	return true;
}

void SongAccessIndex::store(string key, int pos) {
	if (key.empty()) return;
	if ((*_index).contains(key)) (*_index).replace(key, pos);
	else (*_index)[key]=pos;
}

int SongAccessIndex::getSize() { return _index->count(); }

void SongAccessIndex::save(string filename) {
    QMapIterator<string,long> it;
    Song s;
   	QFile file(QString::fromLocal8Bit(filename.c_str()));
    if (!file.open(IO_WriteOnly | IO_Truncate)) {
            logDebug("SongAccess::cannot save index file.");
            logError(string("Could not write index! Please check file permissions or free space to create ") + filename);
            return;
	}
	QTextStream tto(&file); tto.setEncoding(QTextStream::UnicodeUTF8);

    tto << QString("{{index}}\n");
    for (it = (*_index).begin(); it != (*_index).end(); ++it ) {
      tto << it.key().c_str() << "\n";
      tto << it.data() << "\n";
    }
    file.close();
}

int SongAccessIndex::retrieve(string key) {
	QMapIterator<string,long> i = (*_index).find(key);
	if (i == (*_index).end()) { return -1; }
	return i.data();
}

void SongAccessIndex::remove(string key) { (*_index).remove(key); }
