/** LongPlayer - cross-platform multimedia queuer
 *  Copyright (C) 2001-2004 Andrew Wils
 *
 *  This program is free software; See the
 *  GNU General Public License for more details.
 */
#ifndef H_SONGACCESS
#define H_SONGACCESS

#include <fstream>
#include <iostream>

#include "absutil.h"
#include "lpsong.h"
#include "lpdatabasereader.h"
#include "lpdatabaseiterator.h"

#include <vector>
#include <qstring.h>
#include <qdatetime.h>
#include <qdom.h>
#include <qmap.h>
#include <qfile.h>

using namespace std;

/** SongAccessIndex:  an index for song positions in SongAccess */
class SongAccessIndex {
  public:
	SongAccessIndex();
	~SongAccessIndex();

	void store(string key, int pos);
	//returns -1 if not present
	int retrieve(string key);
	void remove(string key);

	bool load(string filename);
	void save(string filename);
	int getSize();
  private:
	//qt hashmap
	QMap<string,long> *_index;
};  

/** SongAccess: a fairly robust song record reader/writer
 */
class SongAccess : public LPDatabaseReader {
	friend class LPDatabaseIterator;
public:
	SongAccess(string filename);
	~SongAccess();

	void flush(); //quicksave?? ensures persistency
	void compact(); //saves, compacts.. whatever

	/** stores the given song description based on the key of the song */
	void store(Song song);
	void remove(Song song);
	/** looks up the song in the database */
	Song retrieve(string key);

	/** returns true when there is a song description in the database */
	bool inDatabase(string key) ;
	/** gives an estimate of the number of song descriptions in the database */
	int getSize();

    LPDatabaseIterator* iterator();
private:
	/* we keep a .lock file while using the db and delete it when
	 * the index is properly saved.
	 * If the lock file exists on startup, we should rebuild the index.
	 */
	void setLock(string filename, bool);
	bool testLock(string filename);
	void test();

	/* Opens a lp2 file to be accessed with getFile().
	 * if this fails, 0 is returned.
	 * The file has to be a valid or non-existing "color" lp2 file.
	 * if it does not exist, it is automatically created.
	 */
	QFile* openFile(string filename);

	//rebuilds a new index for the file (resets and consumes file)
	void buildIndex(QFile*);
	void open(string filename);
	void close();
	SongAccessIndex* getIndex() const;
	QFile* getFile();

	//writes a marker
	void writeMarker(QFile*);
	//writes a marker and the song at the current position
	void writeSong(QFile*, Song);
	
	//resets position and consumes prelude, returns false if it was not found
	bool consumePrelude(QFile*);
	//reads until and consumes next marker; returns 0 if not found, else position of marker
	//reads until a marker, consumes it and returns the position of the marker (0 if it was not encountered)
	int consumeMarker(QFile*);
	//reads until a marker + song is encountered and consumes it
	Song consumeSong(QFile*);

	
	//database filename
	string _filename;
	SongAccessIndex* _index;
	QFile* _file;
	char MAGIC_READ_MARKER[2];
	char MAGIC_WRITE_MARKER[2];
    
    class SongAccessIterator : public LPDatabaseIterator
    {
      public:
        SongAccessIterator(SongAccess* reader);
        void reset();
        Song next();
        bool atEnd();
                                       
      private:
		SongAccess* getReader();
		int _pos;
        SongAccess* _reader;
    };  
    
    friend class SongAccessIterator;
};

#endif
