/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: fitsio.h,v 1.2.2.7 2006/03/10 19:27:28 cag Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Claus A. Goessl <cag@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

/* \file fitsio.h
  I/O adapter class fitsfile \>\> ltl::MArray and fitsfile \<\< ltl::MArray.
*/

#ifndef __FITSIO_H__
#define __FITSIO_H__

#include <ltl/config.h>
#include <ltl/fits.h>
#include <ltl/marray.h>
#include <ltl/misc/exceptions.h>

// templates to read from fitsfile class

LTL_BEGIN_NAMESPACE

// read data
// check dimensions
template<class T, int N>
MArray<T, N> getFitsMArray(FitsIn& fitsfile)
{
   int dimension = fitsfile.getNaxis();
   size_t length = fitsfile.getRegionLength();
   int dim_ [N];

   if(length)
      for(int i = 0; i < dimension; ++i)
      {
         dim_[i] = fitsfile.getRegionLength(i+1);
         if(dim_[i] == 1)
         {
            --dimension;
            --i;
         }
      }
   else
      for(int i = 0; i < dimension; ++i)
         dim_[i] = fitsfile.getNaxis(i+1);

   if( N != dimension )
      throw FitsException("NAXIS or slice does not match MArray dimensions");

   T* dataptr = fitsfile.getDataArray(T(0));

   return MArray<T, N>(dataptr, dim_);
}

template<class T, int N>
void insertFitsMArray( MArray<T, N>& a, FitsIn& fitsfile)
{
   if(N > fitsfile.getNaxis())
      throw FitsException("MArray cannot be filled with FITS file");

   Region fregion(fitsfile.getNaxis());
   // setup a initial region
   if(fitsfile.getRegionLength() == 0)
   {
      for(int i = 1; i <= N; ++i)
         fregion.setRange( i, 1,
                           (a.length(i) < fitsfile.getNaxis(i)) ?
                           a.length(i) : fitsfile.getNaxis(i) );
      for(int i = N+1; i <= fitsfile.getNaxis(); ++i)
         fregion.setRange( i, 1, 1);

      // test if a is exactly conformable with file
      bool exactmatch = (fitsfile.getNaxis() == N);
      for(int i = 1; exactmatch && (i <=N); ++i)
         exactmatch = exactmatch && (fregion.getLength(i) == (unsigned int)fitsfile.getNaxis(i));
      if(!exactmatch)
         fitsfile.setRegion(fregion);
      else
         fitsfile.resetRegion();
   }
	 else
		 fregion.setRegion(fitsfile.getRegion());

   // actually read data
   if(fitsfile.getRegionLength() == 0)
   {
      typename MArray<T, N>::iterator i = a.begin();
      fitsfile.readDataArray(i);
   }
   else{
      if( a.nelements() == fregion.getLength() )
      {
         typename MArray<T, N>::iterator i = a.begin();
         fitsfile.readRegionArray(i);
      }
      else
      {
         a = T(0);
         MArray<T, N> b(getMArrayRegion(a, fregion));
         typename MArray<T, N>::iterator i = b.begin();
         fitsfile.readRegionArray(i);
      }

			// increment region
      bool carriage_return = true;      
      int i = 1;
      while(carriage_return && (i <= N))
      {
         if(carriage_return)
         {
            const int actend = fitsfile.getRegionEnd(i);
            const int actnaxis = fitsfile.getNaxis(i);
            const int actalength = a.length(i);
            // end of line?
            if(actend == actnaxis) // yes => new line and carriage return
               fregion.setRange( i, 1,
                                 (actalength < actnaxis) ? 
                                 actalength : actnaxis );
            else // no => step ahead
            {
               const int newend = actend + actalength;
               if(newend < actnaxis)
                  fregion.setRange(i, actend + 1, newend);
               else
                  fregion.setRange(i, actend + 1, actnaxis);
               carriage_return = false;
            }
         }
         ++i;
      }
      while(carriage_return && (i <= fitsfile.getNaxis()))
      {
         if(carriage_return)
         {
            const int actend = fitsfile.getRegionEnd(i);
            const int actnaxis = fitsfile.getNaxis(i);
            // end of line?
            if(actend == actnaxis) // yes => new line and carriage return
               fregion.setRange( i, 1, 1);
            else // no => step ahead
            {
               const int newend = actend + 1;
               fregion.setRange(i, newend, newend);
               carriage_return = false;
            }
         }
         ++i;
      }

      if(carriage_return)
      {
         fitsfile.resetRegion();
         //throw FitsException("FITS file already completely read, next try starts over.");
      }
      fitsfile.setRegion(fregion);
   }
}

template<class T>
MArray<T, 1> getMArrayRegion(MArray<T, 1>& a, const Region& fregion)
{
   return a( Range(fregion.getStart(1), fregion.getLength(1)) );
}

template<class T>
MArray<T, 2> getMArrayRegion(MArray<T, 2>& a, const Region& fregion)
{
   return a( Range(fregion.getStart(1), fregion.getEnd(1)),
             Range(fregion.getStart(2), fregion.getEnd(2)));
}

template<class T>
MArray<T, 3> getMArrayRegion(MArray<T, 3>& a, const Region& fregion)
{
   return a( Range(fregion.getStart(1), fregion.getEnd(1)),
             Range(fregion.getStart(2), fregion.getEnd(2)),
             Range(fregion.getStart(3), fregion.getEnd(3)));
}

template<class T>
MArray<T, 4> getMArrayRegion(MArray<T, 4>& a, const Region& fregion)
{
   return a( Range(fregion.getStart(1), fregion.getEnd(1)),
             Range(fregion.getStart(2), fregion.getEnd(2)),
             Range(fregion.getStart(3), fregion.getEnd(3)),
             Range(fregion.getStart(4), fregion.getEnd(4)));
}

template<class T>
MArray<T, 5> getMArrayRegion(MArray<T, 5>& a, const Region& fregion)
{
   return a( Range(fregion.getStart(1), fregion.getEnd(1)),
             Range(fregion.getStart(2), fregion.getEnd(2)),
             Range(fregion.getStart(3), fregion.getEnd(3)),
             Range(fregion.getStart(4), fregion.getEnd(4)),
             Range(fregion.getStart(5), fregion.getEnd(5)));
}

/*! \addtogroup ma_fits_io
*/
//@{

/*! \relates ltl::FitsIn
  Read FITS data into ltl::MArray.
*/
template<class T, int N>
FitsIn& operator>>(FitsIn& fitsfile, MArray<T, N> & a)
{
   if(a.isAllocated())
      insertFitsMArray<T, N>(a, fitsfile);
   else
      a.makeReference(getFitsMArray<T, N>(fitsfile));
   return fitsfile;
}

//@}

template<class T>
struct Bitpix
{
   enum { bitpix = 0 };
};

template<>
struct Bitpix<unsigned char>
{
   enum { bitpix = 8 };
};

template<>
struct Bitpix<char>
{
   enum { bitpix = 8 };
};

template<>
struct Bitpix<short int>
{
   enum { bitpix = 16 };
};

template<>
struct Bitpix<int>
{
   enum { bitpix = 32 };
};

template<>
struct Bitpix<float>
{
   enum { bitpix = -32 };
};

template<>
struct Bitpix<double>
{
   enum { bitpix = -64 };
};

/*! \addtogroup ma_fits_io
*/
//@{

/*! \relates ltl::FitsOut
  Write ltl::MArray to FITS file.
*/
template<class T, int N>
FitsOut& operator<<(FitsOut& fitsfile, const MArray<T, N>& a)
{

   // get new NAXIS
   int naxis_i [N];
   for(int i = 0; i < N; ++i)
      naxis_i[i] = a.length(i+1);

   // if neccessary set default BITPIX
   if( fitsfile.getBitpixOut() == 0 )
      fitsfile.setBitpixOut(Bitpix<T>::bitpix);

   // if neccessary reset ORIGIN
   if( (fitsfile.getOrigin()).length() < 1 )
      fitsfile.setOrigin("LTL FITS IO class, July 2002");

   if(fitsfile.isRegion())
   {
      // write data
      typename MArray<T, N>::const_iterator i = a.begin();
      fitsfile.writeRegionArray(i);
   }
   else
   {
      // map file
      fitsfile.openData(fitsfile.getBitpixOut(), N, naxis_i);
      // write data
      typename MArray<T, N>::const_iterator i = a.begin();
      fitsfile.writeDataArray(i);
   }
   // close file
   fitsfile.closeData();
   return fitsfile;
}


//@}

LTL_END_NAMESPACE

#endif
