/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: fmatrix.h,v 1.3.4.3 2004/07/01 02:28:40 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *                         Claus A. Goessl <cag@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#ifndef __LTL_FMATRIX__
#define __LTL_FMATRIX__


// ====================================================================

// little trick to avoid files being included separetely
#define __LTL_IN_FILE_FMATRIX__
// this is #undef'd at the end of this file ...

// ====================================================================

#include <ltl/config.h>
#include <ltl/misc/mdebug.h>

#include <iostream>
#include <iomanip>
#include <cmath>
#include <cstddef>

#ifdef LTL_USING_NAMESPACE
using std::ostream;
using std::endl;
using std::setw;
#endif

#include <ltl/fvector.h> 

// ====================================================================

#include <ltl/misc/staticinit.h>
#include <ltl/fmatrix/fmiter.h>
#include <ltl/fmatrix/fmexpr.h>
#include <ltl/fmatrix/fmtloops.h>
#include <ltl/fmatrix/fmexpr_ops.h>
#include <ltl/fmatrix/fmbool.h>
#include <ltl/fmatrix/fmtranspose.h>
#include <ltl/fmatrix/fmatvec.h>
#include <ltl/fmatrix/fmatmat.h>

// ====================================================================

LTL_BEGIN_NAMESPACE

//! Matrix with compile-time dimensions.
/*! \ingroup fvector_fmatrix 

  Indices are 1-based.
  STL-compatible iterators and types.
*/
template<class T, int M, int N>
class FMatrix
{
      friend class FMIter<T,M,N>;
      friend class FMIterConst<T,M,N>;
      friend class ListInitializationSwitch< FMatrix<T,M,N> >;

   public:
      // STL-compatible type definitions
      typedef T              value_type;
      typedef FMIter<T,M,N>  iterator;
      typedef FMIterConst<T,M,N>       const_iterator;
      typedef T&             reference;
      typedef const T&       const_reference;
      typedef std::size_t    size_type;
      typedef std::ptrdiff_t difference_type;

      typedef FVector<T,M,N>   ColumnVector;
      typedef FVector<T,N,1>   RowVector;
      typedef FVector<T,M,N+1> TraceVector;
      
      enum { static_size = 1 };

      static size_type size()
      {
         return M*N;
      }
      static bool empty()
      {
         return false;
      }
      static size_type max_size()
      {
         return M*N;
      }
      // here we go

      FMatrix()
         : data_(__data_-(N+1))
      { }

      ~FMatrix()
      { }

      FMatrix( const FMatrix<T,M,N>& other );

      FMatrix( const T* t );

      FMatrix( const T t );
      
      template<class Expr>
      FMatrix( const TFMExpr<Expr,M,N>& e );

      //! Assigns t to all elements.
      /*! 
        A bit more comlicated since we have to discriminate between
        A = 3; and A = 1, 2, 3, 4;
        which is done using ListInitializationSwitch which either calls
        ListInitializer or FMatrix::fill().
      */
      ListInitializationSwitch< FMatrix<T,M,N> > operator=( T x )
      {
         return ListInitializationSwitch< FMatrix<T,M,N> >( *this, x );
      }

      int length() const
      {
         return M*N;
      }

      int minIndex( const int dim ) const
      {
         return 1;
      }

      int maxIndex( const int dim ) const
      {
         if( dim == 1 ) return M;
         return N;
      }

      //! 1-based access, also used by expression templates.
      T  operator()( const int i, const int j ) const
      {
         LTL_ASSERT( i>0 && i<=M, "Index 1 out of bounds in FMatrix. Index : "
                     << i << ", size "<<M );
         LTL_ASSERT( j>0 && i<=N, "Index 2 out of bounds in FMatrix. Index : "
                     << j << ", size "<<N );
         return data_[i*N+j];
      }

      T& operator()( const int i, const int j )
      {
         LTL_ASSERT( i>0 && i<=M, 
                     "Index 1 out of bounds in FMatrix(). Index : "
                     << i << ", size "<<M );
         LTL_ASSERT( j>0 && i<=N, 
                     "Index 2 out of bounds in FMatrix(). Index : "
                     << j << ", size "<<N );
         return data_[i*N+j];
      }

      //! Direct access.
      T  operator[]( const int i ) const
      {
         LTL_ASSERT( i>=0 && i<M*N, 
                     "Direct index out of bounds in FMatrix[]. Index : "
                     <<i<<" Range 0-"<<length()-1 );
         return __data_[i];
      }

      T& operator[]( const int i )
      {
         LTL_ASSERT( i>=0 && i<M*N, 
                     "Direct index out of bounds in FMatrix[]. Index : "
                     <<i<<" Range 0-"<<length()-1 );
         return __data_[i];
      }

      //! Return a REFERENCE to column vector.
      ColumnVector col( const int col )
      {
         LTL_ASSERT( col>0 && col<=N, 
                     "Column index out of bounds in FMatrix.col(). Index : "
                     <<col<<" Range 1-"<<N );         
         return ColumnVector( data_+N+col );
      }
      
      //! Return a REFERENCE to row vector.
      RowVector row( const int row )
      {
         LTL_ASSERT( row>0 && row<=M, 
                     "Row index out of bounds in FMatrix.row(). Index : "
                     <<row<<" Range 1-"<<M );         
         return RowVector( data_+row*N+1 );
      }
      
      //! Return a REFERENCE to trace vector.
      TraceVector traceVector()
      {
         LTL_ASSERT( M==N, 
                     "Trace only defined on square Matrix");         
         return TraceVector( data_+N+1 );
      }
      
      //! Direct access to data.
      T* data()
      {
         return __data_;
      }

      const T* data() const
      {
         return __data_;
      }

      //! Iterators.
      iterator begin()
      {
         return FMIter<T,M,N>( *this );
      }
      const_iterator begin() const
      {
         return FMIterConst<T,M,N>( *this );
      }

      void fill( const T x );
      
      // operator x for expressions
      template<class Expr>
      FMatrix<T,M,N>& operator=( const TFMExpr<Expr,M,N>& e );

      template<class Expr>
      FMatrix<T,M,N>& operator+=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator-=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator*=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator/=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator%=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator^=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator&=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator|=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator<<=( const TFMExpr<Expr,M,N>& e );
      template<class Expr>
      FMatrix<T,M,N>& operator>>=( const TFMExpr<Expr,M,N>& e );
      
      // operator x for vectors
      template<class T2>
      FMatrix<T,M,N>& operator=( const FMatrix<T2,M,N>& v );

      FMatrix<T,M,N>& operator=( const FMatrix<T,M,N>& v );

      template<class T2>
      FMatrix<T,M,N>& operator+=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator-=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator*=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator/=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator%=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator^=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator&=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator|=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator<<=( const FMatrix<T2,M,N>& v );
      template<class T2>
      FMatrix<T,M,N>& operator>>=( const FMatrix<T2,M,N>& v );

      FMatrix<T,M,N>& operator+=( const T t );
      FMatrix<T,M,N>& operator-=( const T t );
      FMatrix<T,M,N>& operator*=( const T t );
      FMatrix<T,M,N>& operator/=( const T t );
      FMatrix<T,M,N>& operator%=( const T t );
      FMatrix<T,M,N>& operator^=( const T t );
      FMatrix<T,M,N>& operator&=( const T t );
      FMatrix<T,M,N>& operator|=( const T t );
      FMatrix<T,M,N>& operator<<=( const T t);
      FMatrix<T,M,N>& operator>>=( const T t);

      // special methods
      void swapRows(const int row1, const int row2);      
      void swapCols(const int col1, const int col2);

   protected:      
      T*  data_;
      T __data_[M*N];
};


/*! \relates ltl::FMatrix
 */
template<class T, int M, int N>
ostream& operator<<( ostream& os, const FMatrix<T,M,N>& A )
{
   os << "FMatrix< "<<M<<", "<<N<<" >" << endl;
   os << "[";
   for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
   {
      os << "[ ";
      for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
      {
         os << A(i,j) << " ";
      }
      os << "]";
      if( i<A.maxIndex(1) )
         os << endl << " ";
   }

   os << "]";
   return os;
}

LTL_END_NAMESPACE

#include <ltl/fmatrix/fmatrix_ops.h>
#include <ltl/fmatrix/fmatrix_methods.h>

#undef __LTL_IN_FILE_FMATRIX__

#endif //__LTL_FMATRIX__
