/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: fmexpr.h,v 1.2.2.1 2003/11/02 23:23:43 cag Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */


#ifndef __LTL_IN_FILE_FMATRIX__
#error "<ltl/fmatrix/fmexpr.h> must be included via <ltl/fmatrix.h>, never alone!"
#endif


#ifndef __LTL_FMATRIX_EXPR__
#define __LTL_FMATRIX_EXPR__

#include <ltl/config.h>

LTL_BEGIN_NAMESPACE

template<class T, int M, int N> class FMatrix;


//! Just to keep everything together in class browsers...
class _et_fmatrix_parse_base
{ };

//
//! Binary operation node.
//
template<class A, class B, class Op, int M, int N>
class TBinFMExprOp : public _et_fmatrix_parse_base
{
   private:
      A iter1_;
      B iter2_;

   public:
      typedef typename Op::value_type value_type;
      enum { static_size = A::static_size + B::static_size };

      inline TBinFMExprOp( const A& a, const B& b )
            : iter1_(a), iter2_(b)
      { }

      inline value_type operator()( const int i, const int j ) const
      {
         return Op::eval( iter1_(i,j), iter2_(i,j) );
      }

      inline value_type operator[]( const int i ) const
      {
         return Op::eval( iter1_[i], iter2_[i] );
      }
};

//
//! Unary operation node.
//
template<class A, class Op, int M, int N>
class TUnFMExprOp : public _et_fmatrix_parse_base
{
   private:
      A iter1_;

   public:
      typedef typename Op::value_type value_type;
      enum { static_size = A::static_size };

      inline TUnFMExprOp( const A& a )
            : iter1_(a)
      { }

      inline value_type operator()( const int i, const int j ) const
      {
         return Op::eval( iter1_(i,j) );
      }

      inline value_type operator[]( const int i ) const
      {
         return Op::eval( iter1_[i] );
      }
};


//
//! Literal number.
//
template<class T>
class TFMExprLiteral : public _et_fmatrix_parse_base
{
   private:
      const T f_;

   public:
      typedef T value_type;
      enum { static_size = 1 };

      inline TFMExprLiteral( const T f ) : f_(f)
      { }

      inline const value_type operator()( const int i, const int j ) const
      {
         return f_;
      }

      inline const value_type operator[]( const int i ) const
      {
         return f_;
      }
};


//
//! Now the expression class itself.
//
template<class A, int M, int N>
class TFMExpr : public _et_fmatrix_parse_base
{
   private:
      A iter_;

   public:
      typedef typename A::value_type value_type;
      enum { static_size = A::static_size };

      inline TFMExpr( const A& a )
            : iter_(a)
      { }

      inline value_type operator()( const int i, const int j ) const
      {
         return iter_(i,j);
      }

      inline value_type operator[]( const int i ) const
      {
         return iter_[i];
      }
};


//
//! We need a trait class to deal with literals.
/*!
  Basically convert everything to an expression TExpr.
*/
template<class T>
struct asFMExpr
{
   typedef TFMExprLiteral<T> value_type;
};

//! Already an expression template term.
template<class T, int M, int N>
struct asFMExpr< TFMExpr<T,M,N> >
{
   typedef TFMExpr<T,M,N> value_type;
};

//! An array operand.
template<class T, int M, int N>
struct asFMExpr< FMatrix<T,M,N> >
{
   typedef typename FMatrix<T,M,N>::const_iterator value_type;
};


//
// that's all ...
// the rest are the operator templates
//


//
// BINARY OPERATOR PARSE TREE TEMPLATES
//
// there are basically 8 cases for binary ops, namely:
//
//   matrix  op  matrix,
//   expr    op  matrix,
//   matrix  op  expr,
//   expr    op  expr,
//
//   matrix  op  literal,
//   expr    op  literal,
//   literal op  matrix,  and
//   literal op  expr
//
// a template version of the overloaded operator is provided
// for each case and for each operator

// matrix , matrix
#define FMBINOP_AA(operator, op)                                        \
template<class T1, class T2, int M, int N>                              \
inline TFMExpr<TBinFMExprOp<typename FMatrix<T1,M,N>::const_iterator,   \
                            typename FMatrix<T2,M,N>::const_iterator,   \
                            op<T1,T2>, M, N>, M, N >                    \
operator( const FMatrix<T1,M,N>& a, const FMatrix<T2,M,N>& b )          \
{                                                                       \
   typedef TBinFMExprOp<typename FMatrix<T1,M,N>::const_iterator,       \
                        typename FMatrix<T2,M,N>::const_iterator,       \
                        op<T1,T2>, M, N >                               \
      ExprT;                                                            \
   return TFMExpr<ExprT,M,N>( ExprT(a.begin(), b.begin()) );            \
}

 
// expr , matrix
#define FMBINOP_EA(operator,op)                                         \
template<class A, class T, int M, int N>                                \
inline TFMExpr<TBinFMExprOp<TFMExpr<A,M,N>,                             \
                            typename FMatrix<T,M,N>::const_iterator,    \
                            op <typename A::value_type,T>, M,N >, M,N > \
operator( const TFMExpr<A,M,N>& a, const FMatrix<T,M,N>& b)             \
{                                                                       \
  typedef TBinFMExprOp<TFMExpr<A,M,N>,                                  \
                       typename FMatrix<T,M,N>::const_iterator,         \
                       op <typename A::value_type,T>, M, N >            \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>( ExprT(a, b.begin()) );                     \
}

// array , expr
#define FMBINOP_AE(operator, op)                                        \
template<class A, class T, int M, int N>                                \
inline TFMExpr<TBinFMExprOp<typename FMatrix<T,M,N>::const_iterator,    \
                            TFMExpr<A,M,N>,                             \
                            op <T,typename A::value_type>,M,N >,M,N >   \
operator( const FMatrix<T,M,N>& a, const TFMExpr<A,M,N>& b )            \
{                                                                       \
  typedef TBinFMExprOp<typename FMatrix<T,M,N>::const_iterator,         \
                       TFMExpr<A,M,N>,                                  \
                       op <T,typename A::value_type>,M,N >              \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>( ExprT( a.begin(), b) );                    \
}

// expr , expr
#define FMBINOP_EE(operator, op)                                \
template<class A, class B, int M, int N>                        \
inline TFMExpr<TBinFMExprOp<TFMExpr<A,M,N>,                     \
                            TFMExpr<B,M,N>,                     \
                            op <typename A::value_type,         \
                                typename B::value_type>,        \
                            M,N >, M,N >                        \
operator( const TFMExpr<A,M,N>& a, const TFMExpr<B,M,N>& b )    \
{                                                               \
  typedef TBinFMExprOp<TFMExpr<A,M,N>,                          \
                       TFMExpr<B,M,N>,                          \
                       op <typename A::value_type,              \
                           typename B::value_type>,             \
                       M,N >                                    \
     ExprT;                                                     \
  return TFMExpr<ExprT,M,N>(ExprT(a,b));                        \
}

// array , literal
#define FMBINOP_AL(operator, op, lit_type)                              \
template<class T, int M, int N>                                         \
inline TFMExpr<TBinFMExprOp<typename FMatrix<T,M,N>::const_iterator,    \
                            TFMExprLiteral<lit_type>,                   \
                            op <T,lit_type>, M,N >, M,N >               \
operator( const FMatrix<T,M,N>& a, const lit_type& b )                  \
{                                                                       \
  typedef TBinFMExprOp<typename FMatrix<T,M,N>::const_iterator,         \
                       TFMExprLiteral<lit_type>,                        \
                       op <T,lit_type>, M,N >                           \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>( ExprT(a.begin(),b) );                      \
}

// expr , literal
#define FMBINOP_EL(operator, op, lit_type)                              \
template<class T, int M, int N>                                         \
inline TFMExpr<TBinFMExprOp<TFMExpr<T,M,N>,                             \
                            TFMExprLiteral<lit_type>,                   \
                            op <typename T::value_type, lit_type>,M,N >,M,N > \
operator( const TFMExpr<T,M,N>& a, const lit_type& b )                  \
{                                                                       \
  typedef TBinFMExprOp<TFMExpr<T,M,N>,                                  \
                       TFMExprLiteral<lit_type>,                        \
                       op <typename T::value_type, lit_type>, M,N >     \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>(ExprT(a,b));                                \
}

// literal , array
#define FMBINOP_LA(operator, op, lit_type)                              \
template<class T, int M, int N>                                         \
inline TFMExpr<TBinFMExprOp<TFMExprLiteral<lit_type>,                   \
                            typename FMatrix<T,M,N>::const_iterator,    \
                            op <lit_type, T>, M,N >, M,N >              \
operator( const lit_type& a, const FMatrix<T,M,N>& b )                  \
{                                                                       \
  typedef TBinFMExprOp<TFMExprLiteral<lit_type>,                        \
                       typename FMatrix<T,M,N>::const_iterator,         \
                       op <lit_type, T>, M,N >                          \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>(ExprT(a, b.begin()));                       \
}

// literal , expr
#define FMBINOP_LE(operator, op, lit_type)                              \
template<class T, int M, int N>                                         \
inline TFMExpr<TBinFMExprOp<TFMExprLiteral<lit_type>,                   \
                            TFMExpr<T,M,N>,                             \
                            op <lit_type, typename T::value_type>,M,N >,M,N > \
operator( const lit_type& a, const TFMExpr<T,M,N>& b )                  \
{                                                                       \
  typedef TBinFMExprOp<TFMExprLiteral<lit_type>,                        \
                       TFMExpr<T,M,N>,                                  \
                       op <lit_type, typename T::value_type>,M,N >      \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>(ExprT(a,b));                                \
}

//
// UNARY OPERATOR PARSE TREE TEMPLATES
//
// there are 2 cases for unary ops, namely:
//       op array,
// and   op expr.
// a template function operator is provided
// for each case and for each operator

// array
#define FMUNOP_A(operator, op)                                           \
template<class T, int M, int N>                                          \
inline TFMExpr<TUnFMExprOp<typename FMatrix<T,M,N>::const_iterator,      \
                           op <T>, M,N >, M,N >                          \
operator( const FMatrix<T,M,N>& a )                                      \
{                                                                        \
  typedef TUnFMExprOp<typename FMatrix<T,M,N>::const_iterator,           \
                      op <T>, M,N >                                      \
     ExprT;                                                              \
  return TFMExpr<ExprT,M,N>(ExprT(a.begin()));                           \
}


// expr
#define FMUNOP_E(operator, op)                                          \
template<class A, int M, int N>                                         \
inline TFMExpr<TUnFMExprOp<TFMExpr<A,M,N>,                              \
                           op <typename A::value_type>, M,N >, M,N >    \
operator( const TFMExpr<A,M,N>& a )                                     \
{                                                                       \
  typedef TUnFMExprOp<TFMExpr<A,M,N>, op <typename A::value_type>,M,N > \
     ExprT;                                                             \
  return TFMExpr<ExprT,M,N>(ExprT(a));                                  \
}




// convenience macro to define the parse tree templates
// (the applicative templates for all standard unary and
// binary operations are defined in misc/applicops.h, since
// those are common to MArray, FVector, and FMatrix

// Arguments: binary operator, arbitrary name
//
#define DECLARE_FMBINOP(operation, opname)                      \
   FMBINOP_AA(operator  operation,__ltl_##opname)               \
   FMBINOP_AE(operator  operation,__ltl_##opname)               \
   FMBINOP_EA(operator  operation,__ltl_##opname)               \
   FMBINOP_EE(operator  operation,__ltl_##opname)               \
   FMBINOP_AL(operator  operation,__ltl_##opname, float)        \
   FMBINOP_AL(operator  operation,__ltl_##opname, double)       \
   FMBINOP_AL(operator  operation,__ltl_##opname, int)          \
   FMBINOP_AL(operator  operation,__ltl_##opname, long)         \
   FMBINOP_EL(operator  operation,__ltl_##opname, float)        \
   FMBINOP_EL(operator  operation,__ltl_##opname, double)       \
   FMBINOP_EL(operator  operation,__ltl_##opname, int)          \
   FMBINOP_EL(operator  operation,__ltl_##opname, long)         \
   FMBINOP_LA(operator  operation,__ltl_##opname, float)        \
   FMBINOP_LA(operator  operation,__ltl_##opname, double)       \
   FMBINOP_LA(operator  operation,__ltl_##opname, int)          \
   FMBINOP_LA(operator  operation,__ltl_##opname, long)         \
   FMBINOP_LE(operator  operation,__ltl_##opname, float)        \
   FMBINOP_LE(operator  operation,__ltl_##opname, double)       \
   FMBINOP_LE(operator  operation,__ltl_##opname, int)          \
   FMBINOP_LE(operator  operation,__ltl_##opname, long)


// Arguments: binary operator, return type, arbitrary name
//
#define DECLARE_FMBINOP_RET(operation, ret_type, opname)        \
   FMBINOP_AA(operator  operation,__ltl_##opname)               \
   FMBINOP_AE(operator  operation,__ltl_##opname)               \
   FMBINOP_EA(operator  operation,__ltl_##opname)               \
   FMBINOP_EE(operator  operation,__ltl_##opname)               \
   FMBINOP_AL(operator  operation,__ltl_##opname, float)        \
   FMBINOP_AL(operator  operation,__ltl_##opname, double)       \
   FMBINOP_AL(operator  operation,__ltl_##opname, int)          \
   FMBINOP_AL(operator  operation,__ltl_##opname, long)         \
   FMBINOP_EL(operator  operation,__ltl_##opname, float)        \
   FMBINOP_EL(operator  operation,__ltl_##opname, double)       \
   FMBINOP_EL(operator  operation,__ltl_##opname, int)          \
   FMBINOP_EL(operator  operation,__ltl_##opname, long)         \
   FMBINOP_LA(operator  operation,__ltl_##opname, float)        \
   FMBINOP_LA(operator  operation,__ltl_##opname, double)       \
   FMBINOP_LA(operator  operation,__ltl_##opname, int)          \
   FMBINOP_LA(operator  operation,__ltl_##opname, long)         \
   FMBINOP_LE(operator  operation,__ltl_##opname, float)        \
   FMBINOP_LE(operator  operation,__ltl_##opname, double)       \
   FMBINOP_LE(operator  operation,__ltl_##opname, int)          \
   FMBINOP_LE(operator  operation,__ltl_##opname, long)



// Arguments: unary operator, arbitrary name
//
#define DECLARE_FMUNOP(operation, opname)         \
   FMUNOP_A(operator  operation,__ltl_##opname)   \
   FMUNOP_E(operator  operation,__ltl_##opname)



// Arguments: function, return type
//
#define DECLARE_FMBINARY_FUNC_(function, ret_type)      \
   FMBINOP_AA(function,__ltl_##function)                \
   FMBINOP_AE(function,__ltl_##function)                \
   FMBINOP_EA(function,__ltl_##function)                \
   FMBINOP_EE(function,__ltl_##function)                \
   FMBINOP_AL(function,__ltl_##function, float)         \
   FMBINOP_AL(function,__ltl_##function, double)        \
   FMBINOP_AL(function,__ltl_##function, int)           \
   FMBINOP_AL(function,__ltl_##function, long)          \
   FMBINOP_EL(function,__ltl_##function, float)         \
   FMBINOP_EL(function,__ltl_##function, double)        \
   FMBINOP_EL(function,__ltl_##function, int)           \
   FMBINOP_EL(function,__ltl_##function, long)          \
   FMBINOP_LA(function,__ltl_##function, float)         \
   FMBINOP_LA(function,__ltl_##function, double)        \
   FMBINOP_LA(function,__ltl_##function, int)           \
   FMBINOP_LA(function,__ltl_##function, long)          \
   FMBINOP_LE(function,__ltl_##function, float)         \
   FMBINOP_LE(function,__ltl_##function, double)        \
   FMBINOP_LE(function,__ltl_##function, int)           \
   FMBINOP_LE(function,__ltl_##function, long)



// Arguments: function, return type
//
#define DECLARE_FMUNARY_FUNC_(function, ret_type)              \
FMUNOP_A(function,__ltl_##function)                            \
FMUNOP_E(function,__ltl_##function)


// Finally, to make is easier for the user to declare their
// own functions for use in expression templates, provide
// single macros that declare both the applicative templates
// and the parse tree nodes
//
#define DECLARE_FM_BINARY_FUNC(function, ret_type)            \
MAKE_BINAP_FUNC( __ltl_##function, ret_type, function )       \
DECLARE_FMBINARY_FUNC_(function, ret_type)

#define DECLARE_FM_UNARY_FUNC(function, ret_type)             \
MAKE_UNAP_FUNC( __ltl_##function, ret_type, function )        \
DECLARE_FMUNARY_FUNC_(function, ret_type)

LTL_END_NAMESPACE

#endif // __LTL_FMATRIX_EXPR__

