/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: fvector.h,v 1.3.2.4 2005/10/07 19:55:46 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *                         Claus A. Goessl <cag@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#ifndef __LTL_FVECTOR__
#define __LTL_FVECTOR__


// ====================================================================

// little trick to avoid files being included separetely
#define __LTL_IN_FILE_FVECTOR__
// this is #undef'd at the end of this file ...

// ====================================================================

#include <ltl/config.h>
#include <ltl/misc/mdebug.h>

#include <iostream>
#include <iomanip>
#include <cmath>
#include <cstddef>

#ifdef LTL_USING_NAMESPACE
using std::ostream;
using std::endl;
using std::setw;
#endif

// ====================================================================

LTL_BEGIN_NAMESPACE
template<class T, int N, int S> class FVector;
LTL_END_NAMESPACE

#include <ltl/misc/staticinit.h>

#include <ltl/fvector/fvmemory.h>
#include <ltl/fvector/fviter.h>
#include <ltl/fvector/fvexpr.h>
#include <ltl/fvector/fvtloops.h>
#include <ltl/fvector/fvexpr_ops.h>
#include <ltl/fvector/fvbool.h>
#include <ltl/fvector/fvdot.h>

LTL_BEGIN_NAMESPACE

// ====================================================================

//! Vector whose length is known at compile time.
/*! \ingroup fvector_fmatrix

  Can either have its own memory region or reference foreign
  memory, for example when representing a column-vector of a matrix

  The template parameter S is a 'stride' for the foreign memory: when
  we reference a column vector we need a stride of M (if the matrix 
  is MxN). Memory handling is encapsulated in the class FVMemory.
  To make life easier for the compiler/optimizer, there
  is a specialization of FVMemory with _no_ stride at all and 
  an own new embedded block of memory. This is indicated 
  by S=0, which is also the default case.

  STL-compatible iterators and types.
*/
template<class T, int N, int S=0>
class FVector : public FVMemory<T,N,S>
{
      friend class FVIter<T,N,S>;
      friend class FVIterConst<T,N,S>;
      friend class ListInitializationSwitch<FVector<T,N,S> >;

   public:
      // STL-compatible type definitions
      typedef T                   value_type;
      typedef FVIter<T,N,S>       iterator;
      typedef FVIterConst<T,N,S>  const_iterator;
      typedef T*                  pointer;
      typedef const T*            const_pointer;
      typedef T&                  reference;
      typedef const T&            const_reference;
      typedef std::size_t         size_type;
      typedef std::ptrdiff_t      difference_type;

      //! Used by expression engine to decide which loops to unroll.
      // (See LTL_TEMPLATE_LOOP_LIMIT)
      enum { static_size = 1 };

      inline static size_type size()
      {
         return N;
      }

      inline static bool empty()
      {
         return false;
      }

      inline static size_type max_size()
      {
         return N;
      }

      // here we go
      FVector()
      { }

      ~FVector()
      { }

      FVector( const FVector<T,N,S>& other );

      // in the general case, we reference a
      // specialized for S=0, where we copy a into our memory
      FVector( T* const a );

      FVector( const T t );
      
      // construct from expression
      template<class Expr>
      FVector( const TFVExpr<Expr,N>& e );
      
      // assigns t to all elements
      // a bit more comlicated since we have to discriminate between
      // A = 3; and A = 1, 2, 3, 4;
      // which is done using ListInitializationSwitch which either calls
      // ListInitializer or MArray::fill().
      ListInitializationSwitch< FVector<T,N,S> > operator=( T x )
      {
         return ListInitializationSwitch< FVector<T,N,S> >( *this, x );
      }

      //
      // operator[] and operator() inherited from FVMemory
      //
      
      inline static int length()
      {
         return N;
      }

      inline static int minIndex()
      {
         return 1;
      }

      inline static int maxIndex()
      {
         return N;
      }

      inline iterator begin()
      {
         return FVIter<T,N,S>( *this );
      }

      inline const_iterator begin() const
      {
         return FVIterConst<T,N,S>( *this );
      }

      inline iterator end()
      {
         return FVIter<T,N,S>( *this, fviter_end_tag() );
      }

      inline const_iterator end() const
      {
         return FVIterConst<T,N,S>( *this, fviter_end_tag() );
      }

      void fill( const T x );
      
      // operator x for expressions
      template<class Expr>
      FVector<T,N,S>& operator=( const TFVExpr<Expr,N>& e );

      template<class Expr>
      FVector<T,N,S>& operator+=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator-=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator*=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator/=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator%=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator^=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator&=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator|=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator<<=( const TFVExpr<Expr,N>& e );
      template<class Expr>
      FVector<T,N,S>& operator>>=( const TFVExpr<Expr,N>& e );
      
      // operator x for vectors
      template<class T2, int S2>
      FVector<T,N,S>& operator=( const FVector<T2,N,S2>& v );

      FVector<T,N,S>& operator=( const FVector<T,N,S>& v );

      template<class T2, int S2>
      FVector<T,N,S>& operator+=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator-=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator*=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator/=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator%=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator^=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator&=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator|=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator<<=( const FVector<T2,N,S2>& v );
      template<class T2, int S2>
      FVector<T,N,S>& operator>>=( const FVector<T2,N,S2>& v );
      
      FVector<T,N,S>& operator+=( const T t );
      FVector<T,N,S>& operator-=( const T t );
      FVector<T,N,S>& operator*=( const T t );
      FVector<T,N,S>& operator/=( const T t );
      FVector<T,N,S>& operator%=( const T t );
      FVector<T,N,S>& operator^=( const T t );
      FVector<T,N,S>& operator&=( const T t );
      FVector<T,N,S>& operator|=( const T t );
      FVector<T,N,S>& operator<<=( const T t);
      FVector<T,N,S>& operator>>=( const T t);

      template<class T2, int S2>
      void swap( FVector<T2, N, S2>& other );
};

/*! \relates ltl::FVector
 */
template<class T, int N, int S>
ostream& operator<<(ostream& os, const FVector<T,N,S>& x)
{
   os << "FVector< T,"<<N<<","<<S<<" >" << endl;
   os << " [ ";
   for (int i=x.minIndex(); i <= x.maxIndex(); ++i)
   {
      os << x(i) << " ";
      if (!((i+1-x.minIndex())%9))
         os << endl << "  ";
   }
   os << " ]";
   return os;
}

LTL_END_NAMESPACE

#include <ltl/fvector/fvector_ops.h>
#include <ltl/fvector/fvector_methods.h>

#undef __LTL_IN_FILE_FVECTOR__

#endif //__LTL_FVECTOR__
