/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: fvmemory.h,v 1.4.2.1 2003/11/02 23:23:43 cag Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2001 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#ifndef __LTL_IN_FILE_FVECTOR__
#error "<ltl/fvector/fvmemory.h> must be included via <ltl/fvector.h>, never alone!"
#endif


#ifndef __LTL_FV_MEMORY__
#define __LTL_FV_MEMORY__

#include <ltl/config.h>

LTL_BEGIN_NAMESPACE

/*! \file fvmemory.h
  Encapsulates the memory for a ltl::FVector object: It can either be
  a fixed size embedded block, or a reference to another block.
  In the second case the stride may be != 1, such that column vectors
  of a matrix can be referenced.
  
  The ltl::FVMemory objects implement the memory access interface of
  the ltl::FVector object, i.e. operator() for 1-based access and
  operator[] for 0-based access
*/


/*! The general case stores N objects of type T accessible through
  a T* with stride S.
  This is the reference case
*/ 
template<class T, int N, int S>
class FVMemory
{
   public:
      // reference data
      inline FVMemory()
      { }

      inline FVMemory(  T* const data )
         : __data_(data), data_(__data_-S)
      { }
      
      inline T operator[]( const int i ) const
      {
         LTL_ASSERT( i>=0 && i<N, 
                     "Direct index "<<i<<" out of bounds in FVector[] of length "<<N );
         return __data_[i*S];
      }

      inline T& operator[]( const int i )
      {
         LTL_ASSERT( i>=0 && i<N, 
                     "Direct index "<<i<<" out of bounds in FVector[] of length "<<N );
         return __data_[i*S];
      }

      inline T operator()( const int i ) const
      {
         LTL_ASSERT( i>0 && i<=N, 
                     "Index "<<i<<" out of bounds in FVector() of length "<<N );
         return data_[i*S];
      }

      inline T& operator()( const int i )
      {
         LTL_ASSERT( i>0 && i<=N, 
                     "Index "<<i<<" out of bounds in FVector() of length "<<N );
         return data_[i*S];
      }

      inline T* data()
      {
         return __data_;
      }

      inline const T* data() const
      {
         return __data_;
      }

#ifndef __SUNPRO_CC      
   protected:
#endif
      T* __data_;
      T* data_;
};



/*! Specialization for stride 1 (indicated through S=0).
  This is the case for every freshly allocated FVector object
  having its own memory.
*/ 
template<class T, int N>
class FVMemory<T,N,0>
{
   public:
      inline FVMemory()
         : data_(__data_-1)
      { }

      // copy data over
      inline FVMemory( const T* const restrict_ data )
         : data_(__data_-1)
      { 
         for( int i=0; i<N; ++i )
            (*this)[i] = data[i];
      }

      inline FVMemory( const FVMemory<T,N,0>& restrict_ other )
         : data_(__data_-1)
      {
// copying is done now in fvector_methods.h
//          for( int i=0; i<N; ++i )
//             __data_[i] = other.__data_[i];
      }

      inline T operator[]( const int i ) const
      {
         LTL_ASSERT( i>=0 && i<N, 
                     "Direct index "<<i<<" out of bounds in FVector[] of length "<<N );
         return __data_[i];
      }

      inline T& restrict_ operator[]( const int i )
      {
         LTL_ASSERT( i>=0 && i<N, 
                     "Direct index "<<i<<" out of bounds in FVector[] of length "<<N );
         return __data_[i];
      }

      inline T operator()( const int i ) const
      {
         LTL_ASSERT( i>0 && i<=N, 
                     "Index "<<i<<" out of bounds in FVector() of length "<<N );
         return data_[i];
      }

      inline T& restrict_ operator()( const int i )
      {
         LTL_ASSERT( i>0 && i<=N, 
                     "Index "<<i<<" out of bounds in FVector() of length "<<N );
         return data_[i];
      }

      inline T* restrict_ data()
      {
         return __data_;
      }

      inline const T* restrict_ data() const
      {
         return __data_;
      }
#ifndef __SUNPRO_CC      
   protected:
#endif
      T  __data_[N];
      T* restrict_ data_;
};

LTL_END_NAMESPACE

#endif // __LTL_FV_MEMORY__
