/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: ascio.cpp,v 1.5.2.3 2005/10/03 22:59:48 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Jan Snigula  <sniglua@usm.uni-muenchen.de>
 *                         Niv Drory    <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#include <ltl/ascio.h>

#ifdef LTL_USING_NAMESPACE
LTL_BEGIN_NAMESPACE
#endif


AscFile::AscFile( const string &fname, const char delim,
                  const string &comment )
      : filename_(fname), in_(fname.c_str(), ios::in),
      begin_(0), end_(-1), rows_(0), cols_(0), delim_(delim), comment_(comment)
{
   if( !in_.good() )
      throw IOException( "Cannot open file "+filename_+"!" );
}


AscFile::AscFile( const string &fname, const int b, int e,
                  const char delim,
                  const string &comment )
      : filename_(fname), in_(fname.c_str(), ios::in),
      begin_(0), end_(-1), rows_(0), cols_(0), delim_(delim), comment_(comment)
{
   if( !in_.good() )
      throw IOException( "Cannot open file "+filename_+"!" );

   if( b < 1 || ( b > e && e != -1 ) )
      throw IOException( "Irregular lines parameters!" );

   bool foundline=false;
   string buff;
   int count=1;

   if( b != 1 )
   {
      while( readNextLine__(buff) )
         if( ++count > b )
         {
            foundline=true;
            break;
         }

      if( !foundline )
      {
         OSTREAM err;
         err << "No such line: " << b;
         throw IOException( err.str() );
      }
      // Now we fool reset the linecounter to the beginning of the current line
      //	    cerr << "buff is " << buff << endl;
      int l =  buff.length()+1;
      in_.seekg( -l, ios::cur );
      --count;

      begin_ = in_.tellg();
   }

   foundline=false;

   if( e != -1 )
   {
      while( readNextLine__(buff) )
         if ( ++count > e )
         {
            foundline=true;
            break;
         }

      if( !foundline )
         // Last line cannot be found using EOF
         end_ = -1;
      else
         end_ = in_.tellg();
   }
   rewind__();
}

AscFile::~AscFile()
{
   if( in_.is_open() )
      in_.close();
}

int AscFile::rows() throw(IOException)
{
   if( rows_ == 0 )
      countRows__();
   return rows_;
}

int AscFile::cols() throw(IOException)
{
   if( cols_ == 0 )
      countCols__();
   return cols_;
}


/*! Try to read the next available line from the input filestream.
  Returns false if no more lines can be read.
  \throw ltl::IOException() if the file cannot be opened.
*/
bool AscFile::readNextLine__( string &buff )
{

   buff = ""; // Empty the buffer

   if( eof__() )
   {
      rewind__();
      return false;
   }
   getline( in_, buff );

   if( delim_ == 0 )
   { // Eat leading whitespaces
      unsigned int pos=0;
      while( pos <= buff.length() )
      {
         if( !isspace( buff[pos++] ) )
            break;
      }
      buff = buff.substr( --pos, string::npos );
   }

   string::size_type start = buff.find( comment_, 0 );

   if( start != string::npos )
   { // Found a comment in the line!!
      buff = buff.substr( 0, start );
   }

   // We finished all modifications to buff at this point
   if( buff == "" ) // Line is empty... skip it
      return readNextLine__( buff );

   return true;
}

string AscFile::readColumnFromLine__( const int col, const string &line )
{
   if( delim_ == 0 )
   {
      string buff = "";
      int colcount = 0;

      ISTREAM is(line.c_str());
      while( colcount++ != col )
      {
         if( is.eof() )
            throw IOException( "Unexpected EOF encountered!" );
         is >> buff;
      }

      return buff;
   }
   else
   {
      typedef string::size_type ST;
      int colcount = 0;
      ST start = 0;
      ST end = 0;

      if( col == 1 )
      {
         /* The line starts with a delimiter,
            so the first col is empty... */
         if( line[0] == delim_ )
            return "";
         else
         {
            /* We must treat the first column as a special case, to
               prevent that the first char of the first column is deleted */
            end = line.find( delim_, start );
            return line.substr( start, end );
         }
      }

      while( ++colcount != col )
      {
         start = line.find( delim_, end );
         if( start == string::npos )
            throw IOException( "Column does not exist!" ); // no such column!
         end = start+1;
      }

      end = line.find( delim_, start+1 );
      return line.substr( start+1, (end-start-1) );
   }
}

string AscFile::replaceColumnsInLine__( const int col1, const int col2, string & line, const string & rep )
{
   typedef string::size_type ST;
   
   int colcount = 0;
   ST start = 0;
   ST end = 0;
   
   if( col1 > col2 ) {
      throw IOException( "Start col > end col!" ); // no such column!
   }

   string d="";
   
   if( delim_ == 0 ) 
      d = " \t";
   else
      d=delim_;
   
   if( col2 > cols() ) // Easy hack to replace to end of line
      end = string::npos;
   
   if( col1 == 1 ) 
      start = line.find_first_not_of( d );
   else {
      ST t1 = line.find_first_not_of( d, 0 );
      ST t2 = line.find_first_of( d, t1 );
      colcount = 1;
      while( colcount != col1 ) {
         t1 = line.find_first_not_of( d, t2 );
         t2 = line.find_first_of( d, t1 );
         ++colcount;
      }
      
      start = t1;
   }
      
   if( col1 == col2 )
      end = line.find_first_of( d, start );
   else if( !end ) {

      ST t1 = start;
      ST t2 = line.find_first_of( d, t1 );
      while( colcount != col2 ) {
         t1 = line.find_first_not_of( d, t2 );
         t2 = line.find_first_of( d, t1 );
         ++colcount;
      }
      
      end = t2;
   }

   return line.replace( start, end-start, rep )+"\n"; // Possible bug here!
     
}

void AscFile::rewind__()
{
   in_.clear();
   in_.seekg(begin_);
}

bool AscFile::eof__()
{
   if( end_ == streampos(-1) )
      return in_.eof();
   else
      return ( in_.eof() || (in_.tellg()>=end_) );
}

void AscFile::countCols__()
{
   typedef string::size_type ST;
   string buff;

   // Remember the current position in the file
   streampos fp = in_.tellg();
   rewind__();

   if( !readNextLine__( buff ) )
   {
      cols_ = 0;
      return;
   }

   int colcount = 0;
   ST start = 0;

   if( delim_ == 0 )
   {
      ISTREAM is( buff.c_str() );
      string devnull;
      while( is >> devnull )
      {
         if( devnull != "" )
            ++colcount;
      }
   }
   else
   {
      while( start != string::npos )
      {
         start = buff.find( delim_, start );
         if( start != string::npos )
         {
            ++start;
            ++colcount;
         }
         else
            break;
      }
      ++colcount;
   }
   cols_ = colcount;

   // Reposition the input stream

   in_.clear();
   in_.seekg( fp );

   return;
}

void AscFile::countRows__()
{
   streampos fp = in_.tellg();
   rewind__();

   string buff;
   int rowcount = 0;
   while( readNextLine__(buff) )
   {
      ++rowcount;
   }

   rows_ = rowcount;

   // Rewind the file stream

   in_.clear();
   in_.seekg( fp );

}

template<class T>
MArray<T,1> AscFile::readSingleCol__( const int col )
{
   string buff, b;
   int count = 0;
   T tbuff;

   MArray<T,1> a( rows() );

   while( readNextLine__( buff ) )
   {
      b = readColumnFromLine__( col, buff );
      ISTREAM is( b.c_str() );
      is >> tbuff;
      if( is.bad() )
         throw IOException( "Bad stream state!" );
      if( is.fail() )
         throw IOException( "Failed to read data from file!" );
      a( ++count ) = tbuff;
   }

   return a;
}

template<class T>
MArray<T,2> AscFile::readCols__( int first, int last )
{
   string buff, b;
   T tbuff;

   if( first==0 )
      first = 1;

   if( last==0 )
      last = cols();

   MArray<T,2> a( last - first + 1, rows() );

   int count = 1;
   while( readNextLine__( buff ) )
   {
      for( int i = first; i<= last; i++ )
      {
         b = readColumnFromLine__( i, buff );
         ISTREAM is( b.c_str() );
         is >> tbuff;
         if( is.bad() )
            throw IOException( "Bad stream state!" );
         if( is.fail() )
            throw IOException( "Failed to read data from line: " + buff );
         a( i-first+1, count ) = tbuff;
      }
      count++;
   }
   return a;
}

MArray<int,1> AscFile::readIntColumn( const int col ) throw(IOException)
{
   return readSingleCol__<int>(col);
}

MArray<float,1> AscFile::readFloatColumn( const int col ) throw(IOException)
{
   return readSingleCol__<float>(col);
}

MArray<double,1> AscFile::readDoubleColumn( const int col ) throw(IOException)
{
   return readSingleCol__<double>(col);
}

MArray<int,2> AscFile::readIntColumns( const int first,
                                       const int last ) throw(IOException)
{
   return readCols__<int>( first, last );
}

MArray<float,2> AscFile::readFloatColumns( const int first,
      const int last ) throw(IOException)
{
   return readCols__<float>( first, last );
}

MArray<double,2> AscFile::readDoubleColumns( const int first,
      const int last ) throw(IOException)
{
   return readCols__<double>( first, last );
}

int AscFile::getHeader( vector<string> &v, bool keepcs )
{
   /* Read the comment header from the beginning of the file 
    and return the number of lines read */

   string buff; // Empty the buffer

   rewind__(); // Rewind the filepointer to the begin

   while( !eof__() )
   {
      
      buff = "";

      getline( in_, buff ); // Read one line

      // Strip leading whitespace
      unsigned int pos=0;
      while( isspace( buff[pos] ) && pos <= buff.length() )
         ++pos;

      if( pos )
         buff = buff.substr( pos, string::npos );
      
      // If line empty continue
      if( buff == "" ) 
         continue;

      if( buff.length() >= comment_.length() )
         if( buff.substr( 0, comment_.length() ) != comment_ ) // Found a not empty line 
            break;

      if( !keepcs ) // Strip comment delimiter
         buff = buff.substr( comment_.length(), string::npos );

      // Delete trailing newline
      buff = buff.substr(0, buff.rfind( "\n" ) );
      
      // We came here so this is a valid comment -> add it to the vector
      v.push_back( buff );
   }

   rewind__(); // Rewind the filepointer to the begin

   return v.size();
}

// Specialization for vector<string>, gets rid of istringstream hack

template<>
int AscFile::readColumn( const int col, vector<string> &cont ) throw(IOException)
{
   string buff;
   int count = 0;

   if( size_t(rows()) > cont.max_size() )
      throw IOException( "A memory error occured!" );

   cont.reserve(rows());
   cont.clear();

   while( readNextLine__( buff ) )
   {
      cont.push_back( readColumnFromLine__( col, buff ) );
      ++count;
   }

   return count;
}


MArray<float,2> AscFile::readFloatColumns( const int* cols, const int ncols )
{
   string buff, tmp="";
   float tbuff;
   
   if( delim_ != 0 )
      throw IOException( "Can only read whitespace delims" );

   MArray<float,2> a( ncols, rows() );
   
   int count = 1;
   
   // faster version for whitespace delimiters
   while( readNextLine__( buff ) )
   {
      ISTREAM is( buff.c_str() );

      int j = 0;
      for( int i=1; i<=cols[ncols-1]; ++i )
      {
         if( i == cols[j] )
         {
            is >> tbuff;
            a( ++j, count ) = tbuff;
         }
         else
            is >> tmp;
      }
      if( is.bad() )
         throw IOException( "Bad stream state!" );
      if( is.fail() )
         throw IOException( "Failed to read data from line: " + buff );
      count++;
   }      
   
   return a;
}

#ifdef LTL_USING_NAMESPACE
LTL_END_NAMESPACE
#endif

