/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: expr.h,v 1.2.2.4 2005/03/17 12:33:59 snigula Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */


#ifndef __LTL_IN_FILE_MARRAY__
#error "<ltl/marray/expr.h> must be included via <ltl/marray.h>, never alone!"
#endif


#ifndef __LTL_EXPR__
#define __LTL_EXPR__

#include <ltl/marray/shape_iter.h>
#include <ltl/marray/expr_iter.h>

LTL_BEGIN_NAMESPACE

template<class T, int D>
class MArray;

template<class T, int N>
class IndexIterator;

template<int N>
class Shape;

template<class A, int N>
class TExpr;

template<class A, class B, class Op, int N>
class TBinExprOp;

template<class A, class Op, int N>
class TUnExprOp;

template<class T>
class TExprLiteral;


// this is just to keep everything together in class browsers
class _et_parse_base
   { }
;

// when determining the shape of an expression in MArray<T,N>( TExpr<> )
// ctor we have make sure we don't ask a literal for a shape ...
//
// in the general case, just use the first Shape ... (unless compiling with xlC)
#ifndef __xlC__

template<class A, class B>
inline const Shape<A::dims>*
_expr_getshape( const A& a, const B& b )
{
   return a.shape();
}

#else

// the IBM xlC compiler does not like this, so let it have only 
// specialized versions
template<class A, class T, int N>
inline const Shape<N>*
_expr_getshape( const MArrayIterConst<A,N>& a, const MArrayIterConst<T,N>& b )
{
   return a.shape();
}
template<class A, class T, int N>
inline const Shape<N>*
_expr_getshape( const TExpr<A,N>& a, const TExpr<T,N>& b )
{
   return a.shape();
}
template<class A, class T, int N>
inline const Shape<N>*
_expr_getshape( const TExpr<A,N>& a, const MArrayIterConst<T,N>& b )
{
   return b.shape();
}
template<class A, class T, int N>
inline const Shape<N>*
_expr_getshape( const MArrayIterConst<T,N>& a, const TExpr<A,N>& b ) 
{
   return a.shape();
}
#endif  //__xlC__

// provide specializations for the case one of the operands is
// a literal: in that cas, always use the other operand's Shape.
//
template<class A, class T, int N>
inline const Shape<N>*
_expr_getshape( const TExpr<A,N>& a, const TExprLiteral<T>& l )
{
   return a.shape();
}
template<class A, class T, int N>
inline const Shape<N>*
_expr_getshape( const MArrayIterConst<A,N>& a, const TExprLiteral<T>& l )
{
   return a.shape();
}
template<class T, class A, int N>
inline const Shape<N>*
_expr_getshape( const TExprLiteral<T>& l, const TExpr<A,N>& a )
{
   return a.shape();
}
template<class T, class A, int N>
inline const Shape<N>*
_expr_getshape( const TExprLiteral<T>& l, const MArrayIterConst<A,N>& a )
{
   return a.shape();
}


//
// binary operation node
//
template<class A, class B, class Op, int N>
class TBinExprOp : public _et_parse_base
{
   private:
      A iter1_;
      B iter2_;

   public:
      typedef typename Op::value_type value_type;
      enum { dims=N };
      enum { numIndexIter = A::numIndexIter + B::numIndexIter };
      enum { isVectorizable = A::isVectorizable * B::isVectorizable * Op::isVectorizable };

      inline TBinExprOp( const A& a, const B& b )
            : iter1_(a), iter2_(b)
      { }

      inline void operator++()
      {
         ++iter1_;
         ++iter2_;
      }

      inline void advance()
      {
         iter1_.advance();
         iter2_.advance();
      }

      inline void advanceN( const int i )
      {
         iter1_.advanceN(i);
         iter2_.advanceN(i);
      }

      inline void advanceWithStride1()
      {
         iter1_.advanceWithStride1();
         iter2_.advanceWithStride1();
      }

      inline void advanceDim()
      {
         iter1_.advanceDim();
         iter2_.advanceDim();
      }

      inline value_type operator*() const
      {
         return Op::eval( *iter1_, *iter2_ );
      }

      inline value_type readWithoutStride( const int i ) const
      {
         return Op::eval( iter1_.readWithoutStride(i),
                          iter2_.readWithoutStride(i) );
      }

      inline value_type readWithStride( const int i ) const
      {
         return Op::eval( iter1_.readWithStride(i),
                          iter2_.readWithStride(i) );
      }

#ifdef LTL_USE_SIMD
      inline typename VEC_TYPE(value_type) readVec( const int i ) const
      {
         return Op::eval_vec( iter1_.readVec(i), iter2_.readVec(i) );
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return iter1_.sameAlignmentAs(p) && iter2_.sameAlignmentAs(p);
      }
#endif

      inline bool isStorageContiguous() const
      {
         return iter1_.isStorageContiguous() && iter2_.isStorageContiguous();
      }

      inline bool isStride1() const
      {
         return iter1_.isStride1() && iter2_.isStride1();
      }

      inline bool isConformable( const Shape<N>& other ) const
      {
         return iter1_.isConformable( other ) &&
                iter2_.isConformable( other );
      }

      inline void reset()
      {
         iter1_.reset();
         iter2_.reset();
      }

      inline const Shape<N> *shape() const
      {
         return _expr_getshape( iter1_, iter2_ );
      }
};


//
// unary operation node
//
template<class A, class Op, int N>
class TUnExprOp : public _et_parse_base
{
   private:
      A iter_;

   public:
      typedef typename Op::value_type value_type;
      enum { dims=N };
      enum { numIndexIter = A::numIndexIter };
      enum { isVectorizable = A::isVectorizable * Op::isVectorizable };

      inline TUnExprOp( const A& a )
            : iter_(a)
      { }

      inline void operator++()
      {
         ++iter_;
      }

      inline void advance()
      {
         iter_.advance();
      }

      inline void advanceN( const int i )
      {
         iter_.advanceN(i);
      }

      inline void advanceWithStride1()
      {
         iter_.advanceWithStride1();
      }

      inline void advanceDim()
      {
         iter_.advanceDim();
      }

      inline value_type operator*() const
      {
         return Op::eval( *iter_ );
      }

      inline value_type readWithoutStride( const int i ) const
      {
         return Op::eval( iter_.readWithoutStride(i) );
      }

      inline value_type readWithStride( const int i ) const
      {
         return Op::eval( iter_.readWithStride(i) );
      }

#ifdef LTL_USE_SIMD
      inline typename VEC_TYPE(value_type) readVec( const int i ) const
      {
         return Op::eval_vec( iter_.readVec(i) );
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return iter_.sameAlignmentAs(p);
      }
#endif

      inline bool isStorageContiguous() const
      {
         return iter_.isStorageContiguous();
      }

      inline bool isStride1() const
      {
         return iter_.isStride1();
      }

      inline bool isConformable( const Shape<N>& other ) const
      {
         return iter_.isConformable( other );
      }

      inline void reset()
      {
         iter_.reset();
      }

      inline const Shape<N> *shape() const
      {
         return iter_.shape();
      }
};


//
// Literal number
//
template<class T>
class TExprLiteral : public _et_parse_base
{
   private:
#ifdef LTL_USE_SIMD
     union { typename VEC_TYPE(T) v_; T ff_[]; } vec_;
#endif
     const T f_;

   public:
      typedef T value_type;
      enum { dims=0 };
      enum { numIndexIter = 0 };
      enum { isVectorizable = 1 };

      inline TExprLiteral( const T f ) : f_(f)
      { 
#ifdef LTL_USE_SIMD
         for(unsigned int i=0; i<sizeof(typename VEC_TYPE(T))/sizeof(T); ++i)
            vec_.ff_[i] = f_;
#endif
      }

      inline value_type operator*() const
      {
         return f_;
      }

      inline value_type readWithoutStride( const int i ) const
      {
         return f_;
      }

      inline value_type readWithStride( const int i ) const
      {
         return f_;
      }

#ifdef LTL_USE_SIMD
      inline typename VEC_TYPE(value_type) readVec( const int i ) const
      {
         return vec_.v_;
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return true;
      }
#endif

      inline void advance() const
         { }

      inline void advanceN( const int i ) const
         { }

      inline void advanceWithStride1() const
         { }

      inline void advanceDim() const
         { }

      inline void operator++() const
         { }

      inline bool isStorageContiguous() const
      {
         return true;
      }

      inline bool isStride1() const
      {
         return true;
      }

      template<int N>
      inline bool isConformable( const Shape<N>& other ) const
      {
         return true;
      }

      inline void reset()
      {  }
};


//
// now the expression class itself
//
template<class A, int N>
class TExpr : public _et_parse_base
{
   private:
      A iter_;

   public:
      typedef typename A::value_type value_type;
      enum { dims=N };
      enum { numIndexIter = A::numIndexIter };
      enum { isVectorizable = A::isVectorizable };

      inline TExpr( const A& a )
            : iter_(a)
      { }

      inline void operator++()
      {
         ++iter_;
      }

      inline void advance()
      {
         iter_.advance();
      }

      inline void advanceN( const int i )
      {
         iter_.advanceN(i);
      }

      inline void advanceWithStride1()
      {
         iter_.advanceWithStride1();
      }

      inline void advanceDim()
      {
         iter_.advanceDim();
      }

      inline value_type operator*() const
      {
         return *iter_;
      }

      inline value_type readWithoutStride( const int i ) const
      {
         return iter_.readWithoutStride(i);
      }

      inline value_type readWithStride( const int i ) const
      {
         return iter_.readWithStride(i);
      }

#ifdef LTL_USE_SIMD
      inline typename VEC_TYPE(value_type) readVec( const int i ) const
      {
         return iter_.readVec(i);
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return iter_.sameAlignmentAs(p);
      }
#endif

      inline bool isStorageContiguous() const
      {
         return iter_.isStorageContiguous();
      }

      inline bool isStride1() const
      {
         return iter_.isStride1();
      }

      inline bool isConformable( const Shape<N>& other ) const
      {
         return iter_.isConformable( other );
      }

      inline void reset()
      {
         iter_.reset();
      }

      const Shape<N> *shape() const
      {
         return iter_.shape();
      }

      inline ExprIter<A,N> begin()
      {
         return ExprIter<A,N>( *this );
      }

      inline ExprIter<A,N> end()
      {
         return ExprIter<A,N>( *this, true );
      }
};


//
// we need a trait class to deal with literals
// basically convert everything to an expression TExpr
//
template<class T>
struct asExpr
{
   typedef TExprLiteral<T> value_type;
};

// Already an expression template term
template<class T, int N>
struct asExpr< TExpr<T,N> >
{
   typedef TExpr<T,N> value_type;
};

// An array operand
template<class T, int N>
struct asExpr< MArray<T,N> >
{
   typedef typename MArray<T,N>::ConstIterator value_type;
};


//
// that's all ...
// the rest are the operator templates
//


#define ITER(a) a.begin()

//
// BINARY OPERATOR PARSE TREE TEMPLATES
//
// there are basically 8 cases for binary ops, namely:
//
//   array   op  array,
//   expr    op  array,
//   array   op  expr,
//   expr    op  expr,
//
//   array   op  literal,
//   expr    op  literal,
//   literal op  array,  and
//   literal op  expr
//
// a template version of the overloaded operator is provided
// for each case and for each operator

// array , array
#define BINOP_AA(operator, op)                                  \
template<class T1, class T2, int N>                             \
inline TExpr<TBinExprOp<typename MArray<T1,N>::ConstIterator,   \
                 typename MArray<T2,N>::ConstIterator,          \
                 op <T1,T2>, N >, N >                           \
operator(const MArray<T1,N>& a, const MArray<T2,N>& b)          \
{                                                               \
  typedef TBinExprOp<typename MArray<T1,N>::ConstIterator,      \
                     typename MArray<T2,N>::ConstIterator,      \
                     op <T1,T2>, N >                            \
          ExprT;                                                \
  return TExpr<ExprT,N>(ExprT(ITER(a),ITER(b)));                \
}

// expr , array
#define BINOP_EA(operator,op)                                   \
template<class A, class T, int N>                               \
inline TExpr<TBinExprOp<TExpr<A,N>,                             \
                 typename MArray<T,N>::ConstIterator,           \
                 op <typename A::value_type,T>, N >, N >        \
operator( const TExpr<A,N>& a, const MArray<T,N>& b)            \
{                                                               \
  typedef TBinExprOp<TExpr<A,N>,                                \
                     typename MArray<T,N>::ConstIterator,       \
                     op <typename A::value_type,T>, N >         \
          ExprT;                                                \
  return TExpr<ExprT,N>(ExprT(a,ITER(b)));                      \
}

// array , expr
#define BINOP_AE(operator, op)                                          \
template<class A, class T, int N>                                       \
inline TExpr<TBinExprOp<typename MArray<T,N>::ConstIterator,            \
             TExpr<A,N>,                                                \
             op <T,typename A::value_type>, N >, N >                    \
operator( const MArray<T,N>& a, const TExpr<A,N>& b )                   \
{                                                                       \
  typedef TBinExprOp<typename MArray<T,N>::ConstIterator,               \
                     TExpr<A,N>,                                        \
                     op <T,typename A::value_type>, N >                 \
          ExprT;                                                        \
  return TExpr<ExprT,N>(ExprT(ITER(a),b));                              \
}

// expr , expr
#define BINOP_EE(operator, op)                                                \
template<class A, class B, int N>                                             \
inline TExpr<TBinExprOp<TExpr<A,N>,                                           \
                 TExpr<B,N>,                                                  \
                 op <typename A::value_type, typename B::value_type>,         \
                 N >, N >                                                     \
operator( const TExpr<A,N>& a, const TExpr<B,N>& b )                          \
{                                                                             \
  typedef TBinExprOp<TExpr<A,N>,                                              \
                     TExpr<B,N>,                                              \
                     op <typename A::value_type, typename B::value_type>, N > \
          ExprT;                                                              \
  return TExpr<ExprT,N>(ExprT(a,b));                                          \
}

// array , literal
#define BINOP_AL(operator, op, lit_type)                                \
template<class T, int N>                                                \
inline TExpr<TBinExprOp<typename MArray<T,N>::ConstIterator,            \
                        TExprLiteral<lit_type>,                         \
                        op <T,lit_type>, N >, N >                       \
operator( const MArray<T,N>& a, const lit_type& b )                     \
{                                                                       \
  typedef TBinExprOp<typename MArray<T,N>::ConstIterator,               \
                     TExprLiteral<lit_type>,                            \
                     op <T,lit_type>, N >                               \
          ExprT;                                                        \
  return TExpr<ExprT,N>(ExprT(ITER(a),b));                              \
}

// expr , literal
#define BINOP_EL(operator, op, lit_type)                        \
template<class T, int N>                                        \
inline TExpr<TBinExprOp<TExpr<T,N>,                             \
             TExprLiteral<lit_type>,                            \
             op <typename T::value_type, lit_type>, N >, N >    \
operator( const TExpr<T,N>& a, const lit_type& b )              \
{                                                               \
  typedef TBinExprOp<TExpr<T,N>,                                \
                     TExprLiteral<lit_type >,                   \
                     op <typename T::value_type, lit_type>, N > \
          ExprT;                                                \
  return TExpr<ExprT,N>(ExprT(a,b));                            \
}

// literal , array
#define BINOP_LA(operator, op, lit_type)                                \
template<class T, int N>                                                \
inline TExpr<TBinExprOp<TExprLiteral<lit_type>,                         \
                        typename MArray<T,N>::ConstIterator,            \
                        op <lit_type, T>, N >, N >                      \
operator( const lit_type& a, const MArray<T,N>& b )                     \
{                                                                       \
   typedef TBinExprOp<TExprLiteral<lit_type>,                           \
                     typename MArray<T,N>::ConstIterator,               \
                     op <lit_type, T>, N >                              \
          ExprT;                                                        \
  return TExpr<ExprT,N>(ExprT(a,ITER(b)));                              \
}

// literal , expr
#define BINOP_LE(operator, op, lit_type)                        \
template<class T, int N>                                        \
inline TExpr<TBinExprOp<TExprLiteral<lit_type>,                 \
             TExpr<T,N>,                                        \
             op <lit_type, typename T::value_type>, N >, N >    \
operator( const lit_type& a, const TExpr<T,N>& b )              \
{                                                               \
   typedef TBinExprOp<TExprLiteral<lit_type>,                   \
                     TExpr<T,N>,                                \
                     op <lit_type, typename T::value_type>, N > \
          ExprT;                                                \
  return TExpr<ExprT,N>(ExprT(a,b));                            \
}

//
// UNARY OPERATOR PARSE TREE TEMPLATES
//
// there are 2 cases for unary ops, namely:
//       op array,
// and   op expr.
// a template function operator is provided
// for each case and for each operator

// array
#define UNOP_A(operator, op)                                                 \
template<class T,int N>                                                      \
inline TExpr<TUnExprOp<typename MArray<T,N>::ConstIterator, op <T>, N >, N > \
operator( const MArray<T,N>& a )                                             \
{                                                                            \
  typedef TUnExprOp<typename MArray<T,N>::ConstIterator, op <T>, N >         \
          ExprT;                                                             \
  return TExpr<ExprT,N>(ExprT(ITER(a)));                                     \
}


// expr
#define UNOP_E(operator, op)                                            \
template<class A, int N>                                                \
inline TExpr<TUnExprOp<TExpr<A,N>,op <typename A::value_type>, N >, N > \
operator( const TExpr<A,N>& a )                                         \
{                                                                       \
  typedef TUnExprOp<TExpr<A,N>, op <typename A::value_type>, N >        \
          ExprT;                                                        \
  return TExpr<ExprT,N>(ExprT(a));                                      \
}




// convenience macro to define the parse tree templates
// (the applicative templates for all standard unary and
// binary operations are defined in misc/applicops.h, since
// those are common to MArray, FVector, and FMatrix

// Arguments: binary operator, arbitrary name
//
#define DECLARE_BINOP(operation, opname)                        \
   BINOP_AA(operator  operation,__ltl_##opname)                 \
   BINOP_AE(operator  operation,__ltl_##opname)                 \
   BINOP_EA(operator  operation,__ltl_##opname)                 \
   BINOP_EE(operator  operation,__ltl_##opname)                 \
   BINOP_AL(operator  operation,__ltl_##opname, float)          \
   BINOP_AL(operator  operation,__ltl_##opname, double)         \
   BINOP_AL(operator  operation,__ltl_##opname, int)            \
   BINOP_AL(operator  operation,__ltl_##opname, long)           \
   BINOP_EL(operator  operation,__ltl_##opname, float)          \
   BINOP_EL(operator  operation,__ltl_##opname, double)         \
   BINOP_EL(operator  operation,__ltl_##opname, int)            \
   BINOP_EL(operator  operation,__ltl_##opname, long)           \
   BINOP_LA(operator  operation,__ltl_##opname, float)          \
   BINOP_LA(operator  operation,__ltl_##opname, double)         \
   BINOP_LA(operator  operation,__ltl_##opname, int)            \
   BINOP_LA(operator  operation,__ltl_##opname, long)           \
   BINOP_LE(operator  operation,__ltl_##opname, float)          \
   BINOP_LE(operator  operation,__ltl_##opname, double)         \
   BINOP_LE(operator  operation,__ltl_##opname, int)            \
   BINOP_LE(operator  operation,__ltl_##opname, long)


// Arguments: binary operator, return type, arbitrary name
// E.g. for bool valued functions, the return type is fixed
//
#define DECLARE_BINOP_RET(operation, ret_type, opname)          \
   BINOP_AA(operator  operation,__ltl_##opname)                 \
   BINOP_AE(operator  operation,__ltl_##opname)                 \
   BINOP_EA(operator  operation,__ltl_##opname)                 \
   BINOP_EE(operator  operation,__ltl_##opname)                 \
   BINOP_AL(operator  operation,__ltl_##opname, float)          \
   BINOP_AL(operator  operation,__ltl_##opname, double)         \
   BINOP_AL(operator  operation,__ltl_##opname, int)            \
   BINOP_AL(operator  operation,__ltl_##opname, long)           \
   BINOP_EL(operator  operation,__ltl_##opname, float)          \
   BINOP_EL(operator  operation,__ltl_##opname, double)         \
   BINOP_EL(operator  operation,__ltl_##opname, int)            \
   BINOP_EL(operator  operation,__ltl_##opname, long)           \
   BINOP_LA(operator  operation,__ltl_##opname, float)          \
   BINOP_LA(operator  operation,__ltl_##opname, double)         \
   BINOP_LA(operator  operation,__ltl_##opname, int)            \
   BINOP_LA(operator  operation,__ltl_##opname, long)           \
   BINOP_LE(operator  operation,__ltl_##opname, float)          \
   BINOP_LE(operator  operation,__ltl_##opname, double)         \
   BINOP_LE(operator  operation,__ltl_##opname, int)            \
   BINOP_LE(operator  operation,__ltl_##opname, long)


// Arguments: unary operator, arbitrary name
//
#define DECLARE_UNOP(operation, opname)         \
   UNOP_A(operator  operation,__ltl_##opname)   \
   UNOP_E(operator  operation,__ltl_##opname)



// Arguments: function, return type
// This is for user defined functions
//
#define DECLARE_BINARY_FUNC_(function, ret_type)        \
  BINOP_AA(function,__ltl_##function)                   \
  BINOP_AE(function,__ltl_##function)                   \
  BINOP_EA(function,__ltl_##function)                   \
  BINOP_EE(function,__ltl_##function)                   \
  BINOP_AL(function,__ltl_##function, float)            \
  BINOP_AL(function,__ltl_##function, double)           \
  BINOP_AL(function,__ltl_##function, int)              \
  BINOP_AL(function,__ltl_##function, long)             \
  BINOP_EL(function,__ltl_##function, float)            \
  BINOP_EL(function,__ltl_##function, double)           \
  BINOP_EL(function,__ltl_##function, int)              \
  BINOP_EL(function,__ltl_##function, long)             \
  BINOP_LA(function,__ltl_##function, float)            \
  BINOP_LA(function,__ltl_##function, double)           \
  BINOP_LA(function,__ltl_##function, int)              \
  BINOP_LA(function,__ltl_##function, long)             \
  BINOP_LE(function,__ltl_##function, float)            \
  BINOP_LE(function,__ltl_##function, double)           \
  BINOP_LE(function,__ltl_##function, int)              \
  BINOP_LE(function,__ltl_##function, long)


// Arguments: function, return type
//
#define DECLARE_UNARY_FUNC_(function, ret_type) \
     UNOP_A(function,__ltl_##function)          \
     UNOP_E(function,__ltl_##function)


// Finally, to make is easier for the user to declare their
// own functions for use in expression templates, provide
// single macros that declare both the applicative templates
// and the parse tree nodes
//
#define DECLARE_BINARY_FUNC(function, ret_type)              \
MAKE_BINAP_FUNC( __ltl_##function, ret_type, function );      \
DECLARE_BINARY_FUNC_(function, ret_type)

#define DECLARE_UNARY_FUNC(function, ret_type)               \
MAKE_UNAP_FUNC( __ltl_##function, ret_type, function );       \
DECLARE_UNARY_FUNC_(function, ret_type)

LTL_END_NAMESPACE

#endif

